﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/user/account_UserRegistry.h>

#include "testAccount_Module.h"

#include "testAccount_Mounter.h"
#include "testAccount_RamFs.h"
#include "testAccount_Util.h"
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultPrivate.h>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/nntest.h>

namespace a = nn::account;
namespace t = nnt::account;

#define NNT_ACCOUNT_ENABLE_USER_REGISTRY_MANY

#if defined(NNT_ACCOUNT_ENABLE_USER_REGISTRY_MANY)

// 複数ユーザー追加テスト用のユーティリティ
namespace
{

// 各種検査込みのユーザー登録機構
class Registrar
{
private:
    typedef std::shared_ptr<a::user::UserRegistry> ManagerPtrType;
    ManagerPtrType m_pReg;

public:
    explicit Registrar(ManagerPtrType pReg) NN_NOEXCEPT
        : m_pReg(pReg)
    {
    }

    nn::Result Register(a::Uid *pOutUid) NN_NOEXCEPT
    {
        size_t count = m_pReg->GetUserCount();
        if (count >= a::UserCountMax)
        {
            NN_LOG("Trying to add user to full registry\n");
            a::Uid tmp;
            NNT_ACCOUNT_RESULT_DO(m_pReg->BeginUserRegistration(&tmp)); // ここで返るはず
            NN_ABORT("Unreachable");
        }

        // 登録開始
        a::Uid pending;
        NNT_ACCOUNT_RESULT_DO(m_pReg->BeginUserRegistration(&pending));
        NN_ABORT_UNLESS(pending);

        // assert
        size_t countAfter = m_pReg->GetUserCount();
        NN_ABORT_UNLESS(countAfter == count);

        // 登録完了
        NNT_ACCOUNT_RESULT_DO(m_pReg->CompleteUserRegistration(pending));

        // assert
        countAfter = m_pReg->GetUserCount();
        NN_ABORT_UNLESS(countAfter == count + 1);

        // assert
        bool existence;
        NNT_ACCOUNT_RESULT_DO(m_pReg->GetUserExistence(&existence, pending));
        NN_ABORT_UNLESS(existence);

        // assert
        // TODO: リストの内容を検査

        *pOutUid = pending;
        NN_RESULT_SUCCESS;
    }

    nn::Result Delete(const a::Uid& uid) NN_NOEXCEPT
    {
        bool existence;
        NNT_ACCOUNT_RESULT_DO(m_pReg->GetUserExistence(&existence, uid));
        if (!existence)
        {
            NN_LOG("Trying to delete unregistered user: %016llx-%016llx\n", uid._data[0], uid._data[1]);
            NN_RESULT_DO(m_pReg->DeleteUser(uid)); // ここで返るはず
            NN_ABORT("Unreachable");
        }

        // assert
        size_t count = m_pReg->GetUserCount();
        NN_ABORT_UNLESS(count > 0);

        // 削除実行
        NN_RESULT_DO(m_pReg->DeleteUser(uid));

        // assert
        size_t countAfter = m_pReg->GetUserCount();
        NN_ABORT_UNLESS(countAfter == count - 1);

        // assert
        NNT_ACCOUNT_RESULT_DO(m_pReg->GetUserExistence(&existence, uid));
        NN_ABORT_UNLESS(!existence);

        // assert
        // TODO: リストの内容を検査

        NN_RESULT_SUCCESS;
    }
};

} // ~namespace <anonymous>

// 複数のユーザーを登録しまくるテスト
TEST(AccountUser, UserRegistry_RegisterMany)
{
    const int TestCount = 1000;

    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    auto pManager = t::CreateRegistryManager();
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(pManager->Initialize(s));

    a::Uid users[a::UserCountMax];
    for (auto& u: users)
    {
        u = a::InvalidUid;
    }

    Registrar reg(pManager);
    for (int t = 0; t < TestCount; ++ t)
    {
        for (int step = 1; step < 8; step++)
        {
            // 登録
            for (int i = 0; i < a::UserCountMax; ++ i)
            {
                NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(reg.Register(&users[i]));
            }
            // 登録順に削除
            for (int i = 0; i < step; ++ i)
            {
                for (int j = i; j < a::UserCountMax; j += step)
                {
                    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(reg.Delete(users[j]));
                    users[j] = a::InvalidUid;
                }
            }

            // 登録
            for (int i = 0; i < a::UserCountMax; ++ i)
            {
                NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(reg.Register(&users[i]));
            }
            // 登録と逆順に削除
            for (int i = 0; i < step; ++ i)
            {
                for (int j = a::UserCountMax - 1 - i; j >= 0 ; j -= step)
                {
                    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(reg.Delete(users[j]));
                    users[j] = a::InvalidUid;
                }
            }

            if (step > 1)
            {
                // 登録
                for (int i = 0; i < a::UserCountMax; ++ i)
                {
                    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(reg.Register(&users[i]));
                }
                // 削除と登録を交互に行う
                for (int i = 0; i < step; ++ i)
                {
                    for (int j = a::UserCountMax - 1 - i; j >= 0 ; j -= step)
                    {
                        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(reg.Delete(users[j]));
                        users[j] = a::InvalidUid;
                    }
                    for (int j = a::UserCountMax - 1 - i; j >= 0 ; j -= step)
                    {
                        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(reg.Register(&users[j]));
                    }
                }
                for (int j = 0; j < a::UserCountMax; ++ j)
                {
                    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(reg.Delete(users[j]));
                    users[j] = a::InvalidUid;
                }
            }
        }
    }
}

#endif // NNT_ACCOUNT_ENABLE_USER_REGISTRY_MANY
