﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "http/account_OAuthUtil.h"
#include "http/account_UrlEncoder.h"

#include <cctype>
#include <nnt/nntest.h>
#include <nn/nn_Log.h>

namespace a = nn::account;

#define NNT_ACCOUNT_ENABLE_ENCODING

#if defined(NNT_ACCOUNT_ENABLE_ENCODING)

namespace encoding
{
template <size_t N>
struct TestCase
{
    char src[N];
    char dst[N * 3];
};

const char Unreserved[] = "0123456789-._~ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
const char Reserved[] = " !\"#$%&'()*+,/:;<=>?@[\\]^`{|}";
NN_STATIC_ASSERT(sizeof(Unreserved) + sizeof(Reserved) - 2 == 0x7E - 0x20 + 1);

TestCase<68> tests68[] = {
    {
        " 0123456789-._~ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz",
        "+0123456789-._~ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz"
    },
};
TestCase<29> tests29[] = {
    {
        "!\"#$%&'()*+,/:;<=>?@[\\]^`{|}",
        "%21%22%23%24%25%26%27%28%29%2A%2B%2C%2F%3A%3B%3C%3D%3E%3F%40%5B%5C%5D%5E%60%7B%7C%7D"
    },
};
} // ~namespace <anonymous>

TEST(AccountUtil, UrlEncoding)
{
    for (auto c : encoding::Unreserved)
    {
        if (c == '\0') break;

        EXPECT_TRUE(a::http::IsUnreservedCharacterForUri(c));
        EXPECT_FALSE(a::http::IsReservedCharacterForUri(c));
        char coded[3];
        auto l = a::http::PutCharacterWithUrlEncoding(coded, sizeof(coded), c);
        EXPECT_EQ(1, l);
        EXPECT_EQ(c, coded[0]);
    }
    for (auto c : encoding::Reserved)
    {
        if (c == '\0') break;

        EXPECT_FALSE(a::http::IsUnreservedCharacterForUri(c));
        EXPECT_TRUE(a::http::IsReservedCharacterForUri(c));
        char coded[3];
        auto l = a::http::PutCharacterWithUrlEncoding(coded, sizeof(coded), c);
        if (c == 0x20)
        {
            EXPECT_EQ(1, l);
            EXPECT_EQ('+', coded[0]);
        }
        else
        {
            EXPECT_EQ(3, l);
            EXPECT_EQ('%', coded[0]);
            EXPECT_TRUE(std::isdigit(coded[1]) != 0);
            EXPECT_EQ(c >> 4, coded[1] - '0');
            EXPECT_TRUE(std::isxdigit(coded[2]) != 0);
            EXPECT_EQ(c & 0xF, std::isalpha(coded[2]) ? std::tolower(coded[2]) - 'a' + 10 : coded[2] - '0');
        }
    }

    for (auto& t : encoding::tests68)
    {
        char dst[256];
        auto l = a::http::UrlEncode(dst, sizeof(dst), t.src, sizeof(t.src));
        EXPECT_EQ(strnlen(t.dst, sizeof(t.dst)), l);
        EXPECT_EQ(l, strnlen(dst, sizeof(dst)));
        for (auto i = 0; i < l; ++ i)
        {
            EXPECT_EQ(t.dst[l], dst[l]);
        }
        EXPECT_TRUE(a::http::CompareEncodedStringPartially(t.src, sizeof(t.src), dst, l));
    }

    for (auto& t : encoding::tests29)
    {
        char dst[256];
        auto l = a::http::UrlEncode(dst, sizeof(dst), t.src, sizeof(t.src));
        EXPECT_EQ(strnlen(t.dst, sizeof(t.dst)), l);
        EXPECT_EQ(l, strnlen(dst, sizeof(dst)));
        for (auto i = 0; i < l; ++ i)
        {
            EXPECT_EQ(t.dst[l], dst[l]);
        }
        EXPECT_TRUE(a::http::CompareEncodedStringPartially(t.src, sizeof(t.src), dst, l));
    }
}

namespace oauth
{
const char Scope[] = "0123456789-._~ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz !#$%&'()*+,/:;<=>?@[]^`{|}";
const char State[] = "0123456789-._~ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz !\"#$%&'()*+,/:;<=>?@[\\]^`{|}";
const char Nonce[] = " !\"#$%&'()*+,/:;<=>?@[\\]^`{|}0123456789-._~ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
const char BadScope[][2] = {
    "",
    {0x01, 0x00},
    {0x02, 0x00},
    {0x03, 0x00},
    {0x04, 0x00},
    {0x05, 0x00},
    {0x06, 0x00},
    {0x07, 0x00},
    {0x08, 0x00},
    {0x09, 0x00},
    {0x0A, 0x00},
    {0x0B, 0x00},
    {0x0C, 0x00},
    {0x0D, 0x00},
    {0x0E, 0x00},
    {0x0F, 0x00},
    {0x10, 0x00},
    {0x12, 0x00},
    {0x12, 0x00},
    {0x13, 0x00},
    {0x14, 0x00},
    {0x15, 0x00},
    {0x16, 0x00},
    {0x17, 0x00},
    {0x18, 0x00},
    {0x19, 0x00},
    {0x1A, 0x00},
    {0x1B, 0x00},
    {0x1C, 0x00},
    {0x1D, 0x00},
    {0x1E, 0x00},
    {0x1F, 0x00},
    "\\",
    "\"",
    {0x7F, 0x00},
};
const char BadState[][2] = {
    "",
    {0x01, 0x00},
    {0x02, 0x00},
    {0x03, 0x00},
    {0x04, 0x00},
    {0x05, 0x00},
    {0x06, 0x00},
    {0x07, 0x00},
    {0x08, 0x00},
    {0x09, 0x00},
    {0x0A, 0x00},
    {0x0B, 0x00},
    {0x0C, 0x00},
    {0x0D, 0x00},
    {0x0E, 0x00},
    {0x0F, 0x00},
    {0x10, 0x00},
    {0x12, 0x00},
    {0x12, 0x00},
    {0x13, 0x00},
    {0x14, 0x00},
    {0x15, 0x00},
    {0x16, 0x00},
    {0x17, 0x00},
    {0x18, 0x00},
    {0x19, 0x00},
    {0x1A, 0x00},
    {0x1B, 0x00},
    {0x1C, 0x00},
    {0x1D, 0x00},
    {0x1E, 0x00},
    {0x1F, 0x00},
    {0x7F, 0x00},
};
const char BadNonce[][2] = {
    "",
    {0x01, 0x00},
    {0x02, 0x00},
    {0x03, 0x00},
    {0x04, 0x00},
    {0x05, 0x00},
    {0x06, 0x00},
    {0x07, 0x00},
    {0x08, 0x00},
    {0x09, 0x00},
    {0x0A, 0x00},
    {0x0B, 0x00},
    {0x0C, 0x00},
    {0x0D, 0x00},
    {0x0E, 0x00},
    {0x0F, 0x00},
    {0x10, 0x00},
    {0x12, 0x00},
    {0x12, 0x00},
    {0x13, 0x00},
    {0x14, 0x00},
    {0x15, 0x00},
    {0x16, 0x00},
    {0x17, 0x00},
    {0x18, 0x00},
    {0x19, 0x00},
    {0x1A, 0x00},
    {0x1B, 0x00},
    {0x1C, 0x00},
    {0x1D, 0x00},
    {0x1E, 0x00},
    {0x1F, 0x00},
    {0x7F, 0x00},
};

} // ~namespace oauth

TEST(AccountUtil, OAuthParamValidation)
{
    EXPECT_TRUE(a::http::IsValidScope(oauth::Scope, sizeof(oauth::Scope)));
    for (auto& t : oauth::BadScope)
    {
        EXPECT_FALSE(a::http::IsValidScope(t, sizeof(t)));
    }

    EXPECT_TRUE(a::http::IsValidState(oauth::State, sizeof(oauth::State)));
    for (auto& t : oauth::BadState)
    {
        EXPECT_FALSE(a::http::IsValidState(t, sizeof(t)));
    }

    EXPECT_TRUE(a::http::IsValidNonce(oauth::Nonce, sizeof(oauth::Nonce)));
    for (auto& t : oauth::BadNonce)
    {
        EXPECT_FALSE(a::http::IsValidNonce(t, sizeof(t)));
    }
}

#endif // NNT_ACCOUNT_ENABLE_ENCODING
