﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/detail/account_LocalStorage.h>

#include "detail/account_PathUtil.h"
#include <nn/account/account_Types.h>

#include "testAccount_Mounter.h"
#include "testAccount_RamFs.h"
#include "testAccount_Util.h"

#include <vector>
#include <functional>

#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/os/os_Tick.h>
#include <nn/util/util_ScopeExit.h>

namespace a = nn::account;
namespace t = nnt::account;

#define NNT_ACCOUNT_ENABLE_LOCAL_STORAGE_RAM
#define NNT_ACCOUNT_ENABLE_LOCAL_STORAGE_HOST

namespace
{

bool TestNoEntries(
    const a::detail::AbstractLocalStorage& ls) NN_NOEXCEPT
{
    const auto& fs = ls.GetFileSystem();
    const auto& rootPath = ls.GetRootPath();

    auto uid = a::detail::ConvertToUid(ls.GenerateUuidWithContext());
    auto cacheId = ls.GenerateUuidWithContext();

    auto lock = ls.AcquireWriterLock();

    char path[128] = {0};
    size_t s;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::fs::ResultPathNotFound, fs.GetSize(&s, a::detail::PathUtil::GetUserListPath(path, sizeof(path), rootPath)));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::fs::ResultPathNotFound, fs.Create(a::detail::PathUtil::GetProfileListPath(path, sizeof(path), rootPath), 128));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::fs::ResultPathNotFound, fs.Create(a::detail::PathUtil::GetProfileImagePath(path, sizeof(path), uid, rootPath), 128));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::fs::ResultPathNotFound, fs.Create(a::detail::PathUtil::GetCachePath(path, sizeof(path), cacheId, rootPath), 128));
    return true;
}

bool TestCreateDeleteNormalFiles(
    const a::detail::AbstractLocalStorage& ls,
    const size_t SizeToReserve) NN_NOEXCEPT
{
    const auto& fs = ls.GetFileSystem();
    const auto& rootPath = ls.GetRootPath();

    auto l = ls.AcquireWriterLock();

    char path[128];
    size_t size;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Create(a::detail::PathUtil::GetUserListPath(path, sizeof(path), rootPath), SizeToReserve));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.GetSize(&size, path));
    EXPECT_EQ(SizeToReserve, size);

    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Delete(a::detail::PathUtil::GetUserListPath(path, sizeof(path), rootPath)));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::fs::ResultPathNotFound, fs.GetSize(&size, path));
    return true;
}

bool TestCreateDeleteProfiles(
    const a::detail::AbstractLocalStorage& ls,
    const size_t SizeToReserve,
    const int TestUserCount) NN_NOEXCEPT
{
    const auto& fs = ls.GetFileSystem();
    const auto& rootPath = ls.GetRootPath();

    std::vector<a::Uid> users;

    char path[128];
    size_t size;

    {
        auto l = ls.AcquireWriterLock();
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Create(a::detail::PathUtil::GetProfileListPath(path, sizeof(path), rootPath), SizeToReserve));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.GetSize(&size, path));
    }

    for (auto i = 0; i < TestUserCount; ++ i)
    {
        auto& uid = a::detail::ConvertToUid(ls.GenerateUuidWithContext());
        users.push_back(uid);

        auto l = ls.AcquireWriterLock();
        EXPECT_EQ(SizeToReserve, size);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Create(a::detail::PathUtil::GetProfileImagePath(path, sizeof(path), uid, rootPath), SizeToReserve));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.GetSize(&size, path));
        EXPECT_EQ(SizeToReserve, size);
    }

    for (auto& uid: users)
    {
        auto l = ls.AcquireWriterLock();
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Delete(a::detail::PathUtil::GetProfileImagePath(path, sizeof(path), uid, rootPath)));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::fs::ResultPathNotFound, fs.GetSize(&size, path));
    }

    auto l = ls.AcquireWriterLock();
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Delete(a::detail::PathUtil::GetProfileListPath(path, sizeof(path), rootPath)));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::fs::ResultPathNotFound, fs.GetSize(&size, path));
    return true;
}

bool TestCreateDeleteCache(
    const a::detail::AbstractLocalStorage& ls,
    const size_t SizeToReserve,
    const int TestCacheCount) NN_NOEXCEPT
{
    const auto& fs = ls.GetFileSystem();
    const auto& rootPath = ls.GetRootPath();

    char path[128];
    size_t size;

    std::vector<a::detail::Uuid> cache;
    for (auto i = 0; i < TestCacheCount; ++ i)
    {
        auto cacheId = ls.GenerateUuidWithContext();
        cache.push_back(cacheId);

        auto l = ls.AcquireWriterLock();
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Create(a::detail::PathUtil::GetCachePath(path, sizeof(path), cacheId, rootPath), SizeToReserve));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.GetSize(&size, path));
        EXPECT_EQ(SizeToReserve, size);
    }
    for (auto& cacheId: cache)
    {
        auto l = ls.AcquireWriterLock();
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Delete(a::detail::PathUtil::GetCachePath(path, sizeof(path), cacheId, rootPath)));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::fs::ResultPathNotFound, fs.GetSize(&size, path));
    }

    return true;
}

bool PrepareClearCache(
    const a::detail::AbstractLocalStorage& ls,
    const size_t SizeToReserve,
    const int TestCacheCount) NN_NOEXCEPT
{
    const auto& fs = ls.GetFileSystem();
    const auto& rootPath = ls.GetRootPath();

    char path[128];
    size_t size;

    std::vector<a::detail::Uuid> cache;
    for (auto i = 0; i < TestCacheCount; ++ i)
    {
        auto cacheId = ls.GenerateUuidWithContext();
        cache.push_back(cacheId);

        auto l = ls.AcquireWriterLock();
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.Create(a::detail::PathUtil::GetCachePath(path, sizeof(path), cacheId, rootPath), SizeToReserve));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(fs.GetSize(&size, path));
        EXPECT_EQ(SizeToReserve, size);
    }
    return true;
}

bool RunTest(
    const a::detail::AbstractLocalStorage& ls,
    size_t sizeUnit) NN_NOEXCEPT
{
    // 通常ファイル
    EXPECT_TRUE(TestCreateDeleteNormalFiles(ls, sizeUnit));

    // プロフィール関連ファイル
    EXPECT_TRUE(TestCreateDeleteProfiles(ls, sizeUnit, 128));

    // キャッシュ
    EXPECT_TRUE(TestCreateDeleteCache(ls, 4 * sizeUnit, 128));

    EXPECT_TRUE(PrepareClearCache(ls, 4 * sizeUnit, 128));

    return true;
}

}

#if defined(NNT_ACCOUNT_ENABLE_LOCAL_STORAGE_RAM)

TEST(AccountUtil, LocalStorage_RamFs)
{
    const int TestCount = 100;

    struct LocalVolumeNameGetter
    {
        static const char* Get() NN_NOEXCEPT
        {
            return "local";
        }
    };
    a::detail::LocalStorage<t::RamFs, t::HostSaveData<LocalVolumeNameGetter>::Policy> hs;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(hs.Mount());
    EXPECT_EQ(0,std::memcmp("local:/su", hs.GetRootPath(), sizeof("local:/su")));

    char path[128];
    a::detail::PathUtil::GetCacheDirectoryPath(path, sizeof(path), hs.GetRootPath());

    hs.Clear();

    const size_t SizeToReserve = 1024;
    for (auto i = 0; i < TestCount; ++ i)
    {
        hs.Setup();

        ASSERT_TRUE(RunTest(hs, SizeToReserve));

        hs.ClearCache();

        // キャッシュクリアを確認
        EXPECT_EQ(0, t::RamFs::GetEntryCountOn(path));
    }

    hs.Clear();
}

#endif

#if defined(NNT_ACCOUNT_ENABLE_LOCAL_STORAGE_HOST)

TEST(AccountUtil, LocalStorage_HostFs)
{
    const int TestCount = 2;

    struct LocalVolumeNameGetter
    {
        static const char* Get() NN_NOEXCEPT
        {
            return "local";
        }
    };
    a::detail::LocalStorage<a::detail::DefaultFileSystem, t::HostSaveData<LocalVolumeNameGetter>::Policy> hs;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(hs.Mount());
    EXPECT_EQ(0,std::memcmp("local:/su", hs.GetRootPath(), sizeof("local:/su")));

    char path[128];
    a::detail::PathUtil::GetCacheDirectoryPath(path, sizeof(path), hs.GetRootPath());

    // 初期状態: ディレクトリと通常ファイルがないことの確認
    hs.Clear();
    {
        EXPECT_TRUE(TestNoEntries(hs));
    }

    const size_t SizeToReserve = 1024;
    for (auto i = 0; i < TestCount; ++ i)
    {
        hs.Setup();

        ASSERT_TRUE(RunTest(hs, SizeToReserve));

        hs.ClearCache();

        // キャッシュクリアを確認
        nn::fs::DirectoryHandle dir;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenDirectory(&dir, path, nn::fs::OpenDirectoryMode_All));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseDirectory(dir);
        };
        int64_t count;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetDirectoryEntryCount(&count, dir));
        EXPECT_EQ(0, count);
    }

    // 初期状態の再確認 (*)
    hs.Clear();
    {
        EXPECT_TRUE(TestNoEntries(hs));
    }
}

#endif
