﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/detail/account_CacheBase.h>
#include <nn/account/account_Result.h>
#include "detail/account_CacheUtil.h"
#include "detail/account_PathUtil.h"
#include "detail/account_TimeUtil.h"

#include "testAccount_Util.h"
#include "testAccount_Mounter.h"

#include <nnt/nntest.h>

#include <nn/nn_Log.h>
#include <nn/os/os_Thread.h>

#define NNT_ACCOUNT_ENABLE_CACHE_UTIL
#define NNT_ACCOUNT_ENABLE_CACHE_1
#define NNT_ACCOUNT_ENABLE_CACHE_N
#define NNT_ACCOUNT_ENABLE_CACHE_N_LIST
#define NNT_ACCOUNT_ENABLE_CACHE_N_DATA

namespace a = nn::account;
namespace t = nnt::account;

namespace
{
struct Tag
{
    uint32_t value;
    bool local;

    bool NN_EXPLICIT_OPERATOR ==(const Tag& rhs) const NN_NOEXCEPT
    {
        return value == rhs.value && local == rhs.local;
    }
    bool NN_EXPLICIT_OPERATOR !=(const Tag& rhs) const NN_NOEXCEPT
    {
        return !(*this == rhs);
    }
};

struct Data
{
    a::detail::Uuid uuid;

    Data() NN_NOEXCEPT
        : uuid(a::detail::InvalidUuid)
    {
    }
    explicit Data(const a::detail::Uuid& v) NN_NOEXCEPT
        : uuid(v)
    {
    }
    Data(const Data& d) NN_NOEXCEPT
        : uuid(d.uuid)
    {
    }
    bool NN_EXPLICIT_OPERATOR ==(const Data& rhs) const NN_NOEXCEPT
    {
        return uuid == rhs.uuid;
    }
    bool NN_EXPLICIT_OPERATOR !=(const Data& rhs) const NN_NOEXCEPT
    {
        return !(*this == rhs);
    }
    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return uuid;
    }
};

struct Operator
{
    static bool IsExpired(int64_t e) NN_NOEXCEPT
    {
        return a::detail::CacheUtil::IsExpiredInUptime<0>(e);
    }
    static void DeleteCacheData(const Data& data, const a::detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    {
        a::detail::CacheUtil::DeleteCacheFile(data.uuid, storage);
    }
};

const int CacheCountMax = a::detail::NdasAppAuthCacheCountMax;
typedef a::ResultBaasTokenCacheNotExist ResultNotExist;
typedef a::ResultNdasTokenCacheExpired ResultExpired;
typedef a::ResultNdasTokenLengthUnacceptable ResultSizeUnacceptable;
struct ResultConfig : public a::detail::CacheSearchResultConfig<ResultNotExist, ResultExpired>
{
};

class Cache1 : public a::detail::CacheBase1<Data, Operator, ResultConfig>
{
private:
    typedef a::detail::CacheBase1<Data, Operator, ResultConfig> Base;
public:
    void Store(int64_t expiration, const Data& data) NN_NOEXCEPT
    {
        Base::Store(expiration, data);
    }
    nn::Result Get(Data* pOutData) const NN_NOEXCEPT
    {
        std::lock_guard<decltype(*this)> lock(*this);
        return Base::GetUnsafe(pOutData);
    }
};

class CacheN : public a::detail::CacheBaseN<CacheCountMax, Tag, Data, Operator, ResultConfig>
{
private:
    typedef a::detail::CacheBaseN<CacheCountMax, Tag, Data, Operator, ResultConfig> Base;
public:
    void Store(const Tag& tag, int64_t expiration, const Data& data) NN_NOEXCEPT
    {
        Base::Store(tag, expiration, data);
    }
    nn::Result Find(Data* pOutData, const Tag& tag) const NN_NOEXCEPT
    {
        std::lock_guard<decltype(*this)> lock(*this);
        return Base::FindUnsafe(pOutData, tag);
    }
};

}

#if defined(NNT_ACCOUNT_ENABLE_CACHE_UTIL)

TEST(AccountCacheStorage, CacheUtil)
{
    auto e = a::detail::GetUptimeInSeconds();
    EXPECT_FALSE(a::detail::CacheUtil::IsExpiredInUptime<0>(e + 1));
    EXPECT_TRUE(a::detail::CacheUtil::IsExpiredInUptime<0>(e - 1));
    EXPECT_FALSE(a::detail::CacheUtil::IsExpiredInUptime<10>(e + 10 + 1));
    EXPECT_TRUE(a::detail::CacheUtil::IsExpiredInUptime<10>(e + 10 - 1));

    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    static char data[128] = {};
    std::memset(data, '0', sizeof(data));
    static char dataOut[sizeof(data) * 2] = {};

    Data d;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(a::detail::CacheUtil::StoreCacheFile(&d.uuid, data, sizeof(data), s));
    size_t sizeActual;
    auto r = a::detail::CacheUtil::LoadCacheFile<sizeof(data), ResultSizeUnacceptable>(&sizeActual, dataOut, sizeof(dataOut), d.uuid, s);
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(r);
    EXPECT_EQ(sizeof(data), sizeActual);
    EXPECT_EQ(0, std::memcmp(data, dataOut, sizeof(data)));

    auto r2 = a::detail::CacheUtil::LoadCacheFile<sizeof(data) - 1, ResultSizeUnacceptable>(&sizeActual, dataOut, sizeof(dataOut), d.uuid, s);
    NNT_ACCOUNT_EXPECT_RESULT_INCLUDED(ResultSizeUnacceptable, r2);

    auto r3 = a::detail::CacheUtil::LoadCacheFile<sizeof(data), ResultSizeUnacceptable>(&sizeActual, dataOut, sizeof(data) - 1, d.uuid, s);
    NNT_ACCOUNT_EXPECT_RESULT_INCLUDED(a::ResultInsufficientBuffer, r3);

    a::detail::CacheUtil::DeleteCacheFile(d.uuid, s);
    auto r4 = a::detail::CacheUtil::LoadCacheFile<sizeof(data), ResultSizeUnacceptable>(&sizeActual, dataOut, sizeof(dataOut), d.uuid, s);
    NNT_ACCOUNT_EXPECT_RESULT_INCLUDED(nn::fs::ResultPathNotFound, r4);
}

#endif

#if defined(NNT_ACCOUNT_ENABLE_CACHE_1)

TEST(AccountCacheStorage, Cache1)
{
    static char data1[512] = {};
    std::memset(data1, '1', sizeof(data1));
    static char data2[1024] = {};
    std::memset(data2, '2', sizeof(data2));

    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    Cache1 cache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(cache.Initialize(s));

    // 初期状態: 失効
    ASSERT_FALSE(cache.IsAvailable());

    // 失効→有効
    {
        a::detail::Uuid cacheId = a::detail::InvalidUuid;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(a::detail::CacheUtil::StoreCacheFile(&cacheId, data1, sizeof(data1), s));

        auto current = a::detail::GetUptimeInSeconds();
        cache.Store(current + 10, Data(cacheId));
        EXPECT_TRUE(cache.IsAvailable());

        char path[128];
        a::detail::PathUtil::GetCachePath(path, sizeof(path), cacheId, s.GetRootPath());
        size_t size;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(s.GetFileSystem().GetSize(&size, path));
        ASSERT_EQ(sizeof(data1), size);

        // キャッシュ読み出し
        Data d;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(cache.Get(&d));
        size_t sizeActual;
        t::Buffer buffer(size);
        auto r = a::detail::CacheUtil::LoadCacheFile<sizeof(data1), ResultSizeUnacceptable>(&sizeActual, buffer.Get<char>(), buffer.GetSize(), d.uuid, cache.GetStorageRef());
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(r);
        ASSERT_EQ(size, sizeActual);
        ASSERT_EQ(0, std::memcmp(data1, buffer.GetAddress(), size));

        // 有効なキャッシュを失効させると消える
        cache.Invalidate();
        EXPECT_FALSE(cache.IsAvailable());
        NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(nn::fs::ResultPathNotFound, s.GetFileSystem().GetSize(&size, path));
    }

    // 既に失効しているものをキャッシュすると即座に消える
    {
        a::detail::Uuid cacheId = a::detail::InvalidUuid;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(a::detail::CacheUtil::StoreCacheFile(&cacheId, data1, sizeof(data1), s));

        auto current = a::detail::GetUptimeInSeconds();
        cache.Store(current - 1, Data(cacheId));
        EXPECT_FALSE(cache.IsAvailable());

        char path[128];
        a::detail::PathUtil::GetCachePath(path, sizeof(path), cacheId, s.GetRootPath());
        size_t size;
        NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(nn::fs::ResultPathNotFound, s.GetFileSystem().GetSize(&size, path));
    }

    // 有効でも次をキャッシュすると消える
    {
        a::detail::Uuid cacheId = a::detail::InvalidUuid;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(a::detail::CacheUtil::StoreCacheFile(&cacheId, data1, sizeof(data1), s));

        auto current = a::detail::GetUptimeInSeconds();
        cache.Store(current + 0xFFFF, Data(cacheId));
        EXPECT_TRUE(cache.IsAvailable());

        char path[128];
        a::detail::PathUtil::GetCachePath(path, sizeof(path), cacheId, s.GetRootPath());
        size_t size;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(s.GetFileSystem().GetSize(&size, path));
        EXPECT_EQ(sizeof(data1), size);

        // 新しいキャッシュの作成
        a::detail::Uuid cacheId2 = a::detail::InvalidUuid;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(a::detail::CacheUtil::StoreCacheFile(&cacheId2, data2, sizeof(data2), s));

        // 新しいキャッシュで上書き
        current = a::detail::GetUptimeInSeconds();
        cache.Store(current + 10, Data(cacheId2));
        EXPECT_TRUE(cache.IsAvailable());
        NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(nn::fs::ResultPathNotFound, s.GetFileSystem().GetSize(&size, path));

        char path2[128];
        a::detail::PathUtil::GetCachePath(path2, sizeof(path2), cacheId2, s.GetRootPath());
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(s.GetFileSystem().GetSize(&size, path2));
        EXPECT_EQ(sizeof(data2), size);

        // キャッシュ読み出し
        Data d;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(cache.Get(&d));
        size_t sizeActual;
        t::Buffer buffer(size);
        auto r = a::detail::CacheUtil::LoadCacheFile<sizeof(data2), ResultSizeUnacceptable>(&sizeActual, buffer.Get<char>(), buffer.GetSize(), d.uuid, cache.GetStorageRef());
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(r);
        ASSERT_EQ(size, sizeActual);
        ASSERT_EQ(0, std::memcmp(data2, buffer.GetAddress(), size));
    }
} // NOLINT(readability/fn_size)
#endif //  NNT_ACCOUNT_ENABLE_CACHE_1

#if defined(NNT_ACCOUNT_ENABLE_CACHE_N)
TEST(AccountCacheStorage, CacheN)
{
    static char data1[512] = {};
    std::memset(data1, '1', sizeof(data1));
    static char data2[1024] = {};
    std::memset(data2, '2', sizeof(data2));
    const Tag tag1 = { 0x11110000ul };

    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    CacheN cache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(cache.Initialize(s));

    // 初期状態: 失効
    const Tag invalid = { 0x00ull, false };
    ASSERT_FALSE(cache.IsAvailable(invalid));

    {
        // 失効→有効
        {
            a::detail::Uuid cacheId;
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(a::detail::CacheUtil::StoreCacheFile(&cacheId, data1, sizeof(data1), s));
            auto current = a::detail::GetUptimeInSeconds();
            cache.Store(tag1, current + 10, Data(cacheId));
            EXPECT_TRUE(cache.IsAvailable(tag1));

            char path[128];
            a::detail::PathUtil::GetCachePath(path, sizeof(path), cacheId, s.GetRootPath());
            size_t size;
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(s.GetFileSystem().GetSize(&size, path));
            ASSERT_EQ(sizeof(data1), size);

            // キャッシュ読み出し
            Data d;
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(cache.Find(&d, tag1));
            size_t sizeActual;
            t::Buffer buffer(size);
            auto r = a::detail::CacheUtil::LoadCacheFile<sizeof(data1), ResultSizeUnacceptable>(&sizeActual, buffer.Get<char>(), buffer.GetSize(), d.uuid, cache.GetStorageRef());
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(r);
            ASSERT_EQ(size, sizeActual);
            ASSERT_EQ(0, std::memcmp(data1, buffer.GetAddress(), size));

            // 有効なキャッシュを失効させると消える
            cache.Invalidate(tag1);
            EXPECT_FALSE(cache.IsAvailable(tag1));
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(nn::fs::ResultPathNotFound, s.GetFileSystem().GetSize(&size, path));
        }

        // 既に失効しているものをキャッシュすると即座に消える
        {
            a::detail::Uuid cacheId;
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(a::detail::CacheUtil::StoreCacheFile(&cacheId, data1, sizeof(data1), s));
            auto current = a::detail::GetUptimeInSeconds();
            cache.Store(tag1, current - 1, Data(cacheId));
            EXPECT_FALSE(cache.IsAvailable(tag1));

            char path[128];
            a::detail::PathUtil::GetCachePath(path, sizeof(path), cacheId, s.GetRootPath());
            size_t size;
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(nn::fs::ResultPathNotFound, s.GetFileSystem().GetSize(&size, path));
        }

        // 有効でも次をキャッシュすると消える
        {
            a::detail::Uuid cacheId;
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(a::detail::CacheUtil::StoreCacheFile(&cacheId, data1, sizeof(data1), s));
            auto current = a::detail::GetUptimeInSeconds();
            cache.Store(tag1, current + 0xFFFF, Data(cacheId));
            EXPECT_TRUE(cache.IsAvailable(tag1));

            char path[128];
            a::detail::PathUtil::GetCachePath(path, sizeof(path), cacheId, s.GetRootPath());
            size_t size;
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(s.GetFileSystem().GetSize(&size, path));
            EXPECT_EQ(sizeof(data1), size);

            // 新しいキャッシュの作成
            a::detail::Uuid cacheId2;
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(a::detail::CacheUtil::StoreCacheFile(&cacheId2, data2, sizeof(data2), s));

            // 新しいキャッシュで上書き
            current = a::detail::GetUptimeInSeconds();
            cache.Store(tag1, current + 10, Data(cacheId2));
            EXPECT_TRUE(cache.IsAvailable(tag1));
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(nn::fs::ResultPathNotFound, s.GetFileSystem().GetSize(&size, path));

            char path2[128];
            a::detail::PathUtil::GetCachePath(path2, sizeof(path2), cacheId2, s.GetRootPath());
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(s.GetFileSystem().GetSize(&size, path2));
            EXPECT_EQ(sizeof(data2), size);

            // キャッシュ読み出し
            Data d;
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(cache.Find(&d, tag1));
            size_t sizeActual;
            t::Buffer buffer(size);
            auto r = a::detail::CacheUtil::LoadCacheFile<sizeof(data2), ResultSizeUnacceptable>(&sizeActual, buffer.Get<char>(), buffer.GetSize(), d.uuid, cache.GetStorageRef());
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(r);
            ASSERT_EQ(size, sizeActual);
            ASSERT_EQ(0, std::memcmp(data2, buffer.GetAddress(), size));
        }
    }
} // NOLINT(readability/fn_size)
#endif // NNT_ACCOUNT_ENABLE_CACHE_N

#if defined(NNT_ACCOUNT_ENABLE_CACHE_N_LIST)
namespace
{
template <typename StorageType>
void TestAppAuthCacheListModification(
    const Tag (&tags)[CacheCountMax],
    const int ExpirationActual,
    const StorageType& s) NN_NOEXCEPT
{
    CacheN cache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(cache.Initialize(s));

    auto expiration = a::detail::GetUptimeInSeconds() + ExpirationActual;
    for (const auto& t: tags)
    {
        cache.Store(t, expiration, Data());
        ASSERT_TRUE(cache.IsAvailable(t));
    }

    // 更新
    auto expiration2 = a::detail::GetUptimeInSeconds() + ExpirationActual * 2;
    cache.Store(tags[0], expiration2, Data());
    for (const auto& t: tags)
    {
        ASSERT_TRUE(cache.IsAvailable(t));
    }

    // 無効化
    cache.Invalidate(tags[0]);
    ASSERT_FALSE(cache.IsAvailable(tags[0]));
    for (auto i = 1; i < sizeof(tags) / sizeof(tags[0]); ++ i)
    {
        ASSERT_TRUE(cache.IsAvailable(tags[i]));
    }
}
template <typename StorageType>
void TestAppAuthCacheListClearInvalid(
    const Tag (&tags)[CacheCountMax],
    const int ExpirationActual,
    const StorageType& s) NN_NOEXCEPT
{
    CacheN cache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(cache.Initialize(s));
    Tag tag2 = { 0x00ul, true };

    auto expiration = a::detail::GetUptimeInSeconds() + ExpirationActual;
    for (const auto& t: tags)
    {
        cache.Store(t, expiration, Data());
        ASSERT_TRUE(cache.IsAvailable(t));
    }

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(ExpirationActual + 1));
    for (const auto& t: tags)
    {
        ASSERT_FALSE(cache.IsAvailable(t));
    }

    auto expiration2 = a::detail::GetUptimeInSeconds() + ExpirationActual;
    cache.Store(tag2, expiration2, Data());
    ASSERT_TRUE(cache.IsAvailable(tag2));
}
template <typename StorageType>
void TestAppAuthCacheListPushNew(
    const Tag (&tags)[CacheCountMax],
    const int ExpirationActual,
    const StorageType& s) NN_NOEXCEPT
{
    CacheN cache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(cache.Initialize(s));
    Tag tag2 = { 0x00ul, true };

    auto expiration = a::detail::GetUptimeInSeconds() + ExpirationActual;
    for (const auto& t: tags)
    {
        cache.Store(t, expiration, Data());
        ASSERT_TRUE(cache.IsAvailable(t));
    }

    auto expiration2 = a::detail::GetUptimeInSeconds() + ExpirationActual;
    cache.Store(tag2, expiration2, Data());
    ASSERT_TRUE(cache.IsAvailable(tag2));
    ASSERT_FALSE(cache.IsAvailable(tags[0]));
    for (auto i = 1; i < sizeof(tags) / sizeof(tags[0]); ++ i)
    {
        ASSERT_TRUE(cache.IsAvailable(tags[i]));
    }
}
template <typename StorageType>
void TestAppAuthCacheListStoreInvalid(
    const Tag (&tags)[CacheCountMax],
    const int ExpirationActual,
    const StorageType& s) NN_NOEXCEPT
{
    CacheN cache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(cache.Initialize(s));

    auto expiration = a::detail::GetUptimeInSeconds() + ExpirationActual;
    for (const auto& t: tags)
    {
        cache.Store(t, expiration, Data());
        ASSERT_TRUE(cache.IsAvailable(t));
    }

    auto expiration2 = a::detail::GetUptimeInSeconds() - 1; // Store されないキャッシュ
    cache.Store(tags[0], expiration2, Data());
    for (const auto& t: tags)
    {
        ASSERT_TRUE(cache.IsAvailable(t));
    }
}
} // ~namespace <anonymous>

TEST(AccountCacheStorage, CacheNList)
{
    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    /* キャッシュリストのテスト
     */
    const int ExpirationActual = 1; // このテストで作成する正常なキャッシュは ExpirationActual 秒後に失効する

    // テスト用 AppInfo 生成
    Tag tags[CacheCountMax];
    for (auto i = 0; i < sizeof(tags) / sizeof(tags[0]); ++ i)
    {
        tags[i].value = static_cast<uint32_t>(i);
        tags[i].local = false;
    }

    // ある AppInfo のキャッシュを操作しても、別の有効なキャッシュに影響はない
    TestAppAuthCacheListModification(tags, ExpirationActual, s);

    // 新しくキャッシュを作ると、失効しているものはすべて消える
    TestAppAuthCacheListClearInvalid(tags, ExpirationActual, s);

    // すべてが有効なとき、新しくキャッシュを作ると、一番はじめに追加したキャッシュが削除される
    TestAppAuthCacheListPushNew(tags, ExpirationActual, s);

    // 無効なキャッシュを作っても、自身の AppInfo のキャッシュを含め既に存在するキャッシュに影響はない
    TestAppAuthCacheListStoreInvalid(tags, ExpirationActual, s);
}
#endif // NNT_ACCOUNT_ENABLE_CACHE_N_LIST

#if defined(NNT_ACCOUNT_ENABLE_CACHE_N_DATA)
TEST(AccountCacheStorage, CacheNData)
{
    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    CacheN cache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(cache.Initialize(s));

    // 最初にキャッシュしたデータがいろいろ経た後にも残っているか

    // データ類
    char data[512];
    std::memset(data, 0xFE, sizeof(data));
    a::detail::Uuid cacheId = a::detail::InvalidUuid;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(a::detail::CacheUtil::StoreCacheFile(&cacheId, data, sizeof(data), s));

    char data2[1024];
    std::memset(data2, 0x3C, sizeof(data2));
    a::detail::Uuid cacheId2 = a::detail::InvalidUuid;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(a::detail::CacheUtil::StoreCacheFile(&cacheId2, data2, sizeof(data2), s));

    Tag tags[5];
    for (auto i = 0; i < sizeof(tags) / sizeof(tags[0]); ++ i)
    {
        tags[i].value = static_cast<uint32_t>(i);
        tags[i].local = false;
    }
    auto exprBase = a::detail::GetUptimeInSeconds();

    // 1. #1 追加 (t = 4)
    cache.Store(tags[0], exprBase + 4, Data());

    // 2. #2 追加 (t = 6)
    cache.Store(tags[1], exprBase + 6, Data(cacheId));

    // 3. #3 追加 (t = 5)
    cache.Store(tags[2], exprBase + 5, Data());

    // 4. #4 追加 (t = 4)
    cache.Store(tags[3], exprBase + 4, Data());

    // 5. #3 更新 (t = 5 -> 2)
    cache.Store(tags[2], exprBase + 2, Data());
    // 前方から #3, 4, 2, 1 の順で並ぶ

    // 6. Sleep(3)
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    // #3 が失効する (リスト上にはある)

    // 7. #5 追加 (t = 10)
    cache.Store(tags[4], exprBase + 10, Data(cacheId2));
    // #3 が取り除かれる。
    // 前方から #5, 4, 2, 1 の順で並ぶ

    // 8. Sleep(2)
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(2));
    // #1, 4 が失効する (リスト上にはある)

    // 9. #2, 5 のデータを確認
    Data d2, d5;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(cache.Find(&d2, tags[1]));
    size_t sizeActual;
    t::Buffer buffer(4096);
    auto r2 = a::detail::CacheUtil::LoadCacheFile<sizeof(data), ResultSizeUnacceptable>(&sizeActual, buffer.Get<char>(), buffer.GetSize(), d2.uuid, cache.GetStorageRef());
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(r2);
    ASSERT_EQ(sizeof(data), sizeActual);
    ASSERT_EQ(0, std::memcmp(data, buffer.GetAddress(), sizeActual));

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(cache.Find(&d5, tags[4]));
    auto r5 = a::detail::CacheUtil::LoadCacheFile<sizeof(data2), ResultSizeUnacceptable>(&sizeActual, buffer.Get<char>(), buffer.GetSize(), d5.uuid, cache.GetStorageRef());
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(r5);
    ASSERT_EQ(sizeof(data2), sizeActual);
    ASSERT_EQ(0, std::memcmp(data2, buffer.GetAddress(), sizeActual));

    // 10. #1, 4 が Expired, #3 が Uncached
    ASSERT_FALSE(cache.IsAvailable(tags[0]));
    Data d;
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(ResultExpired, cache.Find(&d, tags[0]));
    ASSERT_FALSE(cache.IsAvailable(tags[3]));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(ResultExpired, cache.Find(&d, tags[3]));

    ASSERT_FALSE(cache.IsAvailable(tags[2]));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(ResultNotExist, cache.Find(&d, tags[2]));
}
#endif // NNT_ACCOUNT_ENABLE_CACHE_N_DATA
