﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "baas/account_BaasUserAdaptor.h"

#include <nn/account/json/account_RapidJsonApi.h>

#include "testAccount_RapidJsonInputStream.h"
#include "testAccount_Mounter.h"
#include "testAccount_Util.h"

namespace a = nn::account;
namespace t = nnt::account;

namespace imageUpload
{
const char* Inputs[] = {
    "{\"content\":{\"url\":\"\",\"format\":\"\"}}",
    "{\"content\":{\"url\":\"http://example.com\",\"format\":\"example\"}}",
    "{\"content\":{\"url\":\"http://example.com?id=AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA\",\"format\":\"jpeg\"}}",
};
struct Expect
{
    const char* format;
    char url[a::baas::UserProfile::ImageUrlBytesForTransfer];
} Expected[] = {
    {"", ""},
    {"example", "http://example.com"},
    {"jpeg", "http://example.com?id=AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"},
};

void RunTest(const Expect& expect, const char* json) NN_NOEXCEPT
{
    t::MemoryInputStreamForRapidJson stream;
    stream.Set(json, strlen(json));

    a::baas::ImageUploadAdaptor adaptor(expect.format, strlen(expect.format));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::json::ImportJsonByRapidJson(adaptor, stream));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor.Adapt(200));

    char url[a::baas::UserProfile::ImageUrlBytesForTransfer];
    adaptor.GetUrl(url, sizeof(url));
    EXPECT_EQ(0, std::strncmp(expect.url, url, sizeof(url)));
}
} // ~namespace imageUpload

TEST(AccountAdaptor, BaasImageUpload)
{
    for (int i = 0; i < sizeof(imageUpload::Expected) / sizeof(imageUpload::Expected[0]); ++ i)
    {
        imageUpload::RunTest(
            imageUpload::Expected[i],
            imageUpload::Inputs[i]);
    }
}

namespace userResource
{
const char* Inputs[] = {
    "{\"id\":\"0000000000000001\",\"nickname\":\"\",\"thumbnailUrl\":\"\",\"extras\":{\"self\":{\"nxAccount\":\"\"}}}",
    "{\"id\":\"0123456789ABCDEF\",\"nickname\":\"UserName\",\"thumbnailUrl\":\"http://example.com\",\"extras\":{\"self\":{\"nxAccount\":\"AAAAAAAAAAAAAAAA\"}}}",
    "{\"id\":\"FEDCBA9876543210\",\"nickname\":\"UserNameUserNameUserNameUserName\",\"thumbnailUrl\":\"http://example.com?id=AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA\",\"extras\":{\"self\":{\"nxAccount\":\"AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA\"}}}",
};
struct Expect
{
    a::NetworkServiceAccountId id;
    a::baas::UserProfile profile;
} Expected[] = {
    {{0x0000000000000001ull}, {{"", ""}, ""}},
    {{0x0123456789ABCDEFull}, {{"UserName", "AAAAAAAAAAAAAAAA"}, "http://example.com"}},
    {{0xFEDCBA9876543210ull}, {{"UserNameUserNameUserNameUserName", "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"}, "http://example.com?id=AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"}}
};

void RunTest(const Expect& expect, const char* json) NN_NOEXCEPT
{
    t::MemoryInputStreamForRapidJson stream;
    stream.Set(json, strlen(json));

    t::DefaultTestStorage s;
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Mount());
    s.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(s.Setup());

    a::baas::UserResourceAdaptor adaptor(s);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::json::ImportJsonByRapidJson(adaptor, stream));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor.Adapt(200));

    EXPECT_EQ(expect.id, adaptor.GetUserId());

    a::baas::UserProfile profile;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor.LoadUserProfile(&profile));
    ASSERT_EQ(0, std::strncmp(expect.profile.base.nickname, profile.base.nickname, sizeof(profile.base.nickname)));
    ASSERT_EQ(0, std::strncmp(expect.profile.base.extraData, profile.base.extraData, sizeof(profile.base.extraData)));
    ASSERT_EQ(0, std::strncmp(expect.profile.imageUrl, profile.imageUrl, sizeof(profile.imageUrl)));
}
} // ~namespace userResource

TEST(AccountAdaptor, BaasUserResource)
{
    for (int i = 0; i < sizeof(userResource::Expected) / sizeof(userResource::Expected[0]); ++ i)
    {
        userResource::RunTest(
            userResource::Expected[i],
            userResource::Inputs[i]);
    }
}

namespace linkUser {

const char* Inputs[] = {
    "{\"id\":\"0123456789ABCDEF\",\"links\":{\"nintendoAccount\":{\"id\":\"2f71d35f47ea8b01\",\"createdAt\":1453780220,\"updatedAt\":1453780220}}}",
    "{\"id\":\"0123456789ABCDEF\",\"links\":{}}",
};
struct Expect
{
    a::NetworkServiceAccountId id;
    a::NintendoAccountId naId;
} Expected[] = {
    {{0x0123456789ABCDEFull}, {0x2f71d35f47ea8b01ull}},
    {{0x0123456789ABCDEFull}, {0x00ull}},
};

void RunTest(const Expect& expect, const char* json) NN_NOEXCEPT
{
    t::MemoryInputStreamForRapidJson stream;
    stream.Set(json, strlen(json));

    a::baas::UserLinkAdaptor adaptor;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(nn::account::json::ImportJsonByRapidJson(adaptor, stream));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(adaptor.Adapt(200));

    EXPECT_EQ(expect.id, adaptor.GetUserId());

    if (expect.naId.id != 0x00ull)
    {
        a::NintendoAccountId naId;
        ASSERT_TRUE(adaptor.TryGetNintendoAccountId(&naId));
        ASSERT_EQ(expect.naId, naId);
    }
    else
    {
        a::NintendoAccountId naId;
        ASSERT_FALSE(adaptor.TryGetNintendoAccountId(&naId));
    }
}
} // ~namespace link

TEST(AccountAdaptor, BaasUserLink)
{
    for (int i = 0; i < sizeof(linkUser::Expected) / sizeof(linkUser::Expected[0]); ++ i)
    {
        linkUser::RunTest(
            linkUser::Expected[i],
            linkUser::Inputs[i]);
    }
}
