﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <iterator>
#include <memory>
#include <vector>

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

namespace nnt {
namespace account {
class NaList;
} // namespace nnt::account
}

namespace nnt {
namespace account {

struct NasLoginInfo
{
    char email[256];
    char password[21];
};

NaList LoadNaList() NN_NOEXCEPT;

class NaList
    : public std::iterator<std::input_iterator_tag, NasLoginInfo>
{
private:
    std::shared_ptr<std::vector<NasLoginInfo>> m_List;
    int m_Count;
    int m_Cursor;

public:
    NaList() NN_NOEXCEPT
        : m_List(nullptr)
        , m_Count(0)
        , m_Cursor(0)
    {
    }
    NaList(const NasLoginInfo* list, int count) NN_NOEXCEPT
        : m_List(new std::vector<NasLoginInfo>())
        , m_Count(count)
        , m_Cursor(0)
    {
        std::vector<NasLoginInfo>& l = *m_List.get();
        l.reserve(count);
        for (auto i = 0; i < count; ++i)
        {
            l.push_back(list[i]);
        }
        NN_SDK_ASSERT(static_cast<int>(l.size()) == count);
    }
    NaList(const NaList& rhs) NN_NOEXCEPT
        : m_List(rhs.m_List)
        , m_Count(rhs.m_Count)
        , m_Cursor(rhs.m_Cursor)
    {
    }
    NaList(NaList&& rhs) NN_NOEXCEPT
        : m_List(std::move(rhs.m_List))
        , m_Count(std::move(rhs.m_Count))
        , m_Cursor(std::move(rhs.m_Cursor))
    {
        rhs.m_List = nullptr;
        rhs.m_Count = 0;
        rhs.m_Cursor = 0;
    }
    NaList& operator = (const NaList& rhs) NN_NOEXCEPT
    {
        NaList tmp(rhs);
        tmp.swap(*this);
        return *this;
    }
    NaList& operator = (NaList&& rhs) NN_NOEXCEPT
    {
        NaList tmp(std::move(rhs));
        tmp.swap(*this);
        return *this;
    }
    int Count() const NN_NOEXCEPT
    {
        return m_Count;
    }
    const NasLoginInfo& operator *() const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_Cursor < m_Count, "[nnt::account::NaList] Array index out of bounds.\n");
        return (*m_List)[m_Cursor];
    }
    const NasLoginInfo& operator[](int index) const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(index < m_Count, "[nnt::account::NaList] Array index out of bounds.\n");
        return (*m_List)[index];
    }
    NaList begin() NN_NOEXCEPT
    {
        NaList r = *this;
        r.m_Cursor = 0;
        return r;
    }
    NaList end() NN_NOEXCEPT
    {
        NaList r = *this;
        r.m_Cursor = m_Count;
        return r;
    }
    void swap(NaList& rhs) NN_NOEXCEPT
    {
        std::swap(this->m_List, rhs.m_List);
        std::swap(this->m_Count, rhs.m_Count);
        std::swap(this->m_Cursor, rhs.m_Cursor);
    }
    NaList& operator ++() NN_NOEXCEPT
    {
        ++ m_Cursor;
        return *this;
    }
    NaList operator ++(int) NN_NOEXCEPT
    {
        NaList r = *this;
        ++ m_Cursor;
        return r;
    }
    bool operator == (const NaList& rhs) const NN_NOEXCEPT
    {
        return m_Cursor == rhs.m_Cursor;
    }
    bool operator != (const NaList& rhs) const NN_NOEXCEPT
    {
        return !(*this == rhs);
    }
    bool operator < (const NaList& rhs) const NN_NOEXCEPT
    {
        return m_Cursor < rhs.m_Cursor;
    }
    bool operator <= (const NaList& rhs) const NN_NOEXCEPT
    {
        return (*this < rhs) || (*this == rhs);
    }
    bool operator > (const NaList& rhs) const NN_NOEXCEPT
    {
        return !(*this <= rhs);
    }
    bool operator >= (const NaList& rhs) const NN_NOEXCEPT
    {
        return !(*this < rhs);
    }
};

}
}
