﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testAccount_NaList.h"
#include "testAccount_Util.h"

#include <nn/account/account_ResultPrivate.h>
#include <nn/http/json/http_RapidJsonConfig.h>

#include <mutex>

// Siglo 環境での警告抑止用
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#pragma warning(push)
#pragma warning(disable : 4244)
#pragma warning(disable : 4668)
#pragma warning(disable : 4702)
#endif
#include <rapidjson/document.h>
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#pragma warning(pop)
#endif

#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_Mount.h>
#include <nn/os/os_MutexTypes.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

namespace nnt {
namespace account {
namespace {

const NasLoginInfo PredefinedLoginInfos[] = {
    {"mori_yoshihito+001.td1.account@exmx.nintendo.co.jp", "N1ntend0"},
    {"mori_yoshihito+002.td1.account@exmx.nintendo.co.jp", "N1ntend0"},
};

Mutex g_lock = NNT_ACCOUNT_MUTEX_INITIALIZER(false /* recursive */);
bool g_Initialized = false;
NaList g_NaList;

const char MountPoint[] = "674071e4";
const char Path[]       = "674071e4" ":/NintendoAccount.json";
nn::Result LoadNaListImpl() NN_NOEXCEPT
{
    nnt::account::Buffer buffer;
    // ファイルの読み込み
    {
        NN_RESULT_DO(nn::fs::MountHost(MountPoint, "C:\\siglo_account"));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountPoint);
        };

        nn::fs::FileHandle file;
        NN_RESULT_DO(nn::fs::OpenFile(&file, Path, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(file);
        };

        int64_t size;
        NN_RESULT_DO(nn::fs::GetFileSize(&size, file));

        buffer = nnt::account::Buffer(static_cast<size_t>(size) + 1);
        NN_RESULT_DO(nn::fs::ReadFile(file, 0, buffer.GetAddress(), static_cast<size_t>(size)));
        buffer.Get<char>()[size] = '\0';
    }

    // パース
    nne::rapidjson::Document document;
    document.ParseInsitu(buffer.Get<char>());
    NN_ABORT_UNLESS(!document.HasParseError(), "Parse error: %d\n", document.GetParseError());
    NN_SDK_ASSERT(document.IsObject());

    // 要素の抽出
    NN_SDK_ASSERT(document.HasMember("list"));
    auto& eList = document["list"];
    NN_SDK_ASSERT(eList.IsArray());
    auto count = static_cast<int>(eList.Size());
    std::unique_ptr<NasLoginInfo[]> list(new NasLoginInfo[count]);
    for (auto i = 0; i < count; ++ i)
    {
        const auto& eElem = eList[i];
        auto& e = list.get()[i];
        std::strncpy(e.email, eElem["id"].GetString(), sizeof(e.email));
        std::strncpy(e.password, eElem["pw"].GetString(), sizeof(e.password));

        NN_LOG("[%d] %s:%s\n", i, e.email, e.password);
    }

    g_NaList = NaList(list.get(), count);
    NN_RESULT_SUCCESS;
}

} // ~namespace nnt::account::<anonymous>

NaList LoadNaList() NN_NOEXCEPT
{
    std::lock_guard<Mutex> lock(g_lock);
    if (!g_Initialized)
    {
        // リストの読み込み等
        NN_RESULT_TRY(LoadNaListImpl())
        NN_RESULT_CATCH_ALL
        {
            g_NaList = NaList(PredefinedLoginInfos, sizeof(PredefinedLoginInfos) / sizeof(PredefinedLoginInfos[0]));
        }
        NN_RESULT_END_TRY;
        g_Initialized = true;
    }

    return NaList(g_NaList);
}

} // ~namespace nnt::account
}
