﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "testAccount_RamFs.h"
#include <nn/account/detail/account_LocalStorage.h>
#include <nn/account/detail/account_InternalConfig.h>

#include "detail/account_UuidUtil.h"

#include <cstdlib>
#include <mutex>
#include <vector>

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/result/result_HandlingUtility.h>

namespace nnt { namespace account {

const char* CreateTestDirectory(char* pathBuffer, size_t pathBufferSize) NN_NOEXCEPT;

struct DefaultVolumeNameGetter
{
    static const char* Get() NN_NOEXCEPT
    {
        return "test";
    }
};

template <typename VolumeNameGetter = DefaultVolumeNameGetter>
struct HostSaveData
{
private:
    mutable nn::os::Mutex m_Mutex;

public:
    struct Policy
    {
        typedef HostSaveData<VolumeNameGetter> SaveData;
    };

    const char* const VolumeName;

    std::vector<const char*> hostLocations;
    HostSaveData() NN_NOEXCEPT
        : m_Mutex(false)
        , VolumeName(VolumeNameGetter::Get())
    {
    }
    ~HostSaveData() NN_NOEXCEPT
    {
        nn::fs::MountHostRoot();
        for (auto p : hostLocations)
        {
            nn::fs::DeleteDirectoryRecursively(p);
            free(const_cast<char*>(p));
        }
        hostLocations.clear();
        nn::fs::UnmountHostRoot();
    }
    template <typename FileSystem>
    nn::Result Mount() NN_NOEXCEPT
    {
        const char* hostLocation = nullptr;
        const auto IsRealFs = !std::is_base_of<nnt::account::RamFs, FileSystem>::value;
        if (NN_STATIC_CONDITION(IsRealFs))
        {
            // もし FileSystem が RamFs の派生でなければ、ホスト上にディレクトリを作る。
            const size_t Length = 128;
            auto path = reinterpret_cast<char*>(malloc(Length));

            hostLocation = CreateTestDirectory(path, Length);
            NN_ABORT_UNLESS_NOT_NULL(hostLocation);
            hostLocations.push_back(hostLocation);
        }
        else
        {
            hostLocation = "__NO_MATTER__";
        }
        return FileSystem::MountHost(VolumeName, hostLocation);
    }
    template <typename FileSystem>
    nn::Result Commit() const NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }

    template <typename FileSystem>
    nn::account::detail::Uuid GenerateUuid() const NN_NOEXCEPT
    {
        return nn::account::detail::GenerateUuid();
    }
};

typedef nn::account::detail::LocalStorage<RamFs, HostSaveData<DefaultVolumeNameGetter>::Policy>
    DefaultTestStorage;

}} // ~namespace nnt::account
