﻿# -*- coding: utf-8 -*-
from __future__ import division, print_function, absolute_import, unicode_literals
from HACDllDriver import factory, Button, CaptureMode, OcrLanguage, LabelingResult, ImageRect, MorphologyConversionType

import cv2
import numpy as np
import os
import sys
import time

os.chdir( os.path.dirname(os.path.abspath(__file__)))

"""
画面キャプチャや、テンプレートマッチングによる画像検出の方法などを示したサンプルです。
サンプル内では、自動テスト支援ツールの API 利用の他に、ツールから受け取った画像リソースを python 側で処理する方法を示しています。
実行するためには、python へ opencv, numpy モジュールをインストールしておく必要があります。
また事前に、TargetManager にて対象開発機へ接続した状態で、InputDirector を起動しておく必要があります。
"""

class ImageProcessorUtil:
    def __init__(self):
        pass
    def __exit__(self):
        pass

    # ウィンドウ作成
    def show_image( self, window_name, image ):
        cv2.imshow(window_name, image)
        cv2.waitKey( 1 )

    # ウィンドウを閉じる
    def destroy_all_window( self ):
        cv2.destroyAllWindows()

    # 二値変換
    def convert_image_to_binarization(self, image, threshold=cv2.THRESH_OTSU):
        if len( image.shape ) == 3:
            image = self.convert_image_to_grayscale( image )

        threshold, ret_image = cv2.threshold(image, 0, 255, threshold)
        return ret_image

    # グレースケール変換
    def convert_image_to_grayscale( self, image ):
        if len( image.shape ) == 3:
            gray = cv2.cvtColor(image, cv2.COLOR_RGB2GRAY)
            template_height, template_width, template_channels = image.shape[:3]
        else:
            gray = image
            template_height, template_width = image.shape[:2]
            template_channels = 1

        return gray

    # 輪郭抽出
    def convert_image_to_contours_image( self, image ):
        # グレースケール変換
        gray = self.convert_image_to_grayscale( image )

        # 二値化
        image_bin = self.convert_image_to_binarization( gray )

        # 輪郭抽出
        _, template_contours, template_hierarchy = cv2.findContours(image_bin, cv2.RETR_LIST, cv2.CHAIN_APPROX_SIMPLE)

        # 輪郭描画
        ret_img = np.zeros(image.shape, dtype=np.uint8)
        ret_img = cv2.drawContours(ret_img, template_contours, -1, (255,255,255)) 

        return ret_img

    # テンプレートマッチング
    def TemplateMatching(self, image, template, threshold=0.8, method=cv2.TM_CCOEFF_NORMED):
        """
        テンプレートマッチング用メソッド

        :param image:           検出対象画像
        :param template:        検出テンプレートイメージ
        :param threshold:       しきい値
        :param method:          マッチングメソット
        :return:                マッチング結果
        :retval                 None                            マッチングする箇所が無かった場合
        :retval                 マッチング箇所の座標配列        [左上 x 座標、左上 y 座標、幅、高さ]
        """
        if len(template.shape) == 3:
            height, width, channel = template.shape
        else:
            height, width = template.shape

        if len(image.shape) == 2 or len(template.shape) == 2:
            # Modify to gray scale image
            template = cv2.cvtColor(template, cv2.COLOR_RGB2GRAY) if len(template.shape) == 3 else template
            image = cv2.cvtColor(image, cv2.COLOR_RGB2GRAY) if len(image.shape) == 3 else image

        res = cv2.matchTemplate(image, template, method)
        min_val, max_val, min_loc, max_loc = cv2.minMaxLoc(res)

        # If the method is TM_SQDIFF or TM_SQDIFF_NORMED, take minimum
        if method in [cv2.TM_SQDIFF, cv2.TM_SQDIFF_NORMED]:
            top_left = min_loc
            if min_val <= threshold:
                result = top_left[0], top_left[1], width, height, min_val
            else:
                result = None
        else:
            top_left = max_loc
            if max_val >= threshold:
                result = top_left[0], top_left[1], width, height, max_val
            else:
                result = None

        return result

    # 矩形描画
    def DrawRectAngle( self, target_image, top_left_x, top_left_y, width, height, color=(0,0,255), thickness=3 ):
        cv2.rectangle( target_image, (top_left_x, top_left_y), (top_left_x + width, top_left_y + height ), color, thickness )


class ImageProcessorExample:
    def __init__(self):
        # シリアル番号取得
        serial_number = None
        if len(sys.argv) != 1:
            serial_number = sys.argv[1]

        self.driver = factory(serial_number,capture_device_id=None, capture_mode=CaptureMode.CaptureMode_720p, is_preview=True, is_open_logreader=False)

        # 画像処理クラスインスタンス作成
        self.__image_processor = ImageProcessorUtil()
        
        # キャプチャウィンドウ名
        self.__capture_window_name = "python preview"

        # キャプチャ取得間隔[ms]
        self.__capture_span = 16

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.driver.__exit__(exc_type, exc_val, exc_tb)
        self.__image_processor.destroy_all_window()

    def detect_object_image( self, template_image_path, timeout=10000 ):
        print("Image process based on dll : detect_object_image: " )

        elapsed_time = 0
        image_id = self.driver.screen.load_image( template_image_path )
        while( elapsed_time < timeout ):
            start_time = time.time()

            result = self.driver.screen.wait_detect_object( image_id )
            if result is None:
                print( template_image_path + " was not found")
            else:
                print( template_image_path + " found at (" + str( result[0] ) + ", " + str( result[1] ) + ")")


            # 間隔
            time.sleep( self.__capture_span / 1000.0 )
            elapsed_time = elapsed_time + ((time.time() - start_time) * 1000.0)
        
        self.driver.screen.release_image( image_id )

    def dot_color(self, timeout=10000):
        print("Image process based on dll : dot_color: " )

        target_position_x = 800
        target_position_y = 495

        elapsed_time = 0
        while( elapsed_time < timeout ):
            image_id = self.driver.screen.get_image()
            start_time = time.time()
            color = self.driver.screen.get_dot_color(image_id, target_position_x, target_position_y)
            print("Color of position ({0}, {1}) is ({2}, {3}, {4})" .format(target_position_x, target_position_y, color.red, color.green, color.blue))
            
            self.driver.screen.release_image( image_id )

            # 間隔
            time.sleep( 0.1 )
            elapsed_time = elapsed_time + ((time.time() - start_time) * 1000.0)

        if self.driver.screen.wait_detect_dot_color(target_position_x, target_position_y, (0,180,180), (128,255,255)):
            print("Setting was selected")
        else:
            print("Setting was not selected")

    def apply_filter(self, timeout=10000):
        print("Image process based on dll : apply_filter: " )

        binarizatoin_filter_id = self.driver.screen.create_filter()
        self.driver.screen.set_binarization_filter(binarizatoin_filter_id)

        color_range_filter_id = self.driver.screen.create_filter()
        self.driver.screen.set_color_range_filter(color_range_filter_id, (255,255,255), (255,255,255))

        morphology_filter_id = self.driver.screen.create_filter()
        self.driver.screen.set_binarization_filter(morphology_filter_id)
        self.driver.screen.set_morphology_filter(morphology_filter_id, MorphologyConversionType.MorphologyConversionType_Erosion, 1)

        elapsed_time = 0
        while( elapsed_time < timeout ):
            start_time = time.time()

            image_id = self.driver.screen.get_image()

            result_id = self.driver.screen.apply_filter(image_id,  binarizatoin_filter_id)
            self.driver.screen.show_image(result_id, 'binarization')
            self.driver.screen.release_image( result_id )

            result_id = self.driver.screen.apply_filter(image_id,  color_range_filter_id)
            self.driver.screen.show_image(result_id, 'color_range')
            self.driver.screen.release_image( result_id )

            result_id = self.driver.screen.apply_filter(image_id,  morphology_filter_id)
            self.driver.screen.show_image(result_id, 'morphology')
            self.driver.screen.release_image( result_id )

            self.driver.screen.release_image( image_id )

            # 間隔
            time.sleep( 0.1 )
            elapsed_time = elapsed_time + ((time.time() - start_time) * 1000.0)

        self.driver.screen.delete_filter(binarizatoin_filter_id)
        self.driver.screen.delete_filter(color_range_filter_id)
        self.driver.screen.delete_filter(morphology_filter_id)

        self.driver.screen.close_image('binarization')
        self.driver.screen.close_image('color_range')
        self.driver.screen.close_image('morphology')

    def detect_object_with_filter(self, template_image_path, timeout=10000):
        print("Image process based on dll : detect_object_with_filter: " )

        image_id = self.driver.screen.load_image( template_image_path )

        self.driver.screen.show_image(image_id, 'template_image')

        filter_id = self.driver.screen.create_filter()
        self.driver.screen.set_color_range_filter(filter_id, (255,255,255), (255,255,255))

        elapsed_time = 0
        while(elapsed_time < timeout):
            start_time = time.time()

            result = self.driver.screen.wait_detect_object(image_id, capture_image_filter=filter_id)
            if result is None:
                print(template_image_path + " was not found")
            else:
                print(template_image_path + " found at (" + str(result[0]) + ", " + str(result[1]) + ")")


            # 間隔
            time.sleep(self.__capture_span / 1000.0)
            elapsed_time = elapsed_time + ((time.time() - start_time) * 1000.0)
        
        self.driver.screen.close_image('template_image')

        self.driver.screen.delete_filter(filter_id)
        self.driver.screen.release_image(image_id)

    def ocr_sample(self, target_image_path, timeout=10000):
        print("Image process based on dll : ocr_sample: " )

        image_id = self.driver.screen.load_image(target_image_path)

        self.driver.screen.show_image(image_id, 'target_image')

        result = self.driver.screen.execute_ocr(image_id, language=OcrLanguage.OcrLanguage_English)

        print("----- Detected texts -----")
        for i in range(0, result[0]):
            ocr_result = self.driver.screen.get_ocr_result(result[1], i)
            print(ocr_result[0])
        self.driver.screen.release_ocr_result(result[1])
        print("--------------------------")

        result_rect = self.driver.screen.detect_text(image_id, r"Sample".encode('utf-8'), language=OcrLanguage.OcrLanguage_English)
        if result_rect is not None:
            detected_image_id = self.driver.screen.extract_image(image_id, result_rect)
            self.driver.screen.show_image(detected_image_id, 'detected_text')
            self.driver.screen.release_image(detected_image_id)

        time.sleep(timeout / 1000.0)

        self.driver.screen.close_image('target_image')
        self.driver.screen.close_image('detected_text')
        self.driver.screen.release_image(image_id)

    def labeling_sample(self, target_image_path, timeout=10000):
        print("Image process based on dll : labeling_sample: " )

        image_id = self.driver.screen.load_image(target_image_path)

        filter_id = self.driver.screen.create_filter()
        self.driver.screen.set_color_range_filter(filter_id, (255,0,0), (255,255,255))
        result_id = self.driver.screen.apply_filter(image_id,  filter_id)
        self.driver.screen.delete_filter(filter_id)

        labeling_result = self.driver.screen.execute_labeling(result_id)
        results_list = self.driver.screen.get_labeling_results(labeling_result[1], labeling_result[0])

        count = 0
        for result in results_list:
            # 検出領域の描画
            detected_image_id = self.driver.screen.extract_image(image_id, result.rect)
            self.driver.screen.show_image(detected_image_id, 'labeling_image{0}'.format(count))
            self.driver.screen.release_image(detected_image_id)
            count += 1

        self.driver.screen.release_labeling_result(labeling_result[1])

        time.sleep(timeout / 1000.0)

        for i in range(count):
            self.driver.screen.close_image('labeling_image{0}'.format(i))
        self.driver.screen.release_image(result_id)
        self.driver.screen.release_image(image_id)

    def preview_capture( self, timeout=10000 ):
        print("Image process based on python : preview capture: ")

        elapsed_time = 0
        while( elapsed_time < timeout ):
            start_time = time.time()

            # キャプチャ画像データ取得
            image_data = self.driver.screen.get_image_data()
            if image_data is None:
                print("get_image_data is failed")
                continue

            height  = image_data[0]
            width   = image_data[1]
            channel = image_data[2]
            buffer  = image_data[3]

            capture_image = np.frombuffer(buffer, dtype=np.uint8).reshape((height, width,  channel))

            # プレビュー表示
            self.__image_processor.show_image( self.__capture_window_name, capture_image)

            # 間隔
            time.sleep( self.__capture_span / 1000.0 )
            elapsed_time = elapsed_time + ((time.time() - start_time) * 1000.0)


    def trace_image( self, template_image_path, timeout=10000 ):
        print("Image process based on python : trace image: " )
        template_image = cv2.imread( template_image_path )

        elapsed_time = 0
        while( elapsed_time < timeout ):
            start_time = time.time()

            # キャプチャ画像データ取得
            image_data = self.driver.screen.get_image_data()
            if image_data is None:
                print("get_image_data is failed")
                continue

            height  = image_data[0]
            width   = image_data[1]
            channel = image_data[2]
            buffer  = image_data[3]

            capture_image = np.frombuffer(buffer, dtype=np.uint8).reshape((height, width,  channel))
            
            # テンプレートマッチング
            result = self.__image_processor.TemplateMatching( capture_image, template_image, 0.8 )

            # 検出領域の描画
            if result is not None:
                self.__image_processor.DrawRectAngle( capture_image, result[0], result[1], result[2], result[3] )

            # プレビュー表示
            self.__image_processor.show_image( self.__capture_window_name, capture_image)
            self.__image_processor.show_image( "Python template preview" , template_image )

            # 間隔
            time.sleep( self.__capture_span / 1000.0 )
            elapsed_time = elapsed_time + ((time.time() - start_time) * 1000.0)

    def show_convert_functions( self, timeout=10000):
        print("Image process based on python : show some convert functions:" )

        elapsed_time = 0
        while( elapsed_time < timeout ):
            start_time = time.time()

            # キャプチャ画像データ取得
            image_data = self.driver.screen.get_image_data()
            if image_data is None:
                print("get_image_data is failed")
                continue

            height  = image_data[0]
            width   = image_data[1]
            channel = image_data[2]
            buffer  = image_data[3]

            capture_image = np.frombuffer(buffer, dtype=np.uint8).reshape((height, width,  channel))

            # グレースケール変換
            gray_image = self.__image_processor.convert_image_to_grayscale( capture_image)
            self.__image_processor.show_image( self.__capture_window_name + "_gray", gray_image)

            # 二値変換
            binary_image = self.__image_processor.convert_image_to_binarization( capture_image)
            self.__image_processor.show_image( self.__capture_window_name + "_binary", binary_image)

            # 輪郭抽出
            edge_image = self.__image_processor.convert_image_to_contours_image( capture_image )
            self.__image_processor.show_image( self.__capture_window_name + "_contours", edge_image )

            # プレビュー表示
            self.__image_processor.show_image( self.__capture_window_name, capture_image)

            # 間隔
            time.sleep( self.__capture_span / 1000.0 )
            elapsed_time = elapsed_time + ((time.time() - start_time) * 1000.0)

if __name__ == "__main__":
    # sys.argv[1] = 対象のシリアルナンバー
    with ImageProcessorExample() as ipe:
        # dll ライブラリベースの画像処理サンプル
        ipe.detect_object_image( "TemplateImage/Home/Config_Icon.png" )
        ipe.dot_color()
        ipe.apply_filter()
        ipe.detect_object_with_filter("ImageProcessSample/Color_Range.png")
        ipe.ocr_sample("ImageProcessSample/Ocr_Sample.png")
        ipe.labeling_sample("ImageProcessSample/Labeling_Sample.png")

        # python ベースの画像処理サンプル
        ipe.preview_capture()
        ipe.trace_image( "TemplateImage/Home/Config_Icon.png" )
        ipe.show_convert_functions()
