﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
 * @brief
 * MovieVideoOutputHandler class for processing video data from movie::decoder
 *
 * @details
 * Purpose:
 *     The main purpose of MovieVideoOutputHandler class is get decoded video data from movie::decoder and render
 * it to an output device capable of displaying the video frames. This class receives callbacks from
 * MovieDecoderEventHandler when video output buffer is available. The buffer index is stored in a list.
 * MovieVideoOutputHandler has a thread which will submit video frames to NvnVideoRenderer. MovieVideoOutputHandler
 * supports usage of CPU buffers for video decoder and Native texture (NVN Texture). When decoder is setup for CPU
 * buffers a NVN buffer is passed to decoder and a NVN texture handle is passed to the decoder in case of native texture
 * mode.
 *
 * Setup:
 *     MoviePlayer class will create an instance of MovieVideoOutputHandler. Based on decoder setup NvnVideoRenderer
 * will be set up to allocate either NVN buffers or NVN textures. When decoder are set up for NVN texture, the decoder
 * output will be in ABGR format. Renderer does not need to do any color conversion. If the decoder is set up to use
 * CPU buffers, the decoder output will be NV12YUV. In this case renderer need to convert decoder output from NV12YUV
 * to ABGR format. During initialization, a thread will created to submit video frames. The thread will use MediaClock
 * for syncing video with audio. Movie decoder handle is passed to MovieVideoOutputHandler with Open() API. This will
 * be used to pull decoded video frames from decoder.
 *
 * Main Processing:
 *     When client calls Start() API, MovieVideoOutputHandler rendering thread will be started. This will result in the
 * thread calling Render() API. This API will check whether there are any decoded video frames available. If there is
 * a decoder video buffer, the presentation time from the buffer is passed to AVSync() API. This will check whether
 * video frame need to be dropped or presented to renderer using MediaClock. If video frame is not dropped, it will
 * be prepared for rendering. NVN video buffer or Texture handle is obtained from NvnVideoRenderer and is submitted
 * to decoder. Video frame is obtained from decoder and sent to NvnVideoRenderer for displaying. Decoder video buffer
 * is released. In texture mode if sync point is passed to the decoder, the renderer needs to use the sync point for
 * decoder to complete blit. Decoder will return immediately after issuing a blit from decoder texture to client
 * texture. When last video frame is received (checked using EOS flag), and if an event is registered it will be
 * signaled.
 *
 * Teardown:
 *    Client need to call Stop() to stop MovieVideoOutputHandler. Any video decoder buffers remaining will be released
 * back to the decoder. When Finalize() API is called, the main thread is stopped and resources are released. When
 * MovieVideoOutputHandler instance is deleted NvnVideoRenderer Finalize() is called and the instance is deleted.
 *
 */
#include <nns/mm/mm_MovieVideoOutputHandler.h>
MovieVideoOutputHandler::MovieVideoOutputHandler(nn::vi::NativeWindowHandle nativeWindow,
    movie::DecoderMode videoDecoderMode,
    movie::DecoderOutputFormat videoDecoderOutputFormat)
    : m_NativeWindow(nativeWindow),
      m_VideoDecoder(nullptr),
      m_NvnVideoRenderer(nullptr),
      m_ThreadStarted(false),
      m_ThreadDone(false),
      m_ThreadCreated(false),
      m_PresentationTimeUs(0),
      m_ThreadstackSize(1024 * 256),
      m_Threadstack(nullptr),
      m_YuvBufferSize(0),
      m_Width(-1),
      m_Height(-1),
      m_AllFramesRendered(false),
      m_MediaClock(nullptr),
      m_VideoNeedDelayTimeUs(0),
      m_VideoStartTime(0),
      m_DroppedVideoFrames(0),
      m_UpdateVideoStartTime(false),
      m_LastPresentationTimeUs(0),
      m_IsPaused(false),
      m_PlaybackPositionUs(0),
      m_VideoOutputBuffersReceived(0),
      m_VideoDecoderMode(videoDecoderMode),
      m_VideoDecoderOutputFormat(videoDecoderOutputFormat)
{
    NvnVideoRenderer::NvnOutputFormat nvnOutputFormat = NvnVideoRenderer::NvnOutputFormat_YuvNv12NvnBuffer;
    if( m_VideoDecoderOutputFormat == movie::DecoderOutputFormat_VideoColorAbgr )
    {
        nvnOutputFormat = NvnVideoRenderer::NvnOutputFormat_AbgrNvnTexture;
    }
    m_NvnVideoRenderer = new NvnVideoRenderer(m_NativeWindow, nvnOutputFormat);
    if( m_NvnVideoRenderer != nullptr )
    {
        m_NvnVideoRenderer->Initialize();
    }
}

MovieVideoOutputHandler::~MovieVideoOutputHandler()
{
    if( m_NvnVideoRenderer != nullptr )
    {
        m_NvnVideoRenderer->Finalize();
        delete m_NvnVideoRenderer;
        m_NvnVideoRenderer = nullptr;
    }
}

movie::Status MovieVideoOutputHandler::Initialize(MediaClock *mediaClock, nn::os::EventType *videoRenderComplete)
{
    nn::Result nnResult = nn::ResultSuccess();
    movie::Status movieStatus = movie::Status_UnknownError;
    m_VideoRenderComplete = videoRenderComplete;
    m_ThreadDone = false;
    m_ThreadStarted = false;
    m_PresentationTimeUs = 0ll;
    m_ThreadCreated = false;
    m_MediaClock = mediaClock;
    m_DroppedVideoFrames = 0ll;
    m_UpdateVideoStartTime = true;
    m_IsPaused = false;
    m_PlaybackPositionUs = 0;
    m_VideoBuffers.reserve(64);
    nn::os::InitializeMutex( &m_VideoMutex, false, 0 );
    m_Threadstack = aligned_alloc(nn::os::ThreadStackAlignment, m_ThreadstackSize);
    if( m_Threadstack == nullptr )
    {
        return movie::Status_OutOfMemory;
    }
    nnResult = nn::os::CreateThread(
        &m_ThreadType,
        &MovieVideoOutputHandlerThreadFunction,
        ( void* )this,
        m_Threadstack,
        m_ThreadstackSize,
        nn::os::DefaultThreadPriority);
    m_ThreadCreated = true;
    if( nnResult.IsSuccess() )
    {
        nn::os::SetThreadName(&m_ThreadType, "MovieVideoOutputHandlerThread");
        movieStatus = movie::Status_Success;
    }
    if( m_NvnVideoRenderer == nullptr )
    {
        return movie::Status_OutOfMemory;
    }
    return movieStatus;
}

movie::Status MovieVideoOutputHandler::Finalize()
{
    if( m_ThreadCreated == true )
    {
        if( m_ThreadStarted == true )
        {
            m_ThreadDone = true;
            nn::os::WaitThread(&m_ThreadType);
            m_ThreadStarted = false;
            m_ThreadDone = false;
        }
        nn::os::DestroyThread(&m_ThreadType);
        m_ThreadCreated = false;
    }
    if( m_Threadstack != nullptr )
    {
        free(m_Threadstack);
        m_Threadstack = nullptr;
    }
    nn::os::FinalizeMutex( &m_VideoMutex );
    return movie::Status_Success;
}

movie::Status MovieVideoOutputHandler::Open(movie::Decoder* videoDecoder, int32_t width, int32_t height)
{
    m_VideoDecoder = videoDecoder;
    m_Width = width;
    m_Height = height;
    return movie::Status_Success;
}

movie::Status MovieVideoOutputHandler::Start(int32_t width, int32_t height)
{
    nn::os::LockMutex(&m_VideoMutex);
    movie::Status movieStatus = movie::Status_UnknownError;
    m_Width = width;
    m_Height = height;
    m_PresentationTimeUs = 0ll;
    m_UpdateVideoStartTime = true;
    m_AllFramesRendered = false;
    m_LastPresentationTimeUs = m_PresentationTimeUs;

    movieStatus = m_NvnVideoRenderer->ResizeTextures(m_Width, m_Height);
    if( movieStatus == movie::Status_Success )
    {
        if( m_ThreadStarted == false )
        {
            nn::os::StartThread(&m_ThreadType);
            m_ThreadStarted = true;
        }
    }
    if( movieStatus != movie::Status_Success )
    {
        m_AllFramesRendered = true;
    }
    nn::os::UnlockMutex(&m_VideoMutex);
    return movieStatus;
}

movie::Status MovieVideoOutputHandler::Render()
{
    VideoData* videoData;
    movie::Buffer videoBufferData;

    nn::os::LockMutex( &m_VideoMutex );
    int videoListSize = m_VideoBuffers.size();
    if( videoListSize > 0 )
    {
        videoData = &m_VideoBuffers.front();
    }
    else
    {
        nn::os::UnlockMutex( &m_VideoMutex );
        return movie::Status_Success;
    }
    int yStride = 0;
    int uvOffset = 0;
    int yOffset = 0;
    int colorSpace = 0;
    int64_t presentationTimeUs = videoData->presentationTimeUs;
    uint32_t flags = videoData->flags;
    uint32_t index = videoData->index;

    bool frameDropped = AVSync(presentationTimeUs);
    if( frameDropped == true )
    {
        if( flags == movie::BufferFlags_EndOfStream )
        {
            m_AllFramesRendered = true;
            if( m_DroppedVideoFrames > 0 )
            {
                m_DroppedVideoFrames -= 1;
            }
            if( m_VideoRenderComplete != nullptr )
            {
                nn::os::SignalEvent(m_VideoRenderComplete);
            }
        }
        m_VideoDecoder->ReleaseOutputBufferIndex(index);
        m_VideoBuffers.erase(m_VideoBuffers.begin());
        nn::os::UnlockMutex(&m_VideoMutex);
        return movie::Status_Success;
    }
    m_PlaybackPositionUs = presentationTimeUs;
    bool frameAvailableForRendering = false;

    if( ( m_VideoDecoderMode == movie::DecoderMode_NativeTexture ) &&
        ( m_VideoDecoderOutputFormat == movie::DecoderOutputFormat_VideoColorAbgr ) )
    {
        void *rgbTextureHandle = nullptr;
        int rgxTextureIndex = 0;
        size_t rgbTextureHandleSize = 0;
        void *rgbNvnDevHandle = nullptr;
        void *rgbNvnSyncHandle = nullptr;
        m_NvnVideoRenderer->GetNvnVideoSync(&rgbNvnDevHandle,&rgbNvnSyncHandle);
        m_NvnVideoRenderer->GetNvnVideoTexture(rgxTextureIndex, &rgbTextureHandle, &rgbTextureHandleSize);
        videoBufferData.SetDataAndCapacity(rgbTextureHandle, rgbTextureHandleSize);
        videoBufferData.SetNvnDev(rgbNvnDevHandle);
        videoBufferData.SetNvnSync(rgbNvnSyncHandle);
        videoBufferData.SetRange(0, m_YuvBufferSize);
    }
    else if( ( m_VideoDecoderMode == movie::DecoderMode_Cpu ) &&
        ( m_VideoDecoderOutputFormat == movie::DecoderOutputFormat_VideoColorNv12 ) )
    {
        char *bufferMemory = nullptr;
        int yuvBufferIndex = 0;
        size_t yuvBufferSize = 0;
        m_NvnVideoRenderer->GetNvnVideoBuffer(yuvBufferIndex, &bufferMemory, &yuvBufferSize);
        m_YuvBufferSize = yuvBufferSize;
        videoBufferData.SetDataAndCapacity(bufferMemory, m_YuvBufferSize);
        videoBufferData.SetRange(0, m_YuvBufferSize);
    }
    else
    {
        return movie::Status_NotSupported;
    }

    movie::MediaData format;
    movie::Status status = m_VideoDecoder->GetOutputBufferFormat(videoData->index, &format);
    if( status == movie::Status_Success )
    {
        int32_t width = 0;
        int32_t newWidth = m_Width;
        if( format.FindInt32("width", &width) )
        {
            newWidth = width;
        }
        int32_t height = 0;
        int32_t newHeight = m_Height;
        if( format.FindInt32("height", &height) )
        {
            newHeight = height;
        }

        if( ( newWidth != m_Width ) || ( newHeight != m_Height ) )
        {
            m_Width = newWidth;
            m_Height = newHeight;
            m_NvnVideoRenderer->ResizeTextures(m_Width, m_Height);
        }
        format.FindInt32("nv12-y-stride", &yStride);
        format.FindInt32("nv12-uv-offset", &uvOffset);
        format.FindInt32("nv12-y-offset", &yOffset);

        format.Clear();
        status = m_VideoDecoder->GetOutputFormat(&format);
        if( status == movie::Status_Success )
        {
            format.FindInt32("nv12-colorspace", &colorSpace);
        }

        movie::Status status = m_VideoDecoder->GetOutputBuffer(videoData->index, &videoBufferData);
        if( status == movie::Status_Success )
        {
            m_VideoDecoder->ReleaseOutputBufferIndex(videoData->index);
            m_VideoBuffers.erase(m_VideoBuffers.begin());
            frameAvailableForRendering = true;
        }
    }
    if( videoData->flags == movie::BufferFlags_EndOfStream )
    {
        m_AllFramesRendered = true;
        if( m_VideoRenderComplete != nullptr )
        {
            nn::os::SignalEvent(m_VideoRenderComplete);
        }
    }
    nn::os::UnlockMutex( &m_VideoMutex );
    if( frameAvailableForRendering == true )
    {
        m_NvnVideoRenderer->Draw(m_Width, m_Height, yOffset, yStride, uvOffset, colorSpace);
    }
    return movie::Status_Success;
}//NOLINT(impl/function_size)

void MovieVideoOutputHandler::Stop()
{
    nn::os::LockMutex( &m_VideoMutex );
    VideoData* videoData;
    int videoListSize = m_VideoBuffers.size();
    for(int i=0; i< videoListSize; i++)
    {
        videoData = &m_VideoBuffers.front();
        m_VideoDecoder->ReleaseOutputBufferIndex(videoData->index);
        m_VideoBuffers.erase(m_VideoBuffers.begin());
    }
    m_VideoOutputBuffersReceived = 0;
    nn::os::UnlockMutex( &m_VideoMutex );
}

void MovieVideoOutputHandler::Pause()
{
    nn::os::LockMutex(&m_VideoMutex);
    m_IsPaused = true;
    m_MediaClock->ClearAnchorTime();
    nn::os::UnlockMutex(&m_VideoMutex);
}

void MovieVideoOutputHandler::Flush()
{
    nn::os::LockMutex(&m_VideoMutex);
    m_VideoBuffers.clear();
    nn::os::UnlockMutex(&m_VideoMutex);
}

void MovieVideoOutputHandler::Resume()
{
    nn::os::LockMutex(&m_VideoMutex);
    m_UpdateVideoStartTime = true;
    m_IsPaused = false;
    nn::os::UnlockMutex(&m_VideoMutex);
}

bool MovieVideoOutputHandler::DoneProcessing()
{
    return m_ThreadDone;
}

void MovieVideoOutputHandler::Close()
{
    if( m_DroppedVideoFrames > 0 )
    {
        NN_LOG("MovieVideoOutputHandler::AVSync: Dropped Frame count: %d \n", m_DroppedVideoFrames);
    }
}

void MovieVideoOutputHandler::OnOutputAvailable(int32_t index, int64_t presentationTimeUs, uint32_t flags)
{
    VideoData videoData;
    videoData.index = index;
    videoData.presentationTimeUs = presentationTimeUs;
    videoData.flags = flags;
    nn::os::LockMutex( &m_VideoMutex );
    m_VideoBuffers.push_back(videoData);
    m_VideoOutputBuffersReceived += 1;
    nn::os::UnlockMutex( &m_VideoMutex );
}

int64_t MovieVideoOutputHandler::GetRealTimeUs(int64_t mediaTimeUs, int64_t nowUs)
{
    int64_t realUs = 0;
    if( m_MediaClock->GetRealTimeForMediaTime(mediaTimeUs, &realUs) != movie::Status_Success )
    {
        // If failed to get current position, then just play out video immediately without delay.
        realUs = nowUs + ( mediaTimeUs - m_LastPresentationTimeUs );
        return realUs;
    }
    return realUs;
}

bool MovieVideoOutputHandler::AVSync(int64_t presentationTimeUs)
{
    bool frameDropped = false;
    m_PresentationTimeUs = presentationTimeUs;
    if( m_UpdateVideoStartTime == true )
    {
        m_UpdateVideoStartTime = false;
        m_VideoStartTime = Clock::GetNowUs();
        m_MediaClock->UpdateAnchorTime(m_PresentationTimeUs, m_VideoStartTime, INT64_MAX);
    }
    int64_t nowUs = Clock::GetNowUs();
    int64_t realTimeUs = GetRealTimeUs(m_PresentationTimeUs, nowUs);
    int64_t videoLatebyUs = nowUs - realTimeUs;
    m_LastPresentationTimeUs = m_PresentationTimeUs;
    m_VideoNeedDelayTimeUs = videoLatebyUs * -1;
    if( videoLatebyUs > 40000 )
    {
        frameDropped = true;
        m_DroppedVideoFrames += 1;
    }
    if( m_VideoNeedDelayTimeUs < 0 )
    {
        m_VideoNeedDelayTimeUs = 0;
    }
    return frameDropped;
}

int64_t MovieVideoOutputHandler::GetVideoPostDelayUs()
{
    // This thread may start much early than first video buffer arrived.
    // To prevent CPU hog, this thread sleeps if no activity in this thread.
    if( m_VideoNeedDelayTimeUs < 2000 )
    {
        m_VideoNeedDelayTimeUs = 2000;
    }
    else if( m_VideoNeedDelayTimeUs > 300000 )
    {
        m_VideoNeedDelayTimeUs = 300000;
    }
    return m_VideoNeedDelayTimeUs;
}

movie::Status MovieVideoOutputHandler::GetPlaybackPosition(int64_t *playbackPositionUs) const
{
    if( playbackPositionUs == nullptr )
    {
        return movie::Status_ErrorBadValue;
    }
    else
    {
        *playbackPositionUs = m_PlaybackPositionUs;
        return movie::Status_Success;
    }
}

bool MovieVideoOutputHandler::CanStartRendering()
{
    if( m_VideoOutputBuffersReceived >= 2 )
    {
        return true;
    }
    else
    {
        return false;
    }
}

void MovieVideoOutputHandlerThreadFunction(void *arg)
{
    MovieVideoOutputHandler *movieVideoOutputHandler = ( MovieVideoOutputHandler*) arg;
    if( movieVideoOutputHandler != nullptr )
    {
        for( ;; )
        {
            if( true == movieVideoOutputHandler->DoneProcessing() )
            {
                break;
            }
            if( false == movieVideoOutputHandler->IsPaused() )
            {
                movieVideoOutputHandler->Render();
            }
            int64_t renderSleepTimeUs = movieVideoOutputHandler->GetVideoPostDelayUs();
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(renderSleepTimeUs));
        }
    }
}
