﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <vector>

#include <nn/nn_Assert.h>
#define NN_PERF_PROFILE_ENABLED
#include <nn/perf.h>

#if NN_GFX_IS_TARGET_NVN
    #include <nvn/nvn.h>
    #include <nvn/nvn_FuncPtrInline.h>
#endif

#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#include "gfxLogGraphicsSystem.h"

namespace nns {
namespace gfxLog {

const size_t MemoryPoolSize = 16 * 1024 * 1024;
const size_t MemoryPoolSizeInvisible = 64 * 1024 * 1024;

const int DescriptorPoolSlotCount = 2048;

const size_t CommandBufferCommandMemoryChunkSize = 512 * 1024;
const size_t CommandBufferControlMemoryChunkSize = 64 * 1024;

const int SwapInterval = 1;

/**
* @brief コンストラクタです。
*
* @details コンストラクタです。
*/
GraphicsSystem::GraphicsSystem() NN_NOEXCEPT
    : m_pApplicationHeap(NULL)
    , m_MemoryPool()
    , m_pMemoryPool(NULL)
    , m_MemoryPoolOffset(0)
    , m_MemoryPoolCommandOffset(0)
    , m_MemoryPoolInvisible()
    , m_pMemoryPoolInvisible(NULL)
    , m_MemoryPoolInvisibleOffset(0)
    , m_BufferDescriptorPool()
    , m_TextureDescriptorPool()
    , m_TextureSlotCount(0)
    , m_SamplerDescriptorPool()
    , m_SamplerSlotCount(0)
    , m_pDisplay(NULL)
    , m_DisplayWidth(1280)
    , m_DisplayHeight(720)
    , m_pLayer(NULL)
    , m_Device()
    , m_SwapChain()
    , m_Queue()
    , m_Fence()
    , m_ColorTargetTexture()
    , m_DepthStencilTexture()
    , m_DepthStencilView()
    , m_CommandBuffer()
    , m_ViewportScissorState()
    , m_BlendState()
    , m_DepthStencilState()
    , m_RasterizerState()
    , m_DebugFontHeap(NULL)
    , m_DebugFont()
    , m_CommandBufferControlMemoryChunks()
    , m_ColorBuffer()
    , m_ColorTargetView()
    , m_ColorTextureView()
    , m_ColorTextureDescriptorIndex(0)
    , m_Sampler()
    , m_BufferDescriptorIndex(0)
    , m_SamplerDescriptorIndex(0)
    , m_ColorDescriptorSlot()
    , m_SamplerSlot()
    , m_BlurColorBuffer()
    , m_BlurColorTargetView()
    , m_BlurColorTextureView()
    , m_VisiblePoolMemorySize(0)
    , m_InvisiblePoolMemorySize(0)
{
}

/**
* @brief アロケータを設定します。
*
* @param[in]    pValue    設定するアロケータ
* @details このクラスで使用するアロケータを設定します。
*/
void GraphicsSystem::SetApplicationHeap(nn::mem::StandardAllocator* pValue) NN_NOEXCEPT
{
    m_pApplicationHeap = pValue;
}

/**
* @brief デバイスハンドルを取得します。
*
* @return  デバイスハンドルを返します。
* @details デバイスハンドルを取得します。
*/
nn::gfx::Device& GraphicsSystem::GetDevice() NN_NOEXCEPT
{
    return m_Device;
}

/**
* @brief コマンドバッファを取得します。
*
* @return  コマンドバッファを返します。
* @details コマンドバッファを取得します。
*/
nn::gfx::CommandBuffer& GraphicsSystem::GetCommandBuffer() NN_NOEXCEPT
{
    return m_CommandBuffer;
}

/**
* @brief デバッグフォントライターを取得します。
*
* @return  デバッグフォントライターを返します。
* @details デバッグフォントライターを取得します。
*/
nn::gfx::util::DebugFontTextWriter& GraphicsSystem::GetDebugFont() NN_NOEXCEPT
{
    return m_DebugFont;
}

/**
* @brief 画面幅を取得します。
*
* @return  画面幅を返します。
* @details 画面幅を取得します。
*/
int GraphicsSystem::GetDisplayWidth() const NN_NOEXCEPT
{
    return m_DisplayWidth;
}

/**
* @brief 画面高さを取得します。
*
* @return  画面高さを返します。
* @details 画面高さを取得します。
*/
int GraphicsSystem::GetDisplayHeight() const NN_NOEXCEPT
{
    return m_DisplayHeight;
}

/**
* @brief 初期化を行います。
*
* @details 初期化を行います。
*/
void GraphicsSystem::Initialize() NN_NOEXCEPT
{
    nn::vi::Initialize();
    this->InitializeDisplay();
    nn::gfx::Initialize();
    this->InitializeDevice();

#if NN_GFX_IS_TARGET_NVN
    nn::gfx::Device::DataType& deviceData = nn::gfx::AccessorToData(&m_Device);
    nvnDeviceGetInteger(deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &m_TextureSlotCount);
    nvnDeviceGetInteger(deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &m_SamplerSlotCount);
#endif

    this->InitializeMemoryPool();
    this->InitializeMemoryPoolInvisible();
    this->InitializeSwapChain();
    this->InitializeQueue();
    this->InitializeFence();
    this->InitializeDepthStencilTexture();
    this->InitializeDepthStencilView();
    this->InitializeBufferDescriptorPool();
    this->InitializeTextureDescriptorPool();
    this->InitializeSamplerDescriptorPool();
    this->InitializeBlurBuffer();
    this->InitializeCommandBuffer();
    this->InitializeViewportScissorState();
    this->InitializeBlendState();
    this->InitializeDepthStencilState();
    this->InitializeRasterizerState();

    InitializeColorBuffer(
        &m_ColorTextureView,
        &m_ColorTargetView,
        &m_ColorBuffer
    );

    m_ColorTextureDescriptorIndex = RegisterTextureViewSlot(
        &m_ColorDescriptorSlot,
        m_ColorTextureView
    );

    InitializeSampler(&m_Sampler);
    m_SamplerDescriptorIndex = RegisterSamplerSlot(&m_SamplerSlot, m_Sampler);

    NN_ASSERT(m_MemoryPoolOffset < static_cast<ptrdiff_t>(m_VisiblePoolMemorySize));
    NN_ASSERT(m_MemoryPoolInvisibleOffset < static_cast<ptrdiff_t>(m_InvisiblePoolMemorySize));
}

/**
* @brief 終了処理を行います。
*
* @details 終了処理を行います。
*/
void GraphicsSystem::Finalize() NN_NOEXCEPT
{
    FinalizeSampler(&m_Sampler);
    FinalizeColorBuffer(
        &m_ColorTextureView,
        &m_ColorTargetView,
        &m_ColorBuffer
    );

    m_RasterizerState.Finalize(&m_Device);
    m_DepthStencilState.Finalize(&m_Device);
    this->FinalizeBlendState();
    this->FinalizeViewportScissorState();
    this->ResetCommandBuffer();
    m_CommandBuffer.Finalize(&m_Device);
    m_SamplerDescriptorPool.Finalize(&m_Device);
    m_TextureDescriptorPool.Finalize(&m_Device);
    m_BufferDescriptorPool.Finalize(&m_Device);
    m_BlurColorTextureView.Finalize(&m_Device);
    m_BlurColorTargetView.Finalize(&m_Device);
    m_BlurColorBuffer.Finalize(&m_Device);
    m_DepthStencilView.Finalize(&m_Device);
    m_DepthStencilTexture.Finalize(&m_Device);
    m_Fence.Finalize(&m_Device);
    m_Queue.Finalize(&m_Device);
    m_SwapChain.Finalize(&m_Device);
    this->FinalizeMemoryPoolInvisible();
    this->FinalizeMemoryPool();
    m_Device.Finalize();
    nn::gfx::Finalize();
    nn::vi::DestroyLayer(m_pLayer);
    nn::vi::CloseDisplay(m_pDisplay);
    nn::vi::Finalize();
}

/**
* @brief 描画の開始処理を行います。
*
* @details 描画の開始処理を行います。
*/
void GraphicsSystem::BeginDraw() NN_NOEXCEPT
{
    this->ResetCommandBuffer();
    this->AddCommandMemoryToCommandBuffer(&m_CommandBuffer);
    this->AddControlMemoryToCommandBuffer(&m_CommandBuffer);

    m_CommandBuffer.Begin();

    m_CommandBuffer.SetDescriptorPool(&m_BufferDescriptorPool);
    m_CommandBuffer.SetDescriptorPool(&m_TextureDescriptorPool);
    m_CommandBuffer.SetDescriptorPool(&m_SamplerDescriptorPool);

    nn::gfx::ColorTargetView* pTarget = NULL;
    pTarget = m_SwapChain.AcquireNextScanBufferView();
    m_CommandBuffer.ClearColor(
        pTarget,
        0.0f,
        0.0f,
        0.0f,
        1.0f,
        0);

    m_CommandBuffer.ClearDepthStencil(
        &m_DepthStencilView,
        1.0f,
        0,
        nn::gfx::DepthStencilClearMode_DepthStencil,
        0);

    nn::gfx::ColorTargetView* pColorTargetView = &m_ColorTargetView;
    m_CommandBuffer.SetRenderTargets(
        1,
        &pColorTargetView,
        &m_DepthStencilView);

    m_CommandBuffer.SetViewportScissorState(&m_ViewportScissorState);
    m_CommandBuffer.SetBlendState(&m_BlendState);
    m_CommandBuffer.SetDepthStencilState(&m_DepthStencilState);
    m_CommandBuffer.SetRasterizerState(&m_RasterizerState);

    m_CommandBuffer.SetRenderTargets(1, &pTarget, NULL);

    // カラーバッファをテクスチャとして利用するのでキャッシュをフラッシュします
    m_CommandBuffer.FlushMemory(nn::gfx::GpuAccess_ColorBuffer);
}

/**
* @brief 描画の終了処理を行います。
*
* @details 描画の終了処理を行います。
*/
void GraphicsSystem::EndDraw() NN_NOEXCEPT
{
    m_CommandBuffer.FlushMemory(nn::gfx::GpuAccess_QueryBuffer);
    m_CommandBuffer.End();
    m_Queue.ExecuteCommand(&m_CommandBuffer, &m_Fence);
    m_Queue.Flush();

    m_Queue.Present(&m_SwapChain, SwapInterval);
    m_Queue.Sync();
}

/**
* @brief テクスチャービューの登録を行います。
*
* @param[out]    pOutValue    ディスクリプタスロット
* @param[in ]    textureView  テクスチャービュー
* @details テクスチャービューの登録を行います。
* @return  テクスチャースロット数を返します。
*/
int GraphicsSystem::RegisterTextureViewSlot(
    nn::gfx::DescriptorSlot* pOutValue,
    const nn::gfx::TextureView& textureView ) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutValue);

    const int Slot = m_TextureSlotCount;
    m_TextureDescriptorPool.BeginUpdate();
    m_TextureDescriptorPool.SetTextureView(Slot, &textureView);
    m_TextureDescriptorPool.EndUpdate();
    m_TextureDescriptorPool.GetDescriptorSlot(pOutValue, Slot);

    ++m_TextureSlotCount;
    return Slot;
}

/**
* @brief サンプラーの登録を行います。
*
* @param[out]    pOutValue    サンプラースロット
* @param[in ]    sampler      サンプラー
* @details サンプラーの登録を行います。
* @return  サンプラースロット数を返します。
*/
int GraphicsSystem::RegisterSamplerSlot(
    nn::gfx::DescriptorSlot* pOutValue,
    const nn::gfx::Sampler& sampler) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutValue);

    const int Slot = m_SamplerSlotCount;
    m_SamplerDescriptorPool.BeginUpdate();
    m_SamplerDescriptorPool.SetSampler(Slot, &sampler);
    m_SamplerDescriptorPool.EndUpdate();
    m_SamplerDescriptorPool.GetDescriptorSlot(pOutValue, Slot);

    ++m_SamplerSlotCount;
    return Slot;
}

/**
* @brief 未使用関数です。
*
* @param[in ]    pOutValue    ディスクリプタスロット
* @param[in ]    textureView  テクスチャービュー
* @details 未使用関数です。
*/
void GraphicsSystem::UnregisterTextureViewSlot(
    nn::gfx::DescriptorSlot* pOutValue,
    const nn::gfx::TextureView& textureView
) const NN_NOEXCEPT
{
    NN_UNUSED(pOutValue);
    NN_UNUSED(textureView);
}

/**
* @brief 未使用関数です。
*
* @param[in ]    pOutValue    サンプラースロット
* @param[in ]    sampler      サンプラー
* @details 未使用関数です。
*/
void GraphicsSystem::UnregisterSamplerSlot(
    nn::gfx::DescriptorSlot* pOutValue,
    const nn::gfx::Sampler& sampler) const NN_NOEXCEPT
{
    NN_UNUSED(pOutValue);
    NN_UNUSED(sampler);
}

/**
* @brief コマンドメモリが不足したイベントを受けとるためのコールバックです。
*
* @param[in]    pCommandBuffer    コマンドバッファ
* @param[in]    arg               追加パラメータ
* @details コマンドメモリが不足したイベントを受けとるためのコールバックです。
*/
void GraphicsSystem::OutOfCommandMemoryEventCallback(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::gfx::OutOfMemoryEventArg& arg) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(pCommandBuffer->GetUserPtr());
    NN_UNUSED(arg);

    reinterpret_cast<GraphicsSystem*>(
        pCommandBuffer->GetUserPtr()
        )->AddCommandMemoryToCommandBuffer(pCommandBuffer);
}

/**
* @brief コントロールメモリが不足したイベントを受けとるためのコールバックです。
*
* @param[in]    pCommandBuffer    コマンドバッファ
* @param[in]    arg               追加パラメータ
* @details コントロールメモリが不足したイベントを受けとるためのコールバックです。
*/
void GraphicsSystem::OutOfControlMemoryEventCallback(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::gfx::OutOfMemoryEventArg& arg) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(pCommandBuffer->GetUserPtr());
    NN_UNUSED(arg);

    reinterpret_cast<GraphicsSystem*>(
        pCommandBuffer->GetUserPtr()
        )->AddControlMemoryToCommandBuffer(pCommandBuffer);
}

/**
* @brief ディスプレイの初期化を行います。
*
* @details ディスプレイの初期化を行います。
*/
void GraphicsSystem::InitializeDisplay() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&m_pDisplay));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::vi::CreateLayer(&m_pLayer,
        m_pDisplay));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::vi::SetLayerScalingMode(m_pLayer,
        nn::vi::ScalingMode_FitToLayer));
}

/**
* @brief デバイスの初期化を行います。
*
* @details デバイスの初期化を行います。
*/
void GraphicsSystem::InitializeDevice() NN_NOEXCEPT
{
    nn::gfx::Device::InfoType info;
    info.SetDefault();

    m_Device.Initialize(info);
}

/**
* @brief メモリプールの初期化を行います。
*
* @details メモリプールの初期化を行います。
*/
void GraphicsSystem::InitializeMemoryPool() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    nn::gfx::MemoryPoolInfo info;
    info.SetDefault();
    info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached |
        nn::gfx::MemoryPoolProperty_GpuCached);

    m_VisiblePoolMemorySize = nn::util::align_up(MemoryPoolSize,
        nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&m_Device, info));

    const size_t Aligment =
        nn::gfx::MemoryPool::GetPoolMemoryAlignment(&m_Device, info);

    m_pMemoryPool = m_pApplicationHeap->Allocate(MemoryPoolSize, Aligment);

    info.SetPoolMemory(m_pMemoryPool, MemoryPoolSize);

    m_MemoryPool.Initialize(&m_Device, info);

    m_MemoryPoolOffset = 0;

    m_MemoryPoolCommandOffset = 0;
}

/**
* @brief メモリプールの初期化を行います。
*
* @details メモリプールの初期化を行います。
*/
void GraphicsSystem::InitializeMemoryPoolInvisible() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    nn::gfx::MemoryPoolInfo info;
    info.SetDefault();
    info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuInvisible |
        nn::gfx::MemoryPoolProperty_GpuCached |
        nn::gfx::MemoryPoolProperty_Compressible);

    m_InvisiblePoolMemorySize = nn::util::align_up(MemoryPoolSizeInvisible,
        nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&m_Device, info));

    const size_t Aligment =
        nn::gfx::MemoryPool::GetPoolMemoryAlignment(&m_Device, info);

    m_pMemoryPoolInvisible =
        m_pApplicationHeap->Allocate(MemoryPoolSizeInvisible, Aligment);

    info.SetPoolMemory(m_pMemoryPoolInvisible, MemoryPoolSizeInvisible);

    m_MemoryPoolInvisible.Initialize(&m_Device, info);

    m_MemoryPoolInvisibleOffset = 0;
}

/**
* @brief スワップチェーンの初期化を行います。
*
* @details スワップチェーンの初期化を行います。
*/
void GraphicsSystem::InitializeSwapChain() NN_NOEXCEPT
{
    nn::gfx::SwapChain::InfoType info;

    info.SetDefault();
    info.SetLayer(m_pLayer);
    info.SetWidth(m_DisplayWidth);
    info.SetHeight(m_DisplayHeight);
    info.SetFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb);
    info.SetBufferCount(2);

    if (NN_STATIC_CONDITION(nn::gfx::SwapChain::IsMemoryPoolRequired))
    {
        m_MemoryPoolInvisibleOffset = nn::util::align_up(
            m_MemoryPoolInvisibleOffset,
            nn::gfx::SwapChain::GetScanBufferAlignment(&m_Device, info));

        size_t size = m_SwapChain.CalculateScanBufferSize(&m_Device, info);

        m_SwapChain.Initialize(&m_Device,
            info,
            &m_MemoryPoolInvisible,
            m_MemoryPoolInvisibleOffset,
            size);

        m_MemoryPoolInvisibleOffset += size;
    }
    else
    {
        m_SwapChain.Initialize(&m_Device, info, NULL, 0, 0);
    }
}

/**
* @brief キューの初期化を行います。
*
* @details キューの初期化を行います。
*/
void GraphicsSystem::InitializeQueue() NN_NOEXCEPT
{
    nn::gfx::Queue::InfoType info;
    info.SetDefault();
    info.SetCapability(nn::gfx::QueueCapability_Graphics);

    m_Queue.Initialize(&m_Device, info);
}

/**
* @brief フェンスの初期化を行います。
*
* @details フェンスの初期化を行います。
*/
void GraphicsSystem::InitializeFence() NN_NOEXCEPT
{
    nn::gfx::Fence::InfoType info;
    info.SetDefault();

    m_Fence.Initialize(&m_Device, info);
}

/**
* @brief デップスステンシルテクスチャの初期化を行います。
*
* @details デップスステンシルテクスチャの初期化を行います。
*/
void GraphicsSystem::InitializeDepthStencilTexture() NN_NOEXCEPT
{
    nn::gfx::Texture::InfoType info;
    info.SetDefault();
    info.SetWidth(m_DisplayWidth);
    info.SetHeight(m_DisplayHeight);
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_DepthStencil);
    info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    info.SetImageFormat(nn::gfx::ImageFormat_D16_Unorm);
    info.SetMipCount(1);
    info.SetDepth(1);

    m_MemoryPoolInvisibleOffset = nn::util::align_up(
        m_MemoryPoolInvisibleOffset,
        nn::gfx::Texture::CalculateMipDataAlignment(&m_Device, info));

    size_t size = nn::gfx::Texture::CalculateMipDataSize(&m_Device, info);

    m_DepthStencilTexture.Initialize(&m_Device,
        info,
        &m_MemoryPoolInvisible,
        m_MemoryPoolInvisibleOffset,
        size);

    m_MemoryPoolInvisibleOffset += size;
}

/**
* @brief デップスステンシルビューの初期化を行います。
*
* @details デップスステンシルビューの初期化を行います。
*/
void GraphicsSystem::InitializeDepthStencilView() NN_NOEXCEPT
{
    nn::gfx::DepthStencilView::InfoType info;
    info.SetDefault();
    info.SetImageDimension(nn::gfx::ImageDimension_2d);
    info.SetTexturePtr(&m_DepthStencilTexture);

    m_DepthStencilView.Initialize(&m_Device, info);
}

/**
* @brief バッファディスクリプタプールの初期化を行います。
*
* @details バッファディスクリプタプールの初期化を行います。
*/
void GraphicsSystem::InitializeBufferDescriptorPool() NN_NOEXCEPT
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_BufferView);
    info.SetSlotCount(DescriptorPoolSlotCount + 20);
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&m_Device, info);
    m_MemoryPoolOffset = nn::util::align_up(m_MemoryPoolOffset,
        nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&m_Device, info));
    m_BufferDescriptorPool.Initialize(
        &m_Device, info, &m_MemoryPool, m_MemoryPoolOffset, size
    );
    m_MemoryPoolOffset += size;
}

/**
* @brief テクスチャディスクリプタプールの初期化を行います。
*
* @details テクスチャディスクリプタプールの初期化を行います。
*/
void GraphicsSystem::InitializeTextureDescriptorPool() NN_NOEXCEPT
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_TextureView);
    info.SetSlotCount(DescriptorPoolSlotCount + 20);

    m_MemoryPoolOffset = nn::util::align_up(
        m_MemoryPoolOffset,
        nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(
        &m_Device, info));

    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(
        &m_Device, info);

    m_TextureDescriptorPool.Initialize(&m_Device,
        info,
        &m_MemoryPool,
        m_MemoryPoolOffset,
        size);

    m_MemoryPoolOffset += size;
}

/**
* @brief サンプラーディスクリプタプールの初期化を行います。
*
* @details サンプラーディスクリプタプールの初期化を行います。
*/
void GraphicsSystem::InitializeSamplerDescriptorPool() NN_NOEXCEPT
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_Sampler);
    info.SetSlotCount(DescriptorPoolSlotCount + 20);

    m_MemoryPoolOffset = nn::util::align_up(
        m_MemoryPoolOffset,
        nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(
        &m_Device, info));

    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(
        &m_Device, info);

    m_SamplerDescriptorPool.Initialize(&m_Device,
        info,
        &m_MemoryPool,
        m_MemoryPoolOffset,
        size);

    m_MemoryPoolOffset += size;
}

/**
* @brief コマンドバッファの初期化を行います。
*
* @details コマンドバッファの初期化を行います。
*/
void GraphicsSystem::InitializeCommandBuffer() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    m_CommandBuffer.Initialize(&m_Device, info);
    m_CommandBuffer.SetUserPtr(this);
    m_CommandBuffer.SetOutOfCommandMemoryEventCallback(
        OutOfCommandMemoryEventCallback);
    m_CommandBuffer.SetOutOfControlMemoryEventCallback(
        OutOfControlMemoryEventCallback);

    m_MemoryPoolOffset = nn::util::align_up(
        m_MemoryPoolOffset,
        nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&m_Device));

    m_MemoryPoolCommandOffset = m_MemoryPoolOffset;
    m_MemoryPoolCommandOffset += CommandBufferCommandMemoryChunkSize;
}

/**
* @brief ビューポイントシザーステートの初期化を行います。
*
* @details ビューポイントシザーステートの初期化を行います。
*/
void GraphicsSystem::InitializeViewportScissorState() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled(true);

    nn::gfx::ViewportStateInfo viewportStateInfo;
    viewportStateInfo.SetDefault();
    viewportStateInfo.SetWidth(static_cast<float>(m_DisplayWidth));
    viewportStateInfo.SetHeight(static_cast<float>(m_DisplayHeight));
    info.SetViewportStateInfoArray(&viewportStateInfo, 1);

    nn::gfx::ScissorStateInfo scissorStateInfo;
    scissorStateInfo.SetDefault();
    scissorStateInfo.SetWidth(m_DisplayWidth);
    scissorStateInfo.SetHeight(m_DisplayHeight);
    info.SetScissorStateInfoArray(&scissorStateInfo, 1);

    size_t memorySize =
        nn::gfx::ViewportScissorState::GetRequiredMemorySize(info);
    m_ViewportScissorState.SetMemory(
        m_pApplicationHeap->Allocate(memorySize), memorySize);

    m_ViewportScissorState.Initialize(&m_Device, info);
}

/**
* @brief ブレンドステートの初期化を行います。
*
* @details ブレンドステートの初期化を行います。
*/
void GraphicsSystem::InitializeBlendState() NN_NOEXCEPT
{
    nn::gfx::BlendState::InfoType info;
    info.SetDefault();

    nn::gfx::BlendTargetStateInfo targetInfo;
    targetInfo.SetDefault();
    targetInfo.SetChannelMask(nn::gfx::ChannelMask_Red |
        nn::gfx::ChannelMask_Green |
        nn::gfx::ChannelMask_Blue);
    targetInfo.SetBlendEnabled(true);
    targetInfo.SetColorBlendFunction(nn::gfx::BlendFunction_Add);
    targetInfo.SetDestinationColorBlendFactor(
        nn::gfx::BlendFactor_OneMinusSourceAlpha);
    targetInfo.SetSourceColorBlendFactor(nn::gfx::BlendFactor_SourceAlpha);
    info.SetBlendTargetStateInfoArray(&targetInfo, 1);

    size_t memorySize = nn::gfx::BlendState::GetRequiredMemorySize(info);
    m_BlendState.SetMemory(m_pApplicationHeap->Allocate(memorySize),
        memorySize);

    m_BlendState.Initialize(&m_Device, info);
}

/**
* @brief デップスステンシルステートの初期化を行います。
*
* @details デップスステンシルステートの初期化を行います。
*/
void GraphicsSystem::InitializeDepthStencilState() NN_NOEXCEPT
{
    nn::gfx::DepthStencilState::InfoType info;
    info.SetDefault();
    info.SetDepthTestEnabled(false);
    info.SetDepthWriteEnabled(true);

    m_DepthStencilState.Initialize(&m_Device, info);
}

/**
* @brief ラスターライザーステートの初期化を行います。
*
* @details ラスターライザーステートの初期化を行います。
*/
void GraphicsSystem::InitializeRasterizerState() NN_NOEXCEPT
{
    nn::gfx::RasterizerState::InfoType info;
    info.SetDefault();
    info.SetCullMode(nn::gfx::CullMode_None);
    info.SetPrimitiveTopologyType(nn::gfx::PrimitiveTopologyType_Triangle);
    info.SetScissorEnabled(true);
    info.SetDepthClipEnabled(false);
    m_RasterizerState.Initialize(&m_Device, info);
}

/**
* @brief デバッグフォントライターの初期化を行います。
*
* @details デバッグフォントライターの初期化を行います。
*/
void GraphicsSystem::InitializeDebugFont() NN_NOEXCEPT
{
    const int MinFontWidth = 4;
    const int FontHeight = 20;
    const int CharCountMax = (m_DisplayWidth / MinFontWidth) * (m_DisplayHeight / FontHeight);
    nn::gfx::util::DebugFontTextWriterInfo info;
    info.SetDefault();
    info.SetCharCountMax(CharCountMax);
    info.SetUserMemoryPoolEnabled(false);

    size_t debugFontHeapSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(
        &m_Device,
        info
        );
    m_DebugFontHeap = nn::util::BytePtr(new uint8_t[debugFontHeapSize]);

    m_DebugFont.Initialize(
        &m_Device,
        info,
        m_DebugFontHeap.Get(),
        debugFontHeapSize,
        NULL,
        0,
        0
        );
    m_DebugFont.SetDisplayWidth(m_DisplayWidth);
    m_DebugFont.SetDisplayHeight(m_DisplayHeight);

    m_DebugFont.SetTextureDescriptor(&m_TextureDescriptorPool, m_TextureSlotCount);
    ++m_TextureSlotCount;
    m_DebugFont.SetSamplerDescriptor(&m_SamplerDescriptorPool, m_SamplerSlotCount);
    ++m_SamplerSlotCount;
}

/**
* @brief メモリプールの終了処理を行います。
*
* @details メモリプールの終了処理を行います。
*/
void GraphicsSystem::FinalizeMemoryPool() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    m_MemoryPool.Finalize(&m_Device);

    m_pApplicationHeap->Free(m_pMemoryPool);
}

/**
* @brief メモリプールの終了処理を行います。
*
* @details メモリプールの終了処理を行います。
*/
void GraphicsSystem::FinalizeMemoryPoolInvisible() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    m_MemoryPoolInvisible.Finalize(&m_Device);

    m_pApplicationHeap->Free(m_pMemoryPoolInvisible);
}

/**
* @brief ビューポイントシザーステートの終了処理を行います。
*
* @details ビューポイントシザーステートの終了処理を行います。
*/
void GraphicsSystem::FinalizeViewportScissorState() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    void* ptr = m_ViewportScissorState.GetMemory();

    m_ViewportScissorState.Finalize(&m_Device);

    m_pApplicationHeap->Free(ptr);
}

/**
* @brief ブレンドステートの終了処理を行います。
*
* @details ブレンドステートの終了処理を行います。
*/
void GraphicsSystem::FinalizeBlendState() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    void* ptr = m_BlendState.GetMemory();

    m_BlendState.Finalize(&m_Device);

    m_pApplicationHeap->Free(ptr);
}

/**
* @brief デバッグフォントライターの終了処理を行います。
*
* @details デバッグフォントライターの終了処理を行います。
*/
void GraphicsSystem::FinalizeDebugFont() NN_NOEXCEPT
{
    m_DebugFont.Finalize();
    delete[] reinterpret_cast<uint8_t*>(m_DebugFontHeap.Get());
    m_DebugFontHeap.Reset(NULL);
}


/**
* @brief コマンドバッファのリセットを行います。
*
* @details コマンドバッファのリセットを行います。
*/
void GraphicsSystem::ResetCommandBuffer() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    m_CommandBuffer.Reset();

    m_MemoryPoolOffset = m_MemoryPoolCommandOffset;

    const size_t Size = m_CommandBufferControlMemoryChunks.size();

    for (size_t i = 0; i < Size; ++i)
    {
        m_pApplicationHeap->Free(
            m_CommandBufferControlMemoryChunks[i]);
    }

    m_CommandBufferControlMemoryChunks.clear();
}

/**
* @brief コマンドバッファにコマンドメモリを追加します。
*
* @param[in]    pCommandBuffer    コマンドバッファ
* @details コマンドバッファにコマンドメモリを追加します。
*/
void GraphicsSystem::AddCommandMemoryToCommandBuffer(nn::gfx::CommandBuffer* pCommandBuffer
) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);

    m_MemoryPoolOffset = nn::util::align_up(
        m_MemoryPoolOffset,
        nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&m_Device));

    pCommandBuffer->AddCommandMemory(&m_MemoryPool,
        m_MemoryPoolOffset,
        CommandBufferCommandMemoryChunkSize);

    m_MemoryPoolOffset += CommandBufferCommandMemoryChunkSize;
}

/**
* @brief コマンドバッファにコントロールメモリを追加します。
*
* @param[in]    pCommandBuffer    コマンドバッファ
* @details コマンドバッファにコントロールメモリを追加します。
*/
void GraphicsSystem::AddControlMemoryToCommandBuffer(nn::gfx::CommandBuffer* pCommandBuffer
) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    void* pControlMemoryChunk = m_pApplicationHeap->Allocate(
        CommandBufferControlMemoryChunkSize,
        nn::gfx::CommandBuffer::GetControlMemoryAlignment(&m_Device));

    pCommandBuffer->AddControlMemory(pControlMemoryChunk,
        CommandBufferControlMemoryChunkSize);

    m_CommandBufferControlMemoryChunks.push_back(pControlMemoryChunk);
}

/**
* @brief カラーバッファを初期化します。
*
* @param[in]    pColorTextureView    カラーテクスチャビュー
* @param[in]    pColorTargetView     カラーターゲットビュー
* @param[in]    pColorBuffer         テスクチャー
* @details カラーバッファを初期化します。
*/
void GraphicsSystem::InitializeColorBuffer(
    nn::gfx::TextureView* pColorTextureView,
    nn::gfx::ColorTargetView* pColorTargetView,
    nn::gfx::Texture* pColorBuffer
) NN_NOEXCEPT
{
    nn::gfx::Texture::InfoType info;
    info.SetDefault();
    info.SetWidth(m_DisplayWidth);
    info.SetHeight(m_DisplayHeight);
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);
    info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    info.SetImageFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
    info.SetMipCount(1);

    m_MemoryPoolInvisibleOffset = nn::util::align_up(
        m_MemoryPoolInvisibleOffset,
        nn::gfx::Texture::CalculateMipDataAlignment(&m_Device, info)
    );
    size_t size = nn::gfx::Texture::CalculateMipDataSize(&m_Device, info);
    pColorBuffer->Initialize(&m_Device, info, &m_MemoryPoolInvisible,
        m_MemoryPoolInvisibleOffset, size);
    m_MemoryPoolInvisibleOffset += size;

    // カラーターゲットビューを初期化します
    nn::gfx::ColorTargetView::InfoType targetInfo;
    targetInfo.SetDefault();
    targetInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
    targetInfo.SetImageFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
    targetInfo.SetTexturePtr(pColorBuffer);
    pColorTargetView->Initialize(&m_Device, targetInfo);

    // カラーバッファのテクスチャビューを初期化します
    nn::gfx::TextureView::InfoType viewInfo;
    viewInfo.SetDefault();
    viewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
    viewInfo.SetImageFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
    viewInfo.SetTexturePtr(pColorBuffer);
    pColorTextureView->Initialize(&m_Device, viewInfo);
}

/**
* @brief カラーバッファの終了処理を行います。
*
* @param[in]    pColorTextureView    カラーテクスチャビュー
* @param[in]    pColorTargetView     カラーターゲットビュー
* @param[in]    pColorBuffer         テスクチャー
* @details カラーバッファの終了処理を行います。
*/
void GraphicsSystem::FinalizeColorBuffer(
    nn::gfx::TextureView* pColorTextureView,
    nn::gfx::ColorTargetView* pColorTargetView,
    nn::gfx::Texture* pColorBuffer
) NN_NOEXCEPT
{
    pColorTextureView->Finalize(&m_Device);
    pColorTargetView->Finalize(&m_Device);
    pColorBuffer->Finalize(&m_Device);
}

/**
* @brief サンプラを初期化します。
*
* @param[in]    pSampler    サンプラー
* @details サンプラを初期化します。
*/
void GraphicsSystem::InitializeSampler(nn::gfx::Sampler* pSampler) NN_NOEXCEPT
{
    nn::gfx::Sampler::InfoType info;
    info.SetDefault();
    info.SetFilterMode(nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint);
    info.SetAddressU(nn::gfx::TextureAddressMode_Mirror);
    info.SetAddressV(nn::gfx::TextureAddressMode_Mirror);
    info.SetAddressW(nn::gfx::TextureAddressMode_Mirror);
    pSampler->Initialize(&m_Device, info);
}

/**
* @brief サンプラの終了処理をします。
*
* @param[in]    pSampler    サンプラー
* @details サンプラの終了処理をします。
*/
void GraphicsSystem::FinalizeSampler(nn::gfx::Sampler* pSampler) NN_NOEXCEPT
{
    pSampler->Finalize(&m_Device);
}

/**
* @brief スワップチェーンを取得します。
*
* @return  スワップチェーンを返します。
* @details スワップチェーンを取得します。
*/
nn::gfx::SwapChain* GraphicsSystem::GetSwapChain() NN_NOEXCEPT
{
    return &m_SwapChain;
}

/**
* @brief メモリプールを取得します。
*
* @return  メモリプールを返します。
* @details メモリプールを取得します。
*/
nn::gfx::MemoryPool* GraphicsSystem::GetMemoryPoolInvisible() NN_NOEXCEPT
{
    return &m_MemoryPoolInvisible;
}

/**
* @brief メモリプールを取得します。
*
* @return  メモリプールを返します。
* @details メモリプールを取得します。
*/
nn::gfx::MemoryPool* GraphicsSystem::GetMemoryPool() NN_NOEXCEPT
{
    return &m_MemoryPool;
}

/**
* @brief ブラーバッファを初期化します。
*
* @details ブラーバッファを初期化します。
*/
void GraphicsSystem::InitializeBlurBuffer() NN_NOEXCEPT
{
    nn::gfx::Texture::InfoType info;
    info.SetDefault();
    info.SetWidth(m_DisplayWidth);
    info.SetHeight(m_DisplayHeight);
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);
    info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    info.SetImageFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
    info.SetMipCount(1);

    m_MemoryPoolInvisibleOffset = nn::util::align_up(m_MemoryPoolInvisibleOffset,
        nn::gfx::Texture::CalculateMipDataAlignment(&m_Device, info));
    size_t size = nn::gfx::Texture::CalculateMipDataSize(&m_Device, info);
    m_BlurColorBuffer.Initialize(
        &m_Device,
        info,
        GetMemoryPoolInvisible(),
        m_MemoryPoolInvisibleOffset, size);
    m_MemoryPoolInvisibleOffset += size;

    // カラーターゲットビューを初期化します
    nn::gfx::ColorTargetView::InfoType targetInfo;
    targetInfo.SetDefault();
    targetInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
    targetInfo.SetImageFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
    targetInfo.SetTexturePtr(&m_BlurColorBuffer);
    m_BlurColorTargetView.Initialize(&m_Device, targetInfo);

    // カラーバッファのテクスチャビューを初期化します
    nn::gfx::TextureView::InfoType viewInfo;
    viewInfo.SetDefault();
    viewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
    viewInfo.SetImageFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
    viewInfo.SetTexturePtr(&m_BlurColorBuffer);
    m_BlurColorTextureView.Initialize(&m_Device, viewInfo);
}


} // namespace gfxLog
} // namespace nns
