﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
#include <nn/gfx.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

namespace nns
{
namespace gfxLog
{

//! @brief GPUバッファを管理します
class GpuBuffer
{
    NN_DISALLOW_COPY(GpuBuffer);
    NN_DISALLOW_MOVE(GpuBuffer);

public:
    //! @brief Initialize() の引数オブジェクトです。
    class InitializeArg
    {
        NN_DISALLOW_COPY(InitializeArg);
        NN_DISALLOW_MOVE(InitializeArg);
    public:
        //! @brief コンストラクタです。
        InitializeArg() NN_NOEXCEPT
            : m_GpuAccessFlag(nn::gfx::GpuAccess_ConstantBuffer)
            , m_BufferSize(0)
            , m_BufferCount(1)
        {}

        //! @brief このバッファがどのような用途のバッファとして使用されるかを指定します。
        //!
        //! @details
        //! この情報に基づいて nn::gfx::Buffer を作成します。
        //!
        //! @param[in] gpuAccess  バッファの種別を nn::gfx::GpuAccess で指定します。
        //!
        //! @return オブジェクト自身への参照を返します。
        //!
        InitializeArg& SetGpuAccessFlag(int gpuAccess) NN_NOEXCEPT
        {
            m_GpuAccessFlag = gpuAccess;
            return *this;
        }

        //! @brief  使用するバッファのサイズを指定します。
        //!
        //! @details
        //! このメソッドに指定されたサイズのメモリが SetBufferCount で指定された数だけメモリプール上の領域を使用します。
        //!
        //! @param[in]  bufferSize  バッファのサイズをバイト単位で指定します。
        //!
        //! @return オブジェクト自身への参照を返します。
        //!
        //! @sa SetBufferCount
        //!
        InitializeArg& SetBufferSize(size_t bufferSize) NN_NOEXCEPT
        {
            m_BufferSize = bufferSize;
            return *this;
        }

        //! @brief  バッファのマルチバッファリングの数をしてします。
        //!
        //! @details
        //! ここで設定した数だけ内部で nn::gfx::Buffer が作成され、切り替えて使用することが
        //! 出来ます。
        //! たとえばダブルバッファリングしたい場合は 2 を指定、トリプルバッファリングしたい
        //! 場合は 3 を指定します。
        //!
        //! @param[in] bufferCount  マルチバッファリングする数を指定します。
        //!
        //! @return オブジェクト自身への参照を返します。
        //!
        InitializeArg& SetBufferCount(int bufferCount) NN_NOEXCEPT
        {
            m_BufferCount = bufferCount;
            return *this;
        }
    private:
        int                   m_GpuAccessFlag;
        size_t                m_BufferSize;
        uint32_t              m_BufferCount;

        friend class GpuBuffer;
    };
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    GpuBuffer() NN_NOEXCEPT
    : m_Buffer()
    , m_Size(0)
    , m_Alignment(0)
    , m_Count(0)
    , m_AllBufferSize(0)
    , m_MappedBufferIndex(-1)
    , m_pMappedPointer(NULL)
    , m_IsInitialized(false)
    , m_Raw(0)
    {
    }

    //@}

    //! @brief  初期化処理です。
    //!
    //! @details
    //! メモリプールからメモリを確保しバッファを作成します。
    //!
    //! @param[in]  pDevice デバイスです。
    //! @param[in]  arg     初期化パラメータです。
    //! @param[in]  pMemoryPool 作成するバッファのメモリを確保するメモリプールです。
    //! @param[in]  memoryPoolOffset    メモリを確保する領域のメモリプール先頭からのオフセットです。
    //!
    //! @return バッファの作成成功可否です。
    //!
    bool Initialize(
        nn::gfx::Device* pDevice,
        const InitializeArg&  arg,
        nn::gfx::MemoryPool* pMemoryPool,
        size_t memoryPoolOffset
        ) NN_NOEXCEPT;

    //! @brief  終了処理です。
    //!
    //! @param[in]  pDevice デバイスです。
    //!
    void Finalize(
        nn::gfx::Device* pDevice) NN_NOEXCEPT;

    //! @brief バッファを CPU で更新するための事前処理を行います。
    //!
    //! @details
    //! バッファの値を CPU で書き換える前に呼び出してください。
    //!
    //! @param[in]  bufferIndex 事前準備を行うバッファのインデックスを指定します。
    //!
    void Map(int bufferIndex) NN_NOEXCEPT;

    //! @brief バッファを CPU で書き換えた後の終了処理を行います。
    //!
    //! @details
    //! バッファの値を CPU で書き換えた後に呼び出してください。
    //!
    void Unmap() NN_NOEXCEPT;

    //! @brief バッファに指定されたサイズ分の領域を確保します。
    //!
    //! @param[in]  size 確保する領域のサイズ
    //!
    //! @return 確保された領域の先頭ポインタ。失敗した場合は、NULLを返す。
    //!
    //! @sa GetGpuAddress
    //!
    void* Allocate(size_t size) NN_NOEXCEPT
    {
        size_t offset = m_Raw + m_MappedBufferIndex * m_Size;
        size_t sizeRaw = nn::util::align_up(m_Raw + size, m_Alignment );

        // バッファサイズを超えていたら、NULLを返す。
        if ( sizeRaw > m_Size )
        {
            NN_LOG("[GpuBuffer] Warning!! m_Raw > m_Size\n");
            return NULL;
        }

        m_Raw = sizeRaw;
        return nn::util::BytePtr( m_pMappedPointer ).Advance( offset ).Get();
    }

    //----------------------------------------
    //! @name 設定/取得
    //@{

    //! @brief バッファのサイズを取得します。
    //!
    //! @return バッファのサイズです。
    //!
    const size_t GetBufferSize() const NN_NOEXCEPT
    {
        return m_Size;
    }

    //! @brief バッファのアライメントを取得します。
    //!
    //! @return バッファのアライメント値です。
    //!
    size_t GetBufferAlignment() const NN_NOEXCEPT
    {
        return m_Alignment;
    }

    //! @brief 確保済みのメモリのサイズを取得します。
    //!
    //! @return 確保済みの領域のサイズです。
    //!
    size_t GetAllocatedSize() const NN_NOEXCEPT
    {
        return m_Raw;
    }

    // バッファの GpuAddr を取得します。

    //! @brief バッファのポインタからGetGpuAddressを取得します。
    //!
    //! @param[out] pGpuAddress  取得したGpuAdress
    //! @param[in]  address      GpuAddressを取得するバッファのポインタ
    //!
    void GetGpuAddress(nn::gfx::GpuAddress* pGpuAddress, void* address) const NN_NOEXCEPT
    {
        NN_ASSERT(m_IsInitialized);
        NN_ASSERT(nn::util::BytePtr( m_pMappedPointer ).Distance( address ) >= 0);
        NN_ASSERT(static_cast<size_t>(nn::util::BytePtr(
            m_pMappedPointer ).Distance( address )) <= m_AllBufferSize
        );

        m_Buffer.GetGpuAddress(pGpuAddress);
        pGpuAddress->Offset( nn::util::BytePtr( m_pMappedPointer ).Distance( address ) );
    }

    //! @brief  初期化パラメータから、バッファのアライメントを計算します。
    //!
    //! @param[in]  pDevice デバイスです。
    //! @param[in]  arg     初期化パラメータです。
    //!
    //! @return バッファのアライメント
    //!
    static size_t GetGpuBufferAlignement(
        nn::gfx::Device* pDevice,
        const InitializeArg& arg) NN_NOEXCEPT;

    //! @brief  初期化パラメータから、必要なメモリプールの容量を計算します。
    //!
    //! @param[in]  pDevice デバイスです。
    //! @param[in]  arg     初期化パラメータです。
    //!
    //! @return 必要なメモリプールの容量
    //!
    static size_t CalculateGpuBufferPoolSize(
        nn::gfx::Device* pDevice,
        const InitializeArg& arg) NN_NOEXCEPT;

    //@}
private:
    // Gpu がアクセスするバッファ
    nn::gfx::Buffer     m_Buffer;
    // バッファのサイズ
    size_t              m_Size;
    // バッファのアライメント
    size_t              m_Alignment;
    // バッファのバッファリングカウント
    uint32_t            m_Count;
    // 全バッファのサイズ(nn::gfx::Bufferであるm_Bufferの実際のサイズ)
    size_t              m_AllBufferSize;
    //  マップしたバッファのインデックス
    int32_t m_MappedBufferIndex;
    //  マップした際のポインタ
    void*   m_pMappedPointer;
    //  初期化したかどうか
    bool    m_IsInitialized;

    //  マップ後に確保された領域のサイズ
    size_t              m_Raw;      // 同期処理不要の時に使用される変数
};

} // namespace gfxLog
} // namespace nns
