﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>

#include "gfx0_DriverSample.h"

namespace nns { namespace gfx0 {

//-----------------------------------------------------------------------------

Gfx0DriverSample::Gfx0DriverSample() NN_NOEXCEPT
    : m_TransferMemoryAvailable(false)
{
    // SharedMemory
    {
        auto result = nn::os::CreateSharedMemory(
                                &m_SharedMemory,
                                SharedMemorySize,
                                nn::os::MemoryPermission_ReadWrite,
                                nn::os::MemoryPermission_ReadWrite);
        NN_SDK_ASSERT(result.IsSuccess());
        NN_UNUSED(result);

        auto address = nn::os::MapSharedMemory(
                                &m_SharedMemory,
                                nn::os::MemoryPermission_ReadWrite);
        NN_SDK_ASSERT(address != NULL);
        NN_UNUSED(address);
    }

    // SystemEvent
    {
        auto result = nn::os::CreateSystemEvent(
                                &m_SystemEvent,
                                nn::os::EventClearMode_AutoClear,
                                true);
        NN_SDK_ASSERT(result.IsSuccess());
        NN_UNUSED(result);
    }
}

Gfx0DriverSample::~Gfx0DriverSample() NN_NOEXCEPT
{
    // Transfer Memory
    if (m_TransferMemoryAvailable)
    {
        nn::os::UnmapTransferMemory( &m_TransferMemory );
        nn::os::DestroyTransferMemory( &m_TransferMemory );
        m_TransferMemoryAvailable = false;
    }

    // Shared Memory
    nn::os::UnmapSharedMemory( &m_SharedMemory );
    nn::os::DestroySharedMemory( &m_SharedMemory );

    // System Event
    nn::os::DestroySystemEvent( &m_SystemEvent );
}

//-----------------------------------------------------------------------------

void Gfx0DriverSample::AttachTransferMemory(nn::os::NativeHandle handle, bool isManaged, size_t size) NN_NOEXCEPT
{
    if (m_TransferMemoryAvailable)
    {
        nn::os::UnmapTransferMemory( &m_TransferMemory );
        nn::os::DestroyTransferMemory( &m_TransferMemory );
        m_TransferMemoryAvailable = false;
    }

    nn::os::AttachTransferMemory( &m_TransferMemory, size, handle, isManaged );
    auto result = nn::os::MapTransferMemory(
                                    &m_TransferMemoryAddress,
                                    &m_TransferMemory,
                                    nn::os::MemoryPermission_ReadWrite);

    NN_SDK_ASSERT(result.IsSuccess());
    NN_UNUSED(result);

    m_TransferMemoryAvailable = true;
}

//-----------------------------------------------------------------------------

}}  // namespace nns::gfx0

