﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nns/gfx/gfx_PrimitiveRendererMeterDrawer.h>

namespace nns
{
namespace gfx
{
namespace PrimitiveRenderer
{

float MeterDrawer::GetHeight(nn::perf::LoadMeterBase* pRootMeter) const NN_NOEXCEPT
{
    // pRootMeter が NULL なら 0 を返す
    if( pRootMeter )
    {
        // m_VisibleNoResult が true なら計測結果を持たない子メーターをカウントしない
        int resultMeterCount = 0;
        if(m_VisibleNoResult)
        {
            resultMeterCount = pRootMeter->m_LoadMeterList.size();
        }
        else
        {
            for ( LoadMeterList::iterator it = pRootMeter->m_LoadMeterList.begin(); it != pRootMeter->m_LoadMeterList.end(); ++it )
            {
                if(it->GetLastSectionCount() > 0)
                {
                    resultMeterCount++;
                }
            }
        }
        if ( resultMeterCount > 0 )
        {
            return m_BarHeight * resultMeterCount;
        }
        else
        {
            return m_BarHeight;
        }
    }
    else
    {
        return 0.f;
    }
}

void MeterDrawer::Draw( nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::PrimitiveRenderer::Renderer* pRenderer, nn::perf::LoadMeterBase* pRootMeter) NN_NOEXCEPT
{
    // pRootMeter が NULL なら描画しない
    if( pRootMeter )
    {
        // フレーム開始時刻を記憶
        nn::os::Tick frameBeginTick = pRootMeter->GetLastResult(0).begin;

        // 区切り線の数を更新
        UpdateScale(pRootMeter);

        // メーター全体の高さを取得
        float height = GetHeight( pRootMeter );

        // renderer のパラメタを初期化
        pRenderer->SetDefaultParameters();

        // メーターの名前と処理率を TextWriter に描画設定
        if (m_pDebugFontTextWriter)
        {
            float scaleX = m_pDebugFontTextWriter->GetScaleX();
            float scaleY = m_pDebugFontTextWriter->GetScaleY();

            // テキストを固定幅に設定
            float fixedWidth = 12.f * scaleX;
            m_pDebugFontTextWriter->SetFixedWidthEnabled(true);
            m_pDebugFontTextWriter->SetFixedWidth(fixedWidth);

            // メーター名と負荷割合を示すテキストが描画されるオフセットを保持
            float textNameOffset = fixedWidth * 7.f;
            float textValueOffset = fixedWidth * 8.f;

            // テキストの描画位置を考慮してメーター描画位置をずらす
            float textOffset = textNameOffset + textValueOffset;
            m_MeterPositionX = m_Position.x + textOffset;
            m_MeterWidth = m_Width - textOffset;

            // テキストの背景を描画
            pRenderer->SetColor(nn::util::Color4u8(m_BackGroundColor.GetR() / 2, m_BackGroundColor.GetG() / 2, m_BackGroundColor.GetB() / 2, m_BackGroundColor.GetA()));
            pRenderer->Draw2DRect(pCommandBuffer, m_Position.x, m_Position.y, textOffset * 0.98f, height);
            pRenderer->SetLineWidth(1.f);
            pRenderer->SetColor( m_BorderColor );
            pRenderer->Draw2DFrame(pCommandBuffer, m_Position.x, m_Position.y, textOffset * 0.98f, height);

            int meterCount = 0;
            for (LoadMeterList::iterator it = pRootMeter->m_LoadMeterList.begin(); it != pRootMeter->m_LoadMeterList.end(); ++it)
            {
                if (m_VisibleNoResult || it->GetLastSectionCount() > 0)
                {
                    float rate = 100.f * it->GetLastTotalSpan().GetMicroSeconds() / m_ReferenceFrame.GetMicroSeconds();
                    const float shadowOffset = 1.8f;
                    const float positionX = m_Position.x;
                    const float positionY = m_Position.y + (m_BarHeight * meterCount) + (m_BarHeight - m_pDebugFontTextWriter->GetFontHeight() * 1.2f) / 2.f;
                    const char* name = it->GetName();

                    if (name != NULL)
                    {
                        float textLength = m_pDebugFontTextWriter->CalculateStringWidth(" %s", name);
                        if (textLength >= textNameOffset)
                        {
                            // メーター名が 7文字以上の時は文字の横サイズを小さくする
                            m_pDebugFontTextWriter->SetScale(scaleX * textNameOffset / textLength, scaleY);
                            m_pDebugFontTextWriter->SetFixedWidth(fixedWidth * textNameOffset / textLength);
                        }
                        else
                        {
                            m_pDebugFontTextWriter->SetScale(scaleX, scaleY);
                            m_pDebugFontTextWriter->SetFixedWidth(fixedWidth);
                        }

                        m_pDebugFontTextWriter->SetTextColor(nn::util::Color4u8::Black());
                        m_pDebugFontTextWriter->SetCursor(positionX + shadowOffset, positionY + shadowOffset);
                        m_pDebugFontTextWriter->Print(" %s", name);
                        m_pDebugFontTextWriter->SetTextColor(m_TextColor);
                        m_pDebugFontTextWriter->SetCursor(positionX, positionY);
                        m_pDebugFontTextWriter->Print(" %s", name);
                    }
                    m_pDebugFontTextWriter->SetFixedWidth(fixedWidth);
                    m_pDebugFontTextWriter->SetScale(scaleX, scaleY);
                    m_pDebugFontTextWriter->SetTextColor(nn::util::Color4u8::Black());
                    m_pDebugFontTextWriter->SetCursor(positionX + textNameOffset + shadowOffset, positionY + shadowOffset);
                    m_pDebugFontTextWriter->Print(":%5.1f%% ", rate);
                    m_pDebugFontTextWriter->SetTextColor(m_TextColor);
                    m_pDebugFontTextWriter->SetCursor(positionX + textNameOffset, positionY);
                    m_pDebugFontTextWriter->Print(":%5.1f%% ", rate);

                    ++meterCount;
                }
            }
            m_pDebugFontTextWriter->SetFixedWidthEnabled(false);
            m_pDebugFontTextWriter->SetScale(scaleX, scaleY);
        }
        else
        {
            m_MeterPositionX = m_Position.x;
            m_MeterWidth = m_Width;
        }

        // 背景を描画（10% 単位を縞々で表現）
        float backGroundOffset = m_MeterWidth / (m_CurrentScale * 10.f);
        for (int i = 0; i < m_CurrentScale * 10; ++i)
        {
            if (i % 2 == 0)
            {
                pRenderer->SetColor(m_BackGroundColor);
            }
            else
            {
                pRenderer->SetColor(nn::util::Color4u8(m_BackGroundColor.GetR() / 2, m_BackGroundColor.GetG() / 2, m_BackGroundColor.GetB() / 2, m_BackGroundColor.GetA()));
            }
            pRenderer->Draw2DRect(pCommandBuffer, m_MeterPositionX + backGroundOffset * i, m_Position.y, backGroundOffset, height);
        }

        // 親負荷メーターの処理時間を描画
        for ( int i = 0; i < pRootMeter->GetLastSectionCount(); ++i )
        {
            const nn::perf::LoadMeterBase::Section& section = pRootMeter->GetLastResult( i );

            float positionLeft = CalculateTimeLinePosition( frameBeginTick, section.begin );
            float positionRight = CalculateTimeLinePosition( frameBeginTick, section.end );

            const float rectX = m_MeterPositionX + positionLeft;
            const float rectY = m_Position.y;
            const float rectWidth  = positionRight - positionLeft;
            const float rectHeight = height;

            pRenderer->SetColor( section.color );
            pRenderer->Draw2DRect( pCommandBuffer, rectX, rectY, rectWidth, rectHeight );
        }

        // 最大の入れ子の深さを取得
        int depthCountMax = 0;
        for (LoadMeterList::iterator it = pRootMeter->m_LoadMeterList.begin(); it != pRootMeter->m_LoadMeterList.end(); ++it)
        {
            for (int i = 0; i < it->GetLastSectionCount(); ++i)
            {
                const nn::perf::LoadMeterBase::Section& section = it->GetLastResult(i);
                depthCountMax = std::max(depthCountMax, section.depth);
                if (depthCountMax > m_DepthCountMax)
                {
                    depthCountMax = m_DepthCountMax;
                    break;
                }
            }
        }

        // メーターとメーターの余白を設定
        float margin = 2.f;
        float depthHeight = (m_BarHeight - margin * 2) / (depthCountMax + 1.f);

        // 登録されている子負荷メーターを描画
        int meterCount = 0;
        for ( LoadMeterList::iterator it = pRootMeter->m_LoadMeterList.begin(); it != pRootMeter->m_LoadMeterList.end(); ++it )
        {
            if (m_VisibleNoResult || it->GetLastSectionCount() > 0)
            {
                for ( int i = 0 ; i < it->GetLastSectionCount(); ++i )
                {
                    const nn::perf::LoadMeterBase::Section& section = it->GetLastResult( i );

                    float positionLeft = CalculateTimeLinePosition( frameBeginTick, section.begin );
                    float positionRight = CalculateTimeLinePosition( frameBeginTick, section.end );

                    const float rectX = m_MeterPositionX + positionLeft;
                    const float rectY = m_Position.y + m_BarHeight * meterCount + depthHeight * section.depth + margin;
                    const float rectWidth  = positionRight - positionLeft;
                    const float rectHeight = depthHeight;

                    if(rectWidth > m_WidthThreshold && positionLeft < m_MeterWidth && section.depth <= depthCountMax)
                    {
                        pRenderer->SetColor( section.color );
                        pRenderer->Draw2DRect( pCommandBuffer, rectX, rectY, rectWidth, rectHeight );
                    }
                }
                ++meterCount;
            }
        }

        // 外枠を描画
        pRenderer->SetLineWidth( 1.f );
        pRenderer->SetColor( m_BorderColor );
        pRenderer->Draw2DFrame( pCommandBuffer, m_MeterPositionX, m_Position.y, m_MeterWidth, height );

        // メーターとメーターを区切る線を描画
        meterCount = 1;
        for (LoadMeterList::iterator it = pRootMeter->m_LoadMeterList.begin(); it != pRootMeter->m_LoadMeterList.end(); ++it)
        {
            if (it != pRootMeter->m_LoadMeterList.begin())
            {
                if (m_VisibleNoResult || it->GetLastSectionCount() > 0)
                {
                    float positionY = m_Position.y + m_BarHeight * meterCount;
                    pRenderer->SetColor(m_BorderColor);
                    pRenderer->SetLineWidth( 2.f );
                    pRenderer->Draw2DLine(pCommandBuffer, m_MeterPositionX, positionY, m_MeterPositionX + m_MeterWidth, positionY);

                    ++meterCount;
                }
            }
        }

        // フレームの区切り線を描画
        pRenderer->SetColor(m_BorderColor);
        nn::os::Tick tick = frameBeginTick;
        for (int i = 0; i < m_CurrentScale - 1; ++i)
        {
            tick += nn::os::Tick(m_ReferenceFrame);
            float positionX = m_MeterPositionX + CalculateTimeLinePosition(frameBeginTick, tick);
            pRenderer->SetLineWidth( 2.f );
            pRenderer->SetColor(m_BorderColor);
            pRenderer->Draw2DLine(pCommandBuffer, positionX, m_Position.y, positionX, m_Position.y + height);
        }
    }
} //NOLINT(impl/function_size)

float MeterDrawer::CalculateTimeLinePosition( nn::os::Tick frameBeginTick, nn::os::Tick tick ) NN_NOEXCEPT
{
    int64_t totalTime = nn::os::Tick( m_ReferenceFrame ).GetInt64Value() * m_CurrentScale;
    return m_MeterWidth * (tick.GetInt64Value() - frameBeginTick.GetInt64Value()) / totalTime;
}

void MeterDrawer::UpdateScale(nn::perf::LoadMeterBase* pRootMeter) NN_NOEXCEPT
{
    if(m_Scale == 0)
    {
        uint64_t referenceFrameTime = nn::os::Tick( m_ReferenceFrame ).GetInt64Value();
        uint64_t actualFrameTime = static_cast<uint64_t>( pRootMeter->GetLastTotalEnd().GetInt64Value() - pRootMeter->GetLastResult( 0 ).begin.GetInt64Value() );
        int scale = static_cast<uint32_t>(  actualFrameTime / referenceFrameTime ) + 1;
        if(scale < 1)
        {
            scale = 1;
        }
        if ( scale > m_ScaleMax )
        {
            scale = m_ScaleMax;
        }

        if (m_CurrentScale == scale)
        {
            m_AdjustAddCounter = m_AdjustFrameCount;
            m_AdjustReduceCounter = m_AdjustFrameCount;
        }
        else if ( m_CurrentScale < scale )
        {
            m_AdjustReduceCounter = m_AdjustFrameCount;
            if ( m_AdjustAddCounter > 0 )
            {
                m_AdjustAddCounter--;
            }
            else
            {
                m_CurrentScale = scale;
                m_AdjustAddCounter = m_AdjustFrameCount;
            }
        }
        else
        {
            m_AdjustAddCounter = m_AdjustFrameCount;
            if ( m_AdjustReduceCounter > 0 )
            {
                m_AdjustReduceCounter--;
            }
            else
            {
                m_CurrentScale = scale;
                m_AdjustReduceCounter = m_AdjustFrameCount;
            }
        }
    }
    else
    {
        m_CurrentScale = m_Scale;
    }
}

} // namespace PrimitiveRenderer
} // namespace gfx
} // namespace nns
