﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nns/gfx/gfx_ColorBuffer.h>
#include <nn/nn_SdkLog.h>

namespace nns
{
namespace gfx
{
//--------------------------------------------------------------------------------------------------

void ColorBuffer::Initialize(nn::gfx::Device* pDevice,
                             const nn::gfx::TextureInfo& infoType,
                             nn::gfx::MemoryPool* pMemoryPool,
                             ptrdiff_t memoryPoolOffset,
                             size_t memorySize) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDevice);
    NN_ASSERT_NOT_NULL(pMemoryPool);
    NN_ASSERT(infoType.GetArrayLength() <= MaxViewCount);
    NN_ASSERT( nn::gfx::GpuAccess_ColorBuffer & infoType.GetGpuAccessFlags() );

    // テクスチャの初期化
    {
        int viewCount = infoType.GetArrayLength() > 1 ? infoType.GetArrayLength() + 1 : 1;

        m_Texture.Initialize(pDevice, infoType, pMemoryPool, memoryPoolOffset, memorySize);
        m_MemoryPoolOffset = memoryPoolOffset;

        {
            nn::gfx::TextureView::InfoType textureViewInfo;
            textureViewInfo.SetDefault();
            if (viewCount > 1)
            {
                textureViewInfo.SetImageDimension(nn::gfx::ImageDimension_2dArray);
                textureViewInfo.EditSubresourceRange().EditArrayRange().SetArrayLength(infoType.GetArrayLength());
                textureViewInfo.EditSubresourceRange().EditArrayRange().SetBaseArrayIndex(0);
            }
            else
            {
                textureViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
                textureViewInfo.EditSubresourceRange().EditArrayRange().SetArrayLength(1);
                textureViewInfo.EditSubresourceRange().EditArrayRange().SetBaseArrayIndex(0);
            }
            textureViewInfo.SetImageFormat(infoType.GetImageFormat());
            textureViewInfo.SetTexturePtr(&m_Texture);
            m_TextureView.Initialize(pDevice, textureViewInfo);
        }

        m_ViewCount = viewCount;
    }

    {
        nn::gfx::ColorTargetView::InfoType targetViewInfo;
        targetViewInfo.SetDefault();
        if (GetViewCount() > 1)
        {
            targetViewInfo.SetImageDimension(nn::gfx::ImageDimension_2dArray);
            targetViewInfo.EditArrayRange().SetArrayLength(infoType.GetArrayLength());
            targetViewInfo.EditArrayRange().SetBaseArrayIndex(0);
        }
        else
        {
            targetViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
            targetViewInfo.EditArrayRange().SetArrayLength(1);
            targetViewInfo.EditArrayRange().SetBaseArrayIndex(0);
        }
        targetViewInfo.SetImageFormat( infoType.GetImageFormat());
        targetViewInfo.SetTexturePtr(GetTexture());
        m_ColorTargetView[0].Initialize(pDevice, targetViewInfo);
    }

    // テクスチャ配列の場合、各要素ごとのビューを作成する
    if (GetViewCount() > 1)
    {
        for (int viewIndex = 1; viewIndex < GetViewCount(); ++viewIndex)
        {
            nn::gfx::ColorTargetView* view = &m_ColorTargetView[viewIndex];
            {
                nn::gfx::ColorTargetView::InfoType targetViewInfo;
                targetViewInfo.SetDefault();
                targetViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
                targetViewInfo.SetTexturePtr(GetTexture());
                targetViewInfo.EditArrayRange().SetArrayLength(1);
                targetViewInfo.EditArrayRange().SetBaseArrayIndex(viewIndex - 1);
                view->Initialize(pDevice, targetViewInfo);
            }
        }
    }
}

void ColorBuffer::Finalize(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDevice);

    for (int index = 0; index < GetViewCount(); ++index)
    {
        GetColorTargetView(index)->Finalize(pDevice);
    }

    m_TextureView.Finalize(pDevice);
    m_Texture.Finalize(pDevice);
}


} // namespace gfx
} // namespace nns
