﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nns/g3d/g3d_System.h>
#include <nns/g3d/g3d_RenderModel.h>
#include <nn/gfx/util/gfx_ObjectDebugLabel.h>

namespace nns { namespace g3d {

namespace {

bool g_VerboseEnabled = false;
bool g_Initialized = false;
nn::mem::StandardAllocator g_MemoryAllocator;
nn::gfx::util::MemoryPoolAllocator g_MemoryPoolAllocator;
nn::gfx::MemoryPool g_MemoryPool;

void* AllocateFunction(size_t size, void* pParam) NN_NOEXCEPT
{
    NN_UNUSED(pParam);
    return g_MemoryAllocator.Allocate(size);
}

void FreeFunction(void* ptr, void* pParam) NN_NOEXCEPT
{
    NN_UNUSED(pParam);
    g_MemoryAllocator.Free(ptr);
}

}

void SetVerbose(bool enabled) NN_NOEXCEPT
{
    g_VerboseEnabled = enabled;
}

bool IsVerbose() NN_NOEXCEPT
{
    return g_VerboseEnabled;
}

nn::mem::StandardAllocator* GetMemoryAllocator() NN_NOEXCEPT
{
    return &g_MemoryAllocator;
}

nn::gfx::util::MemoryPoolAllocator* GetMemoryPoolAllocator() NN_NOEXCEPT
{
    return &g_MemoryPoolAllocator;
}

nn::gfx::MemoryPool* GetMemoryPool() NN_NOEXCEPT
{
    return &g_MemoryPool;
}

size_t GetMemoryPoolAlignment(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    nn::gfx::MemoryPoolInfo memoryPoolInfo;
    memoryPoolInfo.SetDefault();
    memoryPoolInfo.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached);

    size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice, memoryPoolInfo);

    return alignment;
}

void Initialize(
    nn::gfx::Device* pDevice,
    void* pMemory,
    size_t memorySize,
    void* pMemoryPoolMemory,
    size_t memoryPoolMemorySize
) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pMemory);
    NN_ASSERT_NOT_NULL(pMemoryPoolMemory);
    NN_ASSERT(memorySize > 0);
    NN_ASSERT(memoryPoolMemorySize > 0);

    g_MemoryAllocator.Initialize(pMemory, memorySize);

    nn::gfx::MemoryPoolInfo memoryPoolInfo;
    memoryPoolInfo.SetDefault();
    memoryPoolInfo.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached);
    memoryPoolInfo.SetPoolMemory(pMemoryPoolMemory, memoryPoolMemorySize);

    size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice, memoryPoolInfo);
    NN_UNUSED(alignment);
    NN_ASSERT_ALIGNED(pMemoryPoolMemory, alignment);
    NN_ASSERT((memoryPoolMemorySize % nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(pDevice, memoryPoolInfo)) == 0);
    g_MemoryPool.Initialize(pDevice, memoryPoolInfo);
    nn::gfx::util::SetMemoryPoolDebugLabel(&g_MemoryPool, "nns_g3d");
    g_MemoryPoolAllocator.Initialize(AllocateFunction, NULL, FreeFunction, NULL, &g_MemoryPool, 0, memoryPoolMemorySize, 4096, true);

    nns::g3d::RenderMaterial::InitializeDefaultRenderState(pDevice);

    g_Initialized = true;
}

void Finalize(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    if (g_Initialized)
    {
        nns::g3d::RenderMaterial::FinalizeDefaultRenderState(pDevice);

        g_MemoryPoolAllocator.Finalize();
        g_MemoryPool.Finalize(pDevice);
        //g_MemoryAllocator.Dump();
        g_MemoryAllocator.Finalize();
        g_Initialized = false;
    }
}

}}
