﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nns/g3d/g3d_RenderModel.h>
#include <nns/g3d/g3d_Utility.h>

namespace nns { namespace g3d {

nn::gfx::BlendState         RenderMaterial::g_DefaultBlendState;
nn::gfx::DepthStencilState  RenderMaterial::g_DefaultDepthStencilState;
nn::gfx::RasterizerState    RenderMaterial::g_DefaultRasterizerState;

void RenderMaterial::InitializeDefaultRenderState(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // デフォルトのブレンドステート初期化
    if (!nn::gfx::IsInitialized(g_DefaultBlendState))
    {
        nn::gfx::BlendStateInfo blendStateInfo;
        blendStateInfo.SetDefault();
        nn::gfx::BlendTargetStateInfo blendTargetStateInfo;
        blendTargetStateInfo.SetDefault();
        blendStateInfo.SetBlendTargetStateInfoArray(&blendTargetStateInfo, 1);
        size_t memorySize = nn::gfx::BlendState::GetRequiredMemorySize(blendStateInfo);

        void* pData = Allocate(memorySize, nn::gfx::BlendState::RequiredMemoryInfo_Alignment);
        g_DefaultBlendState.SetMemory(pData, memorySize);
        g_DefaultBlendState.Initialize(pDevice, blendStateInfo);
    }

    // デフォルトのデプスステンシルステート初期化
    if (!nn::gfx::IsInitialized(g_DefaultDepthStencilState))
    {
        nn::gfx::DepthStencilStateInfo depthStencilStateInfo;
        depthStencilStateInfo.SetDefault();
        depthStencilStateInfo.SetDepthTestEnabled(true);
        depthStencilStateInfo.SetDepthWriteEnabled(true);
        g_DefaultDepthStencilState.Initialize(pDevice, depthStencilStateInfo);
    }

    // デフォルトのラスタライザーステート初期化
    if (!nn::gfx::IsInitialized(g_DefaultRasterizerState))
    {
        nn::gfx::RasterizerStateInfo rasterizerStateInfo;
        rasterizerStateInfo.SetDefault();
        rasterizerStateInfo.SetScissorEnabled(true);
        g_DefaultRasterizerState.Initialize(pDevice, rasterizerStateInfo);
    }
}

void RenderMaterial::FinalizeDefaultRenderState(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // デフォルトのブレンドステート初期化
    if (nn::gfx::IsInitialized(g_DefaultBlendState))
    {
        void* pData = g_DefaultBlendState.GetMemory();
        g_DefaultBlendState.Finalize(pDevice);
        if (pData)
        {
            Free(pData);
        }
    }

    // デフォルトのデプスステンシルステート初期化
    if (nn::gfx::IsInitialized(g_DefaultDepthStencilState))
    {
        g_DefaultDepthStencilState.Finalize(pDevice);
    }

    // デフォルトのラスタライザーステート初期化
    if (nn::gfx::IsInitialized(g_DefaultRasterizerState))
    {
        g_DefaultRasterizerState.Finalize(pDevice);
    }
}

bool RenderMaterial::Initialize(nn::gfx::Device* pDevice, nn::g3d::ResMaterial* pResMaterial, nn::g3d::ResShaderArchive* pResShaderArchive) NN_NOEXCEPT
{
    NN_ASSERT(!IsInitialized());
    NN_ASSERT_NOT_NULL(pDevice);
    NN_ASSERT_NOT_NULL(pResMaterial);
    NN_ASSERT_NOT_NULL(pResShaderArchive);

    const nn::g3d::ResShaderAssign* pResShaderAssign = pResMaterial->GetShaderAssign();
    nn::g3d::ResShadingModel* pResShadingModel = pResShaderArchive->FindShadingModel(pResShaderAssign->GetShadingModelName());
    if (pResShadingModel && strcmp(pResShaderArchive->GetName(), pResShaderAssign->GetShaderArchiveName()) == 0)
    {
        Initialize(pDevice, pResMaterial, pResShadingModel);
        return true;
    }
    else
    {
        NNS_G3D_WARNING("Failed to initialize RenderMaterial. Shading model \"%s\" bound to material \"%s\" was not found in shader archive \"%s\".",
            pResMaterial->GetName(),
            pResShaderAssign->GetShadingModelName(),
            pResShaderArchive->GetName());
        return false;
    }
}

bool RenderMaterial::Initialize(nn::gfx::Device* pDevice, nn::g3d::ResMaterial* pResMaterial, nn::g3d::ResShadingModel* pResShadingModel) NN_NOEXCEPT
{
    NN_ASSERT(!IsInitialized());

    m_pResMaterial = pResMaterial;
    m_pResShadingModel = pResShadingModel;

    // シェーダパラメータの変換コンバータを設定します
    int materialBlockIndex = pResShadingModel->GetSystemBlockIndex(nn::g3d::ResUniformBlockVar::Type_Material);
    if (materialBlockIndex >= 0)
    {
        nn::g3d::ResUniformBlockVar* pMaterialBlockVar = pResShadingModel->GetUniformBlock(materialBlockIndex);
        int shaderParamCount = pResMaterial->GetShaderParamCount();
        for (int shaderParamIndex = 0; shaderParamIndex < shaderParamCount; ++shaderParamIndex)
        {
            nn::g3d::ResShaderParam* pResShaderParam = pResMaterial->GetShaderParam(shaderParamIndex);
            if (nn::g3d::ResUniformVar* pUniformVar = pMaterialBlockVar->FindUniform(pResShaderParam->GetId()))
            {
                const char* pConverter = pUniformVar->GetConverter();

                // デフォルトの処理。
                if (pConverter && strcmp(pConverter, "texsrt_ex") == 0)
                {
                    pResShaderParam->SetConvertShaderParamCallback(nn::g3d::ResShaderParam::ConvertTexSrtExCallback);
                    pResMaterial->SetVolatile(shaderParamIndex);
                }
            }
        }
    }

    // サンプラーを初期化します
    int samplerCount = pResShadingModel->GetSamplerCount();
    const nn::g3d::ResShaderAssign* pResShaderAssign = pResMaterial->GetShaderAssign();
    if (samplerCount)
    {
        m_pModelSamplerIndex = Allocate<int>(sizeof(int) * samplerCount, NN_ALIGNOF(int));
        for (int samplerIndex = 0; samplerIndex < samplerCount; ++samplerIndex)
        {
            const char* id = pResShadingModel->GetSamplerName(samplerIndex);
            // id で assign を参照します
            const char* name = pResShaderAssign->FindSamplerAssign(id);
            if (name)
            {
                // ツールでサンプラーが関連付けた場合
                int modelSamplerIndex = pResMaterial->FindSamplerIndex(name);
                m_pModelSamplerIndex[samplerIndex] = modelSamplerIndex;
                if (m_pModelSamplerIndex[samplerIndex] == nn::util::ResDic::Npos)
                {
                    NNS_G3D_WARNING("%s dose not exist.(samplerIndex = %d)\n", name, samplerIndex);
                }
            }
            else
            {
                // ツールでサンプラーが関連付けられていない場合
                // 決め打ちのサンプラーを挿入するもしくは
                // シェーダアノテーションで指定した alt に応じた処理を行うことができます
                m_pModelSamplerIndex[samplerIndex] = nn::util::ResDic::Npos;
            }
        }
    }

    // マテリアルと関連付けられているシェーディングモデルなら、マテリアルのシェーダーパラメーターの情報を反映
    m_IsMaterialShaderAssigned = (strcmp(pResShaderAssign->GetShadingModelName(), pResShadingModel->GetName()) == 0);
    if (m_IsMaterialShaderAssigned)
    {
        nn::g3d::ShaderUtility::BindShaderParam(pResMaterial, pResShadingModel);
    }

    // nn::g3d::ShadingModelObj を作成
    {
        nn::g3d::ShadingModelObj::Builder builder(m_pResShadingModel);
        m_pShadingModelObj = CreateShadingModelObj(pDevice, builder);
    }

    nn::g3d::ShaderUtility::InitializeShaderKey(m_pShadingModelObj, pResShaderAssign, m_IsMaterialShaderAssigned);
    m_pShadingModelObj->UpdateShaderRange();
    m_pShadingModelObj->CalculateOptionBlock();

    m_IsInitialized = true;
    return true;
}

void RenderMaterial::Finalize(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());

    Free(m_pModelSamplerIndex);
    m_pModelSamplerIndex = NULL;
    m_pRasterizerState = NULL;
    m_pDepthStencilState = NULL;
    m_pBlendState = NULL;

    // nn::g3d::ShadingModelObj を破棄
    DestroyShadingModelObj(pDevice, m_pShadingModelObj);
    m_pShadingModelObj = NULL;
    m_pResShadingModel = NULL;
    m_pResMaterial = NULL;

    m_IsInitialized = false;
}

void RenderMaterial::SetRenderStateTo(nn::gfx::CommandBuffer* pCommandBuffer) const NN_NOEXCEPT
{
    // ブレンドステート設定
    if (m_pBlendState)
    {
        pCommandBuffer->SetBlendState(m_pBlendState);
    }
    else
    {
        pCommandBuffer->SetBlendState(&g_DefaultBlendState);
    }
    // ラスタライザーステート設定
    if (m_pRasterizerState)
    {
        pCommandBuffer->SetRasterizerState(m_pRasterizerState);
    }
    else
    {
        pCommandBuffer->SetRasterizerState(&g_DefaultRasterizerState);
    }
    // デプスステンシルステート設定
    if (m_pDepthStencilState)
    {
        pCommandBuffer->SetDepthStencilState(m_pDepthStencilState);
    }
    else
    {
        pCommandBuffer->SetDepthStencilState(&g_DefaultDepthStencilState);
    }
}

//------------------------------------------------------------------------------------------------------------

bool RenderShape::Initialize(nn::gfx::Device* pDevice, nn::g3d::ResShape* pResShape, RenderMaterial* pRenderMaterial) NN_NOEXCEPT
{
    NN_ASSERT(!IsInitialized());

    if (!pRenderMaterial->IsInitialized())
    {
        NNS_G3D_WARNING("Failed to initialize RenderShape for shape \"%s\". Because specified RenderMaterial was not initialized.", pResShape->GetName());
        return false;
    }

    m_pResShape = pResShape;
    m_pRenderMaterial = pRenderMaterial;

    // 頂点ステートを初期化
    InititalizeVertexState(pDevice, &m_VertexState);

    // シェーダーセレクターを初期化
    nn::g3d::ShaderSelector::Builder builder(pRenderMaterial->GetShadingModelObj());
    m_pShaderSelector = CreateShaderSelector(builder);

    // サンプラー管理領域を初期化
    InitializeSamplerList();

    m_IsInitialized = true;
    return true;
}

void RenderShape::Finalize(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());

    m_pUniformBlockList = NULL;
    m_pShaderStorageBlockList = NULL;

    FinalizeSamplerList();

    DestroyShaderSelector(m_pShaderSelector);
    m_pShaderSelector = NULL;

    FinalizeVertexState(pDevice);

    m_pRenderMaterial = NULL;
    m_pResShape = NULL;

    m_IsInitialized = false;
}


void RenderShape::SetRenderStateTo(nn::gfx::CommandBuffer* pCommandBuffer, const nns::g3d::RenderUnitObj* pRenderUnitObj, int viewIndex, int bufferIndex) const NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());

    // 頂点ステートを設定
    pCommandBuffer->SetVertexState(&m_VertexState);

    m_pRenderMaterial->SetRenderStateTo(pCommandBuffer);

    // シェーダーのロード
    const nn::g3d::ResShaderProgram* pResShaderProgram = m_pShaderSelector->GetProgram();
    if (pResShaderProgram == nullptr)
    {
        pResShaderProgram = m_pShaderSelector->GetDefaultProgram();

        if (pResShaderProgram == nullptr)
        {
            nn::g3d::ShadingModelObj* pShadingModelObj = m_pShaderSelector->GetShadingModel();

            NNS_G3D_WARNING("%s is not found default shader program.\n", pShadingModelObj->GetName());
            NNS_G3D_WARNING("shape: %s\n", this->m_pResShape->GetName());

            const int stringCount = 1024;
            char pString[stringCount];

            m_pShaderSelector->PrintRawOptionTo(pString, sizeof(pString));
            NNS_G3D_WARNING("dynamic optionkey: %s\n", pString);

            pShadingModelObj->PrintRawOptionTo(pString, sizeof(pString));
            NNS_G3D_WARNING("static optionkey: %s\n", pString);
        }
    }
    NN_ASSERT_NOT_NULL(pResShaderProgram);
    pResShaderProgram->Load(pCommandBuffer);

    // サンプラーのロード
    LoadTextureAndSampler(pCommandBuffer, pResShaderProgram, pRenderUnitObj);

    // ユニフォームブロックのロード
    LoadUniformBlock(pCommandBuffer, pResShaderProgram, viewIndex, bufferIndex, pRenderUnitObj);

    // シェーダーストレージブロックのロード
    LoadShaderStorageBlock(pCommandBuffer, pResShaderProgram, bufferIndex, pRenderUnitObj);
}

// モデルの描画が大量に行われる際にボトルネックになるためインラインにする
NN_FORCEINLINE
void RenderShape::LoadTextureAndSampler(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::gfx::DescriptorSlot& textureDescriptorSlot,
    const nn::gfx::DescriptorSlot& samplerDescriptorSlot,
    const nn::g3d::ResShaderProgram* pResShaderProgram,
    int samplerIndex
) const NN_NOEXCEPT
{
    int locationVertexShader = pResShaderProgram->GetSamplerLocation(samplerIndex, nn::g3d::Stage_Vertex);
    if (locationVertexShader != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetTextureAndSampler(locationVertexShader, nn::gfx::ShaderStage_Vertex, textureDescriptorSlot, samplerDescriptorSlot);
    }
    int locationGeometryShader = pResShaderProgram->GetSamplerLocation(samplerIndex, nn::g3d::Stage_Geometry);
    if (locationGeometryShader != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetTextureAndSampler(locationGeometryShader, nn::gfx::ShaderStage_Geometry, textureDescriptorSlot, samplerDescriptorSlot);
    }
    int locationFragmentShader = pResShaderProgram->GetSamplerLocation(samplerIndex, nn::g3d::Stage_Pixel);
    if (locationFragmentShader != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetTextureAndSampler(locationFragmentShader, nn::gfx::ShaderStage_Pixel, textureDescriptorSlot, samplerDescriptorSlot);
    }
    int locationComputeShader = pResShaderProgram->GetSamplerLocation(samplerIndex, nn::g3d::Stage_Compute);
    if (locationComputeShader != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetTextureAndSampler(locationComputeShader, nn::gfx::ShaderStage_Compute, textureDescriptorSlot, samplerDescriptorSlot);
    }
}

// モデルの描画が大量に行われる際にボトルネックになるためインラインにする
NN_FORCEINLINE
void RenderShape::LoadTextureAndSampler(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::g3d::ResShaderProgram* pResShaderProgram,
    const nns::g3d::RenderUnitObj* pRenderUnitObj
) const NN_NOEXCEPT
{
    const nn::g3d::ShadingModelObj* pShadingModelObj = m_pShaderSelector->GetShadingModel();
    const nn::g3d::ModelObj* pModelObj = pRenderUnitObj->GetModelObj();
    const nn::g3d::MaterialObj* pMaterialObj = pModelObj->GetMaterial(pRenderUnitObj->GetShapeObj()->GetMaterialIndex());
    int samplerCount = pShadingModelObj->GetResource()->GetSamplerCount();

    const nns::g3d::Sampler* pDummySampler = pRenderUnitObj->GetDummySampler();
    for (int samplerIndex = 0; samplerIndex < samplerCount; ++samplerIndex)
    {
        // ユーザー指定のサンプラーを優先的に設定
        Sampler& sampler = m_pSamplerList[samplerIndex];
        if (sampler.IsInitialized())
        {
            LoadTextureAndSampler(pCommandBuffer, sampler.GetTextureDescriptorSlot(), sampler.GetSamplerDescriptorSlot(), pResShaderProgram, samplerIndex);
            continue;
        }

        // モデルのサンプラーを設定
        int modelSamplerIndex = m_pRenderMaterial->GetModelSamplerIndex(samplerIndex);
        if (modelSamplerIndex != nn::util::ResDic::Npos)
        {
            nn::g3d::TextureRef textureRef = pMaterialObj->GetTexture(modelSamplerIndex);
            if (textureRef.GetTextureView())
            {
                nn::g3d::SamplerRef samplerRef = pMaterialObj->GetSampler(modelSamplerIndex);
                LoadTextureAndSampler(pCommandBuffer, textureRef.GetDescriptorSlot(), samplerRef.GetDescriptorSlot(), pResShaderProgram, samplerIndex);
                continue;
            }
        }

        if(pDummySampler->IsInitialized())
        {
            LoadTextureAndSampler(pCommandBuffer, pDummySampler->GetTextureDescriptorSlot(), pDummySampler->GetSamplerDescriptorSlot(), pResShaderProgram, samplerIndex);
            continue;
        }

        NNS_G3D_WARNING("Texture is not found.\n (shape: %s , samplerIndex: %d)\n", m_pResShape->GetName(), samplerIndex);
    }
}

// モデルの描画が大量に行われる際にボトルネックになるためインラインにする
NN_FORCEINLINE
void RenderShape::LoadUniformBlock(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::gfx::Buffer* pBuffer,
    const nn::g3d::ResShaderProgram* pResShaderProgram,
    int blockIndex,
    size_t size
) const NN_NOEXCEPT
{
    nn::gfx::GpuAddress gpuAddress;
    pBuffer->GetGpuAddress(&gpuAddress);
    int locationVS = pResShaderProgram->GetUniformBlockLocation(blockIndex, nn::g3d::Stage_Vertex);
    if (locationVS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetConstantBuffer(locationVS, nn::gfx::ShaderStage_Vertex, gpuAddress, size);
    }
    int locationGS = pResShaderProgram->GetUniformBlockLocation(blockIndex, nn::g3d::Stage_Geometry);
    if (locationGS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetConstantBuffer(locationGS, nn::gfx::ShaderStage_Geometry, gpuAddress, size);
    }
    int locationFS = pResShaderProgram->GetUniformBlockLocation(blockIndex, nn::g3d::Stage_Pixel);
    if (locationFS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetConstantBuffer(locationFS, nn::gfx::ShaderStage_Pixel, gpuAddress, size);
    }
    int locationCS = pResShaderProgram->GetUniformBlockLocation(blockIndex, nn::g3d::Stage_Compute);
    if (locationCS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetConstantBuffer(locationCS, nn::gfx::ShaderStage_Compute, gpuAddress, size);
    }
}

// モデルの描画が大量に行われる際にボトルネックになるためインラインにする
NN_FORCEINLINE
void RenderShape::LoadUniformBlock(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::g3d::ResShaderProgram* pResShaderProgram,
    int viewIndex,
    int bufferIndex,
    const nns::g3d::RenderUnitObj* pRenderUnitObj
) const NN_NOEXCEPT
{
    const nn::g3d::ModelObj*    pModelObj = pRenderUnitObj->GetModelObj();
    const nn::g3d::ShapeObj*    pShapeObj = pRenderUnitObj->GetShapeObj();
    nn::g3d::ShadingModelObj*   pShadingModelObj = m_pShaderSelector->GetShadingModel();

    // オプション
    const nn::g3d::ResShadingModel* pResShadingModel = pShadingModelObj->GetResource();
    int optionUniformBlockIndex = pResShadingModel->GetSystemBlockIndex(nn::g3d::ResUniformBlockVar::Type_Option);
    const nn::gfx::Buffer* pOptionUniformBlock = pShadingModelObj->GetOptionBlock();
    if (optionUniformBlockIndex >= 0 && pShadingModelObj->IsBlockBufferValid())
    {
        size_t size = pShadingModelObj->GetOptionBlockSize();
        LoadUniformBlock(pCommandBuffer, pOptionUniformBlock, pResShaderProgram, optionUniformBlockIndex, size);
    }

    // マテリアル
    const nn::g3d::MaterialObj* pMaterialObj = pModelObj->GetMaterial(pShapeObj->GetMaterialIndex());
    int materialUniformBlockIndex = pResShadingModel->GetSystemBlockIndex(nn::g3d::ResUniformBlockVar::Type_Material);
    const nn::gfx::Buffer* pMaterialUniformBlock = pMaterialObj->GetMaterialBlock(bufferIndex);
    if (materialUniformBlockIndex >= 0 && pMaterialObj->IsBlockBufferValid())
    {
        size_t size = pMaterialObj->GetMaterialBlockSize();
        LoadUniformBlock(pCommandBuffer, pMaterialUniformBlock, pResShaderProgram, materialUniformBlockIndex, size);
    }

    // スケルトン
    const nn::g3d::SkeletonObj* pSkeletonObj = pModelObj->GetSkeleton();
    int skeletonUniformBlockIndex = pResShadingModel->GetSystemBlockIndex(nn::g3d::ResUniformBlockVar::Type_Skeleton);
    const nn::gfx::Buffer* pSkeletonUniformBlock = pSkeletonObj->GetMtxBlock(bufferIndex);
    if (skeletonUniformBlockIndex >= 0 && pSkeletonObj->IsBlockBufferValid())
    {
        size_t size = pSkeletonObj->GetMtxBlockSize();
        LoadUniformBlock(pCommandBuffer, pSkeletonUniformBlock, pResShaderProgram, skeletonUniformBlockIndex, size);
    }

    // シェイプ
    int shapeUniformBlockIndex = pResShadingModel->GetSystemBlockIndex(nn::g3d::ResUniformBlockVar::Type_Shape);
    const nn::gfx::Buffer* pShapeUniformBlock = pShapeObj->GetShapeBlock(viewIndex, bufferIndex);
    if (shapeUniformBlockIndex >= 0 && pShapeObj->IsBlockBufferValid())
    {
        size_t size = sizeof(nn::g3d::ShapeBlock);
        LoadUniformBlock(pCommandBuffer, pShapeUniformBlock, pResShaderProgram, shapeUniformBlockIndex, size);
    }

    // ユーザーが指定したユニフォームブロック。これ以前に設定したユニフォームブロックの設定を上書き可能。
    int uniformBlockCount = pResShadingModel->GetUniformBlockCount();
    for (int uniformBlockIndex = 0; uniformBlockIndex < uniformBlockCount; ++uniformBlockIndex)
    {
        const UniformBlock& uniformBlock = m_pUniformBlockList[uniformBlockIndex];
        // ユニフォームブロックのインデックスが無効でない
        if (uniformBlock.IsInitialized())
        {
            NN_ASSERT(bufferIndex < uniformBlock.GetCount());
            LoadUniformBlock(pCommandBuffer, uniformBlock.GetBuffer(bufferIndex), pResShaderProgram, uniformBlockIndex, uniformBlock.GetSize());
        }
    }
}

NN_FORCEINLINE void RenderShape::LoadShaderStorageBlock(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::gfx::Buffer* pBuffer,
    const nn::g3d::ResShaderProgram* pShaderProgram,
    int blockIndex,
    size_t size
) const NN_NOEXCEPT
{
    nn::gfx::GpuAddress gpuAddress;
    pBuffer->GetGpuAddress(&gpuAddress);
    int locationVS = pShaderProgram->GetShaderStorageBlockLocation(blockIndex, nn::g3d::Stage_Vertex);
    if (locationVS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetUnorderedAccessBuffer(locationVS, nn::gfx::ShaderStage_Vertex, gpuAddress, size);
    }
    int locationGS = pShaderProgram->GetShaderStorageBlockLocation(blockIndex, nn::g3d::Stage_Geometry);
    if (locationGS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetUnorderedAccessBuffer(locationGS, nn::gfx::ShaderStage_Geometry, gpuAddress, size);
    }
    int locationFS = pShaderProgram->GetShaderStorageBlockLocation(blockIndex, nn::g3d::Stage_Pixel);
    if (locationFS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetUnorderedAccessBuffer(locationFS, nn::gfx::ShaderStage_Pixel, gpuAddress, size);
    }
    int locationCS = pShaderProgram->GetShaderStorageBlockLocation(blockIndex, nn::g3d::Stage_Compute);
    if (locationCS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetUnorderedAccessBuffer(locationCS, nn::gfx::ShaderStage_Compute, gpuAddress, size);
    }
}

NN_FORCEINLINE
void RenderShape::LoadShaderStorageBlock(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::g3d::ResShaderProgram* pResShaderProgram,
    int bufferIndex,
    const nns::g3d::RenderUnitObj* pRenderUnitObj
) const NN_NOEXCEPT
{
    nn::g3d::ShadingModelObj*       pShadingModelObj = m_pShaderSelector->GetShadingModel();
    const nn::g3d::ResShadingModel* pResShadingModel = pShadingModelObj->GetResource();
    const nn::g3d::ModelObj*        pModelObj = pRenderUnitObj->GetModelObj();

    // スケルトン
    const nn::g3d::SkeletonObj* pSkeletonObj = pModelObj->GetSkeleton();
    int skeletonShaderStorageBlockIndex = pResShadingModel->GetSystemShaderStorageBlockIndex(nn::g3d::ResShaderStorageBlockVar::Type_Skeleton);
    const nn::gfx::Buffer* pSkeletonShaderStorageBlock = pSkeletonObj->GetMtxBlock(bufferIndex);
    if (skeletonShaderStorageBlockIndex >= 0 && pSkeletonObj->IsBlockBufferValid())
    {
        size_t size = pSkeletonObj->GetMtxBlockSize();
        LoadShaderStorageBlock(pCommandBuffer, pSkeletonShaderStorageBlock, pResShaderProgram, skeletonShaderStorageBlockIndex, size);
    }

    // ユーザーが指定したシェーダーストレージブロック。これ以前に設定したシェーダーストレージブロックの設定を上書き可能。
    int shaderStorageBlockCount = pResShadingModel->GetShaderStorageBlockCount();
    for (int shaderStorageBlockIndex = 0; shaderStorageBlockIndex < shaderStorageBlockCount; ++shaderStorageBlockIndex)
    {
        const ShaderStorageBlock& shaderStorageBlock = m_pShaderStorageBlockList[shaderStorageBlockIndex];
        // シェーダーストレージブロックのインデックスが無効でない
        if (shaderStorageBlock.IsInitialized())
        {
            NN_ASSERT(bufferIndex < shaderStorageBlock.GetCount());
            LoadShaderStorageBlock(pCommandBuffer, shaderStorageBlock.GetBuffer(bufferIndex), pResShaderProgram, shaderStorageBlockIndex, shaderStorageBlock.GetSize());
        }
    }
}

void RenderShape::UpdateShader(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    bool isSucceeded = m_pShaderSelector->UpdateVariation(pDevice);
    if (!isSucceeded)
    {
        nn::g3d::ShadingModelObj* pShadingModelObj = m_pShaderSelector->GetShadingModel();
        const int stringCount = 1024;
        char pDynamicOptionKey[stringCount];
        char pStaticOptionKey[stringCount];
        m_pShaderSelector->PrintRawOptionTo(pDynamicOptionKey, sizeof(pDynamicOptionKey));
        pShadingModelObj->PrintRawOptionTo(pStaticOptionKey, sizeof(pStaticOptionKey));

        NN_ASSERT(isSucceeded,
            "Failed to update a shader.\n"
            "shading model: %s\n"
            "shape: %s\n"
            "dynamic option key: %s\n"
            "static option key: %s\n",
            pShadingModelObj->GetName(),
            m_pResShape->GetName(),
            pDynamicOptionKey,
            pStaticOptionKey);
    }
}

void RenderShape::InitializeSampler(
    const char* id,
    int samplerIndex,
    const nn::gfx::DescriptorSlot& textureDescriptorSlot,
    const nn::gfx::DescriptorSlot& samplerDescriptorSlot
) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(id);
    NN_ASSERT(textureDescriptorSlot.IsValid());
    NN_ASSERT(samplerDescriptorSlot.IsValid());

    m_pSamplerList[samplerIndex].Initialize(id, samplerDescriptorSlot, textureDescriptorSlot);
}

nns::g3d::Sampler* RenderShape::FindSampler(const char* id) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(id);
    NN_ASSERT_RANGE(strlen(id), 0UL, nns::g3d::Sampler::MaxIdSize - 1);

    const nn::g3d::ResShadingModel* pResShadingModel = m_pShaderSelector->GetShadingModel()->GetResource();
    int samplerCount = pResShadingModel->GetSamplerCount();
    for (int samplerIndex = 0; samplerIndex < samplerCount; ++samplerIndex)
    {
        nns::g3d::Sampler& sampler = m_pSamplerList[samplerIndex];
        if (strncmp(sampler.GetId(), id, nns::g3d::Sampler::MaxIdSize) != 0)
        {
            continue;
        }

        return &sampler;
    }
    return NULL;
}

void RenderShape::FinalizeSampler(const char* id) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(id);
    NN_ASSERT_RANGE(strlen(id), 0UL, nns::g3d::Sampler::MaxIdSize - 1);

    nns::g3d::Sampler* pSampler = FindSampler(id);
    if (!pSampler)
    {
        return;
    }

    pSampler->Finalize();
}

void RenderShape::InititalizeVertexState(nn::gfx::Device* pDevice, nn::gfx::VertexState* pVertexState) NN_NOEXCEPT
{
    const nn::g3d::ResShadingModel* pResShadingModel = m_pRenderMaterial->GetResShadingModel();
    const nn::g3d::ResVertex*       pResVertex = m_pResShape->GetVertex();

    const int shaderVertexAttrCount = pResShadingModel->GetAttrCount();
    const int vertexBufferCount = pResVertex->GetVertexBufferCount();

    // 必要なメモリー量を計算
    nn::util::MemorySplitter::MemoryBlock memoryBlock[MemoryBlock_Max];
    for (int memoryBlockIndex = MemoryBlock_AttribStateInfo; memoryBlockIndex < MemoryBlock_Max; ++memoryBlockIndex)
    {
        memoryBlock[memoryBlockIndex].Initialize();
    }
    memoryBlock[MemoryBlock_AttribStateInfo].SetSizeBy<nn::gfx::VertexAttributeStateInfo>(NN_ALIGNOF(nn::gfx::VertexAttributeStateInfo), shaderVertexAttrCount);
    memoryBlock[MemoryBlock_BufferStateInfo].SetSizeBy<nn::gfx::VertexBufferStateInfo>(NN_ALIGNOF(nn::gfx::VertexBufferStateInfo), vertexBufferCount);

    nn::util::MemorySplitter memorySplitter;
    memorySplitter.Initialize();
    for (int memoryBlockIndex = MemoryBlock_AttribStateInfo; memoryBlockIndex < MemoryBlock_Max; ++memoryBlockIndex)
    {
        memorySplitter.Append(&memoryBlock[memoryBlockIndex]);
    }

    // Info 用のワークメモリーを確保します
    void* pData = Allocate(memorySplitter.GetSize(), memorySplitter.GetAlignment());
    NN_ASSERT_NOT_NULL(pData);

    nn::gfx::VertexAttributeStateInfo*  pVertexAttributeStateInfo = memoryBlock[MemoryBlock_AttribStateInfo].Get<nn::gfx::VertexAttributeStateInfo>(pData);
    nn::gfx::VertexBufferStateInfo*     pVertexBufferStateInfo = memoryBlock[MemoryBlock_BufferStateInfo].Get<nn::gfx::VertexBufferStateInfo>(pData);

    // 頂点ステートを構築
    nn::gfx::VertexState::InfoType vertexStateInfo;
    vertexStateInfo.SetDefault();

    int validVertexAttributeCount = 0;
    const nn::g3d::ResShaderAssign* pResShaderAssign = m_pRenderMaterial->GetResMaterial()->GetShaderAssign();
    for (int shaderVertexAttributeIndex = 0; shaderVertexAttributeIndex < shaderVertexAttrCount; ++shaderVertexAttributeIndex)
    {
        // シェーダーの頂点属性の ID を取得
        const char* attributeId = pResShadingModel->GetAttrName(shaderVertexAttributeIndex);

        // 頂点属性の ID でモデルにおける頂点属性名を取得
        const char* attributeName = pResShaderAssign->FindAttrAssign(attributeId);
        if (!attributeName)
        {
            continue;
        }

        const nn::g3d::ResVertexAttr* pResVertexAttribute = pResVertex->FindVertexAttr(attributeName);
        if (!pResVertexAttribute)
        {
            continue;
        }

        const nn::g3d::ResAttrVar* pResAttributeVariable = pResShadingModel->GetAttr(shaderVertexAttributeIndex);

        // 頂点属性情報を設定
        pVertexAttributeStateInfo[validVertexAttributeCount].SetDefault();
        pVertexAttributeStateInfo[validVertexAttributeCount].SetBufferIndex(pResVertexAttribute->GetBufferIndex());
        pVertexAttributeStateInfo[validVertexAttributeCount].SetFormat(pResVertexAttribute->GetFormat());
        pVertexAttributeStateInfo[validVertexAttributeCount].SetOffset(pResVertexAttribute->GetOffset());
        pVertexAttributeStateInfo[validVertexAttributeCount].SetShaderSlot(pResAttributeVariable->GetLocation());
        pVertexAttributeStateInfo[validVertexAttributeCount].SetNamePtr(NULL);

        ++validVertexAttributeCount;
    }
    vertexStateInfo.SetVertexAttributeStateInfoArray(pVertexAttributeStateInfo, validVertexAttributeCount);

    // ResVertex 中のすべてのバッファーを設定
    for (int vertexBufferIndex = 0; vertexBufferIndex < vertexBufferCount; ++vertexBufferIndex)
    {
        pVertexBufferStateInfo[vertexBufferIndex].SetDefault();
        pVertexBufferStateInfo[vertexBufferIndex].SetStride(pResVertex->GetVertexBufferStride(vertexBufferIndex));
    }
    vertexStateInfo.SetVertexBufferStateInfoArray(pVertexBufferStateInfo, vertexBufferCount);

    // VertexState を初期化
    size_t memorySize = nn::gfx::VertexState::GetRequiredMemorySize(vertexStateInfo);
    void* pVertexStateMemory = Allocate(memorySize, nn::gfx::VertexState::RequiredMemoryInfo_Alignment);
    NN_ASSERT_NOT_NULL(pVertexStateMemory);

    pVertexState->SetMemory(pVertexStateMemory, memorySize);
    pVertexState->Initialize(pDevice, vertexStateInfo, NULL);

    // Info用のワークメモリを解放
    Free(pData);
}

void RenderShape::FinalizeVertexState(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    void* ptr = m_VertexState.GetMemory();
    m_VertexState.Finalize(pDevice);
    Free(ptr);
}

void RenderShape::InitializeSamplerList() NN_NOEXCEPT
{
    const nn::g3d::ResShadingModel* pResShadingModel = m_pRenderMaterial->GetResShadingModel();
    int samplerCount = pResShadingModel->GetSamplerCount();
    if (samplerCount <= 0)
    {
        return;
    }

    m_pSamplerList = Allocate<nns::g3d::Sampler>(sizeof(nns::g3d::Sampler) * samplerCount);
    NN_ASSERT_NOT_NULL(m_pSamplerList);

    for (int samplerIndex = 0; samplerIndex < samplerCount; ++samplerIndex)
    {
        new(&m_pSamplerList[samplerIndex]) nns::g3d::Sampler();
    }
}

void RenderShape::FinalizeSamplerList() NN_NOEXCEPT
{
    const nn::g3d::ResShadingModel* pResShadingModel = m_pRenderMaterial->GetResShadingModel();
    int samplerCount = pResShadingModel->GetSamplerCount();
    if (samplerCount == 0)
    {
        return;
    }

    for (int samplerIndex = 0; samplerIndex < samplerCount; ++samplerIndex)
    {
        if (!m_pSamplerList[samplerIndex].IsInitialized())
        {
            continue;
        }

        m_pSamplerList[samplerIndex].Finalize();
    }

    Free(m_pSamplerList);
    m_pSamplerList = NULL;
}

//------------------------------------------------------------------------------------------------------------

void RenderModel::Initialize(nn::gfx::Device* pDevice, nn::g3d::ResModel* pResModel, int passCount) NN_NOEXCEPT
{
    NN_ASSERT(!IsInitialized());
    NN_ASSERT_NOT_NULL(pResModel);
    NN_ASSERT_GREATER(passCount, 0);

    NN_UNUSED(pDevice);

    int materialCount = pResModel->GetMaterialCount();
    if (materialCount > 0)
    {
        // RenderMaterial を マテリアル数 x パス数、生成します。
        nns::g3d::RenderMaterial* pRenderMaterial = Allocate<RenderMaterial>(sizeof(RenderMaterial) * materialCount * passCount);
        NN_ASSERT_NOT_NULL(pRenderMaterial);

        m_pRenderMaterialPtr = Allocate<RenderMaterial*>(sizeof(RenderMaterial*) * passCount);
        NN_ASSERT_NOT_NULL(m_pRenderMaterialPtr);

        for (int passIndex = 0; passIndex < passCount; ++passIndex)
        {
            m_pRenderMaterialPtr[passIndex] = &pRenderMaterial[materialCount * passIndex];
            for (int materialIndex = 0; materialIndex < materialCount; ++materialIndex)
            {
                new(&m_pRenderMaterialPtr[passIndex][materialIndex]) RenderMaterial();
            }
        }
    }

    int shapeCount = pResModel->GetShapeCount();
    if (shapeCount > 0)
    {
        // RenderShape を シェイプ数 x パス数、生成します。
        nns::g3d::RenderShape* pRenderShape = Allocate<RenderShape>(sizeof(RenderShape) * shapeCount * passCount);
        NN_ASSERT_NOT_NULL(pRenderShape);

        m_pRenderShapePtr = Allocate<RenderShape*>(sizeof(RenderShape*) * passCount);
        NN_ASSERT_NOT_NULL(m_pRenderShapePtr);

        for (int passIndex = 0; passIndex < passCount; ++passIndex)
        {
            m_pRenderShapePtr[passIndex] = &pRenderShape[shapeCount * passIndex];
            for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
            {
                new(&m_pRenderShapePtr[passIndex][shapeIndex]) RenderShape();
            }
        }
    }

    m_pResModel = pResModel;
    m_PassCount = passCount;
    m_IsInitialized = true;
}

void RenderModel::Initialize(nn::gfx::Device* pDevice, nn::g3d::ResModel* pResModel) NN_NOEXCEPT
{
    Initialize(pDevice, pResModel, 1);
}

void RenderModel::Finalize(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDevice);
    NN_ASSERT(nn::gfx::IsInitialized(*pDevice));
    NN_ASSERT(IsInitialized());

    for (int passIndex = 0; passIndex < m_PassCount; ++passIndex)
    {
        DestroyDrawPass(pDevice, passIndex);
    }

    // メモリを解放
    if (m_pResModel->GetShapeCount() > 0)
    {
        Free(m_pRenderShapePtr[0]);
        Free(m_pRenderShapePtr);
    }
    if (m_pResModel->GetMaterialCount() > 0)
    {
        Free(m_pRenderMaterialPtr[0]);
        Free(m_pRenderMaterialPtr);
    }

    m_pRenderShapePtr = NULL;
    m_pRenderMaterialPtr = NULL;
    m_IsInitialized = false;
}

bool RenderModel::CreateDrawPass(nn::gfx::Device* pDevice, int passIndex, nn::g3d::ResShaderArchive* pResShaderArchive) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(pDevice);
    NN_ASSERT(nn::gfx::IsInitialized(*pDevice));
    NN_ASSERT_RANGE(passIndex, 0, m_PassCount);
    NN_ASSERT_NOT_NULL(pResShaderArchive);

    bool isSucceeded = true;
    for (int materialIndex = 0, materialCount = m_pResModel->GetMaterialCount(); materialIndex < materialCount; ++materialIndex)
    {
        NN_ASSERT(!m_pRenderMaterialPtr[passIndex][materialIndex].IsInitialized());
        isSucceeded &= m_pRenderMaterialPtr[passIndex][materialIndex].Initialize(pDevice, m_pResModel->GetMaterial(materialIndex), pResShaderArchive);
    }

    for (int shapeIndex = 0, shapeCount = m_pResModel->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
    {
        NN_ASSERT(!m_pRenderShapePtr[passIndex][shapeIndex].IsInitialized());

        nn::g3d::ResShape* pResShape = m_pResModel->GetShape(shapeIndex);
        int materialIndex = pResShape->GetMaterialIndex();
        isSucceeded &= m_pRenderShapePtr[passIndex][shapeIndex].Initialize(pDevice, pResShape, &m_pRenderMaterialPtr[passIndex][materialIndex]);
    }

    return isSucceeded;
}

bool RenderModel::CreateDrawPass(nn::gfx::Device* pDevice, nn::g3d::ResShaderArchive* pResShaderArchive) NN_NOEXCEPT
{
    return CreateDrawPass(pDevice, 0, pResShaderArchive);
}

bool RenderModel::CreateDrawPass(nn::gfx::Device * pDevice, int passIndex, nn::g3d::ResShadingModel* pResShadingModel) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(pDevice);
    NN_ASSERT(nn::gfx::IsInitialized(*pDevice));
    NN_ASSERT_RANGE(passIndex, 0, m_PassCount);
    NN_ASSERT_NOT_NULL(pResShadingModel);

    bool isSucceeded = true;
    for (int materialIndex = 0, materialCount = m_pResModel->GetMaterialCount(); materialIndex < materialCount; ++materialIndex)
    {
        NN_ASSERT(!m_pRenderMaterialPtr[passIndex][materialIndex].IsInitialized());
        m_pRenderMaterialPtr[passIndex][materialIndex].Initialize(pDevice, m_pResModel->GetMaterial(materialIndex), pResShadingModel);
    }

    for (int shapeIndex = 0, shapeCount = m_pResModel->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
    {
        NN_ASSERT(!m_pRenderShapePtr[passIndex][shapeIndex].IsInitialized())

        nn::g3d::ResShape* pResShape = m_pResModel->GetShape(shapeIndex);
        int materialIndex = pResShape->GetMaterialIndex();
        isSucceeded &= m_pRenderShapePtr[passIndex][shapeIndex].Initialize(pDevice, pResShape, &m_pRenderMaterialPtr[passIndex][materialIndex]);
    }

    return isSucceeded;
}

bool RenderModel::CreateDrawPass(nn::gfx::Device* pDevice, nn::g3d::ResShadingModel* pResShadingModel) NN_NOEXCEPT
{
    return CreateDrawPass(pDevice, 0, pResShadingModel);
}

bool RenderModel::CreateDrawPass(nn::gfx::Device* pDevice, int passIndex, nn::g3d::ResShaderArchive** pResShaderArchiveArray, int resShaderArchiveCount) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(pDevice);
    NN_ASSERT(nn::gfx::IsInitialized(*pDevice));
    NN_ASSERT_RANGE(passIndex, 0, m_PassCount);
    NN_ASSERT_NOT_NULL(pResShaderArchiveArray);

    NN_ASSERT_EQUAL(resShaderArchiveCount, m_pResModel->GetMaterialCount());
    NN_UNUSED(resShaderArchiveCount);

    bool isSucceeded = true;
    for (int materialIndex = 0, materialCount = m_pResModel->GetMaterialCount(); materialIndex < materialCount; ++materialIndex)
    {
        NN_ASSERT(!m_pRenderMaterialPtr[passIndex][materialIndex].IsInitialized());
        isSucceeded &= m_pRenderMaterialPtr[passIndex][materialIndex].Initialize(pDevice, m_pResModel->GetMaterial(materialIndex), pResShaderArchiveArray[materialIndex]);
    }

    for (int shapeIndex = 0, shapeCount = m_pResModel->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
    {
        NN_ASSERT(!m_pRenderShapePtr[passIndex][shapeIndex].IsInitialized());

        nn::g3d::ResShape* pResShape = m_pResModel->GetShape(shapeIndex);
        int materialIndex = pResShape->GetMaterialIndex();
        isSucceeded &= m_pRenderShapePtr[passIndex][shapeIndex].Initialize(pDevice, pResShape, &m_pRenderMaterialPtr[passIndex][materialIndex]);
    }

    return isSucceeded;
}

bool RenderModel::CreateDrawPass(nn::gfx::Device* pDevice, nn::g3d::ResShaderArchive** pResShaderArchiveArray, int resShaderArchiveCount) NN_NOEXCEPT
{
    return CreateDrawPass(pDevice, 0, pResShaderArchiveArray, resShaderArchiveCount);
}

void RenderModel::DestroyDrawPass(nn::gfx::Device* pDevice, int passIndex) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(pDevice);
    NN_ASSERT(nn::gfx::IsInitialized(*pDevice));
    NN_ASSERT_RANGE(passIndex, 0, m_PassCount);

    for (int shapeIndex = 0, shapeCount = m_pResModel->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
    {
        if (!m_pRenderShapePtr[passIndex][shapeIndex].IsInitialized())
        {
            continue;
        }

        m_pRenderShapePtr[passIndex][shapeIndex].Finalize(pDevice);
    }

    for (int materialIndex = 0, materialCount = m_pResModel->GetMaterialCount(); materialIndex < materialCount; ++materialIndex)
    {
        if (!m_pRenderMaterialPtr[passIndex][materialIndex].IsInitialized())
        {
            continue;
        }

        m_pRenderMaterialPtr[passIndex][materialIndex].Finalize(pDevice);
    }
}

void RenderModel::DestroyDrawPass(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    DestroyDrawPass(pDevice, 0);
}

}}
