﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dbgui_ImguiIO.h"

#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>

#include <imgui.h>
#include <imgui_internal.h>

#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>

namespace nns { namespace dbgui { namespace io {

namespace {

struct IoState
{
    int         finishTextInputCounter      = 0;
    ImGuiID     finishTextInputActiveId     = 0;
    bool        isFrameStarted              = false;
    bool        wantTextInput               = false;

    int         npadKeyBindings[ImGuiNavInput_COUNT] = { -1 };
    int         debugpadKeyBindings[ImGuiNavInput_COUNT] = { -1 };


    float       mouseWheelScrollRatio       = 1.0f;
};

IoState g_IoState;


template<typename BitfieldType>
void UpdateNavInputState(
    const BitfieldType& bitfield,
    const nn::hid::AnalogStickState& analogStickState,
    int32_t stickActivationThreshold,
    const int* pNavigationKeyBinding)
{
    ImGuiIO& io = ImGui::GetIO();

    for (int navInputIndex = 0; navInputIndex < ImGuiNavInput_COUNT; ++navInputIndex)
    {
        int buttonIndex = pNavigationKeyBinding[navInputIndex];
        if ((buttonIndex >= 0) && bitfield.Test(buttonIndex))
        {
            io.NavInputs[navInputIndex] = 1.0f;
        }
    }

    if (analogStickState.y > stickActivationThreshold)
    {
        io.NavInputs[ImGuiNavInput_LStickUp] = 1.0f;
    }
    if (analogStickState.y < -stickActivationThreshold)
    {
        io.NavInputs[ImGuiNavInput_LStickDown] = 1.0f;
    }
    if (analogStickState.x < -stickActivationThreshold)
    {
        io.NavInputs[ImGuiNavInput_LStickLeft] = 1.0f;
    }
    if (analogStickState.x > stickActivationThreshold)
    {
        io.NavInputs[ImGuiNavInput_LStickRight] = 1.0f;
    }
}

void SetKeyBindings(int* pNavInputBidings, const int* pInterfaceInfoBindings)
{
    pNavInputBidings[ImGuiNavInput_Activate]     = pInterfaceInfoBindings[NavigationKeyBinding_Activate];
    pNavInputBidings[ImGuiNavInput_Cancel]       = pInterfaceInfoBindings[NavigationKeyBinding_Cancel];
    pNavInputBidings[ImGuiNavInput_Input]        = pInterfaceInfoBindings[NavigationKeyBinding_Input];
    pNavInputBidings[ImGuiNavInput_Menu]         = pInterfaceInfoBindings[NavigationKeyBinding_Menu];
    pNavInputBidings[ImGuiNavInput_DpadUp]       = pInterfaceInfoBindings[NavigationKeyBinding_Up];
    pNavInputBidings[ImGuiNavInput_DpadDown]     = pInterfaceInfoBindings[NavigationKeyBinding_Down];
    pNavInputBidings[ImGuiNavInput_DpadLeft]     = pInterfaceInfoBindings[NavigationKeyBinding_Left];
    pNavInputBidings[ImGuiNavInput_DpadRight]    = pInterfaceInfoBindings[NavigationKeyBinding_Right];
    pNavInputBidings[ImGuiNavInput_LStickUp]     = -1;
    pNavInputBidings[ImGuiNavInput_LStickDown]   = -1;
    pNavInputBidings[ImGuiNavInput_LStickLeft]   = -1;
    pNavInputBidings[ImGuiNavInput_LStickRight]  = -1;
    pNavInputBidings[ImGuiNavInput_FocusPrev]    = pInterfaceInfoBindings[NavigationKeyBinding_FocusPrev];
    pNavInputBidings[ImGuiNavInput_FocusNext]    = pInterfaceInfoBindings[NavigationKeyBinding_FocusNext];
    pNavInputBidings[ImGuiNavInput_TweakSlow]    = pInterfaceInfoBindings[NavigationKeyBinding_TweakSlow];
    pNavInputBidings[ImGuiNavInput_TweakFast]    = pInterfaceInfoBindings[NavigationKeyBinding_TweakFast];
}

} // anonymous namespace

void Initialize(const ImguiIoInfo* pInfo)
{
    ImGui::SetAllocatorFunctions(
        pInfo->pMemoryCallbackAllocate,
        pInfo->pMemoryCallbackFree,
        pInfo->pMemoryCallbackUserData);
    ImGui::CreateContext();

    ImGuiIO& io = ImGui::GetIO();

    if (pInfo->pCustomFontBuffer != nullptr)
    {
        if (pInfo->customFontIsCompressed)
        {
            io.Fonts->AddFontFromMemoryCompressedTTF(
                pInfo->pCustomFontBuffer,
                static_cast<int>(pInfo->customFontBufferSize),
                pInfo->customFontCharacterSize,
                NULL,
                io.Fonts->GetGlyphRangesJapanese());
        }
        else
        {
            void* pCopyBuffer =
                pInfo->pMemoryCallbackAllocate(pInfo->customFontBufferSize, pInfo->pMemoryCallbackUserData);
            memcpy(pCopyBuffer, pInfo->pCustomFontBuffer, pInfo->customFontBufferSize);

            ImFontConfig fontConfiguration = ImFontConfig();
            fontConfiguration.FontData = pCopyBuffer;
            fontConfiguration.FontDataSize = static_cast<int>(pInfo->customFontBufferSize);
            fontConfiguration.SizePixels = pInfo->customFontCharacterSize;
            fontConfiguration.GlyphRanges = io.Fonts->GetGlyphRangesJapanese();
            // バッファはimgui内側で解放する
            fontConfiguration.FontDataOwnedByAtlas = true;
            io.Fonts->AddFont(&fontConfiguration);
        }
    }
    else
    {
        io.Fonts->AddFontDefault();
    }

    io.DisplaySize = ImVec2(
        static_cast<float>(pInfo->displayWidth),
        static_cast<float>(pInfo->displayHeight));

    io.DisplayFramebufferScale = ImVec2(1.0f, 1.0f);

    io.DeltaTime = 1.0f / 60.0f;

    io.IniFilename = pInfo->iniFilePath;

    ImGui::GetStyle().WindowRounding = 4;

    io.RenderDrawListsFn = nullptr;

    SetKeyBindings(g_IoState.npadKeyBindings, pInfo->pNpadNavigationKeyBindings);
    SetKeyBindings(g_IoState.debugpadKeyBindings, pInfo->pDebugNpadNavigationKeyBindings);

    NN_ASSERT(pInfo->mouseWheelScrollRatio > 0.0f);
    g_IoState.mouseWheelScrollRatio = pInfo->mouseWheelScrollRatio;

    g_IoState.isFrameStarted = false;
}

void Finalize()
{
    NN_ASSERT(!g_IoState.isFrameStarted);

    ImGuiIO& io = ImGui::GetIO();
    io.RenderDrawListsFn = nullptr;

    ImGui::DestroyContext();
    ImGui::SetAllocatorFunctions(nullptr, nullptr, nullptr);
}

bool IsMoveMouseRequested()
{
    NN_ASSERT(!g_IoState.isFrameStarted);
    ImGuiIO& io = ImGui::GetIO();
    return io.WantMoveMouse;
}

void GetMoveMouseRequestedPos(int* pPosX, int* pPosY)
{
    NN_ASSERT(!g_IoState.isFrameStarted);
    ImGuiIO& io = ImGui::GetIO();
    NN_ASSERT(io.WantMoveMouse);
    *pPosX = static_cast<int>(io.MousePos.x);
    *pPosY = static_cast<int>(io.MousePos.y);
}

void NewFrame(nn::TimeSpan frameDuration)
{
    NN_ASSERT(!g_IoState.isFrameStarted);

    ImGuiIO& io = ImGui::GetIO();
    uint64_t durationInUs = frameDuration.GetMicroSeconds();
    float deltaTime = static_cast<float>(durationInUs) / (1000.f * 1000.0f);

    io.DeltaTime = deltaTime;

    ImGui::NewFrame();

    g_IoState.isFrameStarted = true;
}

void Render()
{
    NN_ASSERT(g_IoState.isFrameStarted);
    ImGui::Render();
    g_IoState.isFrameStarted = false;
}

ImDrawData* GetDrawData()
{
    NN_ASSERT(!g_IoState.isFrameStarted);
    ImDrawData* pDrawData = ImGui::GetDrawData();
    NN_ASSERT_NOT_NULL(pDrawData);
    return pDrawData;
}

void SetupUserInputConfiguration()
{
    NN_ASSERT(!g_IoState.isFrameStarted);

    ImGuiIO& io = ImGui::GetIO();
    io.KeyMap[ImGuiKey_Tab] = nn::hid::KeyboardKey::Tab::Index;
    io.KeyMap[ImGuiKey_LeftArrow] = nn::hid::KeyboardKey::LeftArrow::Index;
    io.KeyMap[ImGuiKey_RightArrow] = nn::hid::KeyboardKey::RightArrow::Index;
    io.KeyMap[ImGuiKey_UpArrow] = nn::hid::KeyboardKey::UpArrow::Index;
    io.KeyMap[ImGuiKey_DownArrow] = nn::hid::KeyboardKey::DownArrow::Index;
    io.KeyMap[ImGuiKey_PageUp] = nn::hid::KeyboardKey::PageUp::Index;
    io.KeyMap[ImGuiKey_PageDown] = nn::hid::KeyboardKey::PageDown::Index;
    io.KeyMap[ImGuiKey_Home] = nn::hid::KeyboardKey::Home::Index;
    io.KeyMap[ImGuiKey_End] = nn::hid::KeyboardKey::End::Index;
    io.KeyMap[ImGuiKey_Delete] = nn::hid::KeyboardKey::Delete::Index;
    io.KeyMap[ImGuiKey_Backspace] = nn::hid::KeyboardKey::Backspace::Index;
    io.KeyMap[ImGuiKey_Enter] = nn::hid::KeyboardKey::Return::Index;
    io.KeyMap[ImGuiKey_Escape] = nn::hid::KeyboardKey::Escape::Index;
    io.KeyMap[ImGuiKey_A] = nn::hid::KeyboardKey::A::Index;
    io.KeyMap[ImGuiKey_C] = nn::hid::KeyboardKey::C::Index;
    io.KeyMap[ImGuiKey_V] = nn::hid::KeyboardKey::V::Index;
    io.KeyMap[ImGuiKey_X] = nn::hid::KeyboardKey::X::Index;
    io.KeyMap[ImGuiKey_Y] = nn::hid::KeyboardKey::Y::Index;
    io.KeyMap[ImGuiKey_Z] = nn::hid::KeyboardKey::Z::Index;

    io.NavFlags |= ImGuiNavFlags_EnableGamepad;
    io.NavFlags |= ImGuiNavFlags_MoveMouse;

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    io.NavActive = true;
    io.MouseDrawCursor = true;
    ImGui::SetMouseCursor(ImGuiMouseCursor_Arrow);
#else
    io.NavActive = false;
    io.MouseDrawCursor = false;
#endif
}

void ResetPadNavigationInputs()
{
    NN_ASSERT(!g_IoState.isFrameStarted);

    ImGuiIO& io = ImGui::GetIO();

    io.NavInputs[ImGuiNavInput_Activate] = 0.0f;
    io.NavInputs[ImGuiNavInput_Cancel] = 0.0f;
    io.NavInputs[ImGuiNavInput_Input] = 0.0f;
    io.NavInputs[ImGuiNavInput_Menu] = 0.0f;
    io.NavInputs[ImGuiNavInput_DpadUp] = 0.0f;
    io.NavInputs[ImGuiNavInput_DpadDown] = 0.0f;
    io.NavInputs[ImGuiNavInput_DpadLeft] = 0.0f;
    io.NavInputs[ImGuiNavInput_DpadRight] = 0.0f;
    io.NavInputs[ImGuiNavInput_LStickUp] = 0.0f;
    io.NavInputs[ImGuiNavInput_LStickDown] = 0.0f;
    io.NavInputs[ImGuiNavInput_LStickLeft] = 0.0f;
    io.NavInputs[ImGuiNavInput_LStickRight] = 0.0f;
    io.NavInputs[ImGuiNavInput_FocusPrev] = 0.0f;
    io.NavInputs[ImGuiNavInput_FocusNext] = 0.0f;
    io.NavInputs[ImGuiNavInput_TweakSlow] = 0.0f;
    io.NavInputs[ImGuiNavInput_TweakFast] = 0.0f;
}

void ResetMouseInputs()
{
    NN_ASSERT(!g_IoState.isFrameStarted);

    ImGuiIO& io = ImGui::GetIO();
    //io.MousePos = ImVec2(0.0f, 0.0f);
    io.MouseWheel = 0;

    const int mouseButtonCount = sizeof(io.MouseDown) / sizeof(io.MouseDown[0]);
    for (int mouseButtonIndex = 0; mouseButtonIndex < mouseButtonCount; mouseButtonIndex++)
    {
        io.MouseDown[mouseButtonIndex] = false;
    }
}

void UpdateNavigationFromDebugPadState(
    const nn::hid::DebugPadButtonSet& debugPadButtonSet,
    const nn::hid::AnalogStickState& analogStickState,
    int32_t stickActivationThreshold)
{
    NN_ASSERT(!g_IoState.isFrameStarted);

    UpdateNavInputState(
        debugPadButtonSet, analogStickState,
        stickActivationThreshold,
        g_IoState.debugpadKeyBindings);
}

void UpdateNavigationFromNpadState(
    const nn::hid::NpadButtonSet& buttons,
    const nn::hid::AnalogStickState& analogStickState,
    int32_t stickActivationThreshold)
{
    NN_ASSERT(!g_IoState.isFrameStarted);

    UpdateNavInputState(
        buttons, analogStickState,
        stickActivationThreshold,
        g_IoState.npadKeyBindings);
}


void UpdateKeyboardState(const nn::hid::KeyboardState& keyboardState)
{
    NN_ASSERT(!g_IoState.isFrameStarted);

    ImGuiIO& io = ImGui::GetIO();
    io.KeyCtrl = keyboardState.modifiers.Test<nn::hid::KeyboardModifier::Control>();
    io.KeyShift = keyboardState.modifiers.Test<nn::hid::KeyboardModifier::Shift>();
    io.KeyAlt = keyboardState.modifiers.Test<nn::hid::KeyboardModifier::LeftAlt>()
        || keyboardState.modifiers.Test<nn::hid::KeyboardModifier::RightAlt>();
    io.KeySuper = keyboardState.modifiers.Test<nn::hid::KeyboardModifier::Gui>();

    int keyCount = keyboardState.keys.GetCount();
    NN_ASSERT(keyCount < (sizeof(ImGuiIO::KeysDown) / sizeof(ImGuiIO::KeysDown[0])));
    for (int keyIndex = 0; keyIndex < keyCount; ++keyIndex)
    {
        bool isPressed = keyboardState.keys.Test(keyIndex);
        io.KeysDown[keyIndex] = isPressed;
    }
}

void UpdateMouseState(const nn::hid::MouseState& mouseState)
{
    NN_ASSERT(!g_IoState.isFrameStarted);

    ImGuiIO& io = ImGui::GetIO();
    io.MousePos = ImVec2(
        static_cast<float>(mouseState.x),
        static_cast<float>(mouseState.y));

    io.MouseWheel = static_cast<float>(mouseState.wheelDelta) * g_IoState.mouseWheelScrollRatio;
    io.MouseWheelH = 0.0f;

    if (mouseState.buttons.Test<nn::hid::MouseButton::Left>())
        io.MouseDown[0] = true;
    if (mouseState.buttons.Test<nn::hid::MouseButton::Right>())
        io.MouseDown[1] = true;
    if (mouseState.buttons.Test<nn::hid::MouseButton::Middle>())
        io.MouseDown[2] = true;
}

void UpdateMouseStateFromTouchState(const nn::hid::TouchState& touchState)
{
    NN_ASSERT(!g_IoState.isFrameStarted);

    ImGuiIO& io = ImGui::GetIO();
    io.MousePos = ImVec2(
        static_cast<float>(touchState.x),
        static_cast<float>(touchState.y));

    if (touchState.fingerId == 0)
        io.MouseDown[0] = true;
}

void AddInputCharacter(char charToAdd)
{
    NN_ASSERT(!g_IoState.isFrameStarted);

    ImGuiIO& io = ImGui::GetIO();
    io.AddInputCharacter(charToAdd);
}

void AddInputCharactersUtf8(const char* utf8CharsToAdd)
{
    NN_ASSERT(!g_IoState.isFrameStarted);

    ImGuiIO& io = ImGui::GetIO();
    io.AddInputCharactersUTF8(utf8CharsToAdd);
}

#if defined(NNS_DBGUI_ENABLE_SOFTWARE_KEYBOARD)

bool WantTextInput()
{
    NN_ASSERT(!g_IoState.isFrameStarted);

    ImGuiIO& io = ImGui::GetIO();
    NN_ASSERT(g_IoState.finishTextInputCounter == 0);

    bool wantTextInput = io.WantTextInput;
    g_IoState.wantTextInput = wantTextInput;
    if (wantTextInput)
    {
        ImGuiTextEditState* pInputTextState = &ImGui::GetCurrentContext()->InputTextState;
        pInputTextState->SelectAll();
    }

    return wantTextInput;
}

void BeginTextInput(TextInputProperties* pOutTextInputProperties)
{
    NN_ASSERT_NOT_NULL(pOutTextInputProperties);
    NN_ASSERT(!g_IoState.isFrameStarted);
    NN_ASSERT(ImGui::GetIO().WantTextInput);

    const ImGuiTextEditState* pInputTextState = &ImGui::GetCurrentContext()->InputTextState;
    const char* pInitialText = &pInputTextState->TempTextBuffer[0];
    bool isMultiline = !pInputTextState->StbState.single_line;

    g_IoState.finishTextInputCounter = 1;
    g_IoState.finishTextInputActiveId = ImGui::GetActiveID();

    pOutTextInputProperties->pInitialText = pInitialText;
    pOutTextInputProperties->isMultiline = isMultiline;
}

bool FinishTextInput()
{
    NN_ASSERT(!g_IoState.isFrameStarted);

    bool result = true;

    if (g_IoState.finishTextInputCounter > 0)
    {
        switch (g_IoState.finishTextInputCounter)
        {
        case 1:
            result = false;
            g_IoState.finishTextInputCounter++;
            break;

        case 2:
            if (g_IoState.finishTextInputActiveId == ImGui::GetActiveID())
            {
                result = false;
                ImGui::ClearActiveID();
                g_IoState.finishTextInputCounter++;
            }
            else
            {
                g_IoState.finishTextInputCounter = 0;
            }
            break;

        case 3:
            result = false;
            g_IoState.finishTextInputCounter++;
            break;

        case 4:
            g_IoState.finishTextInputCounter = 0;
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    return result;
}
#endif

} } } // namespace nns { namespace dbgui { namespace io {
