﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nns/console/console_SimpleConsole.h>

#include <cstring>
#include <cstdlib>
#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Mutex.h>

namespace nns {
namespace console {

const size_t SimpleConsole::DefaultBufferSize = 1024;
const size_t SimpleConsole::BufferSizeMax = 4096;

SimpleConsole::SimpleConsole() NN_NOEXCEPT
    : m_Lock(false)
    , m_BlinkEnabled(true)
    , m_Counter(0)
    , m_CurrentLength(0)
{
    m_Buffer = reinterpret_cast<char*>(std::malloc(DefaultBufferSize));
    NN_ABORT_UNLESS_NOT_NULL(m_Buffer);
    m_Buffer[0] = '\0';
    m_BufferSize = DefaultBufferSize;
}
SimpleConsole::~SimpleConsole() NN_NOEXCEPT
{
    std::free(m_Buffer);
}
void SimpleConsole::UpdateImpl() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    m_Counter = (m_Counter + 1) % 60;
}

void SimpleConsole::MakeCommandImpl(nn::gfx::util::DebugFontTextWriter* pTextWriter) const NN_NOEXCEPT
{
    auto& textWriter = *pTextWriter;

    std::lock_guard<nn::os::Mutex> lock(m_Lock);

    bool printed = false;
    if (m_LineBreaks.size() > 0)
    {
        // 一画面に収まる範囲を決める
        auto dimension = ConsoleBase::GetDimension();
        auto lineCapacity = static_cast<size_t>(dimension.height / textWriter.GetLineHeight());

        if (m_LineBreaks.size() + 1 > lineCapacity)
        {
            textWriter.Print("%s", m_LineBreaks.at(m_LineBreaks.size() - lineCapacity) + 1);
            printed = true;
        }
    }

    if (!printed)
    {
        textWriter.Print("%s", m_Buffer);
    }

    if (m_BlinkEnabled && m_Counter < 30) {
        textWriter.Print("■");
    }
}

void SimpleConsole::UpdateLineBreaksUnsafe(const char* buffer) NN_NOEXCEPT
{
    auto current = buffer;
    const char* p = nullptr;
    while ((p = std::strchr(current, '\n')) != nullptr)
    {
        m_LineBreaks.push_back(p);
        current = p + 1;
    }
}
bool SimpleConsole::TryExpandBufferUnsafe(size_t expect) NN_NOEXCEPT
{
    auto allocateSize = std::max(m_BufferSize * 2, m_CurrentLength + expect);
    auto memory = (allocateSize <= BufferSizeMax? reinterpret_cast<char*>(std::realloc(m_Buffer, allocateSize)): nullptr);
    if (!(memory != nullptr))
    {
        m_Buffer[0] = '\0';
        m_CurrentLength = 0;
        m_LineBreaks.clear();
        return false;
    }

    if (m_Buffer != memory)
    {
        // 改行リストを作り直す
        m_Buffer = memory;
        m_LineBreaks.clear();
        UpdateLineBreaksUnsafe(m_Buffer);
    }
    m_BufferSize = allocateSize;
    return true;
}

void SimpleConsole::Puts(const char* string, size_t size) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);

    const auto Length = strnlen(string, size);
    NN_ABORT_UNLESS_LESS(Length, size);
    if (Length <= 0)
    {
        return;
    }

    // バッファの調整
    if (!(m_CurrentLength + Length < m_BufferSize))
    {
        if (!TryExpandBufferUnsafe(Length + 1))
        {
            NN_LOG("[nns::console::SimpleConsole] Log buffer emptied\n");
        }
    }

    // 受け取った文字列をコピー
    auto head = m_Buffer + m_CurrentLength;
    strncpy(head, string, Length);
    head[Length] = '\0';
    m_CurrentLength += Length;

    // 改行リストの更新
    UpdateLineBreaksUnsafe(head);
}

void SimpleConsole::Puts(const char* string) NN_NOEXCEPT
{
    const auto Length = strlen(string);
    Puts(string, Length + 1);
}

void SimpleConsole::Clear() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    m_Buffer[0] = '\0';
    m_CurrentLength = 0;
    m_LineBreaks.clear();
}

} // ~namespace nn::console
}
