﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{TextureIDManager.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a class that wraps the texture
 *  and sampler registering process. Textures and samplers
 *  must be registered with the device with an ID so that
 *  they can be used by a shader program.
 */

#include <algorithm>
#include <nvn/nvn_FuncPtrInline.h>
#include <nn/nn_Assert.h>
#include <nvntutorial/TextureIDManager.h>

static const int g_NumPublicTextures = 8192;
static const int g_NumPublicSamplers = 2048;

/*
 * TextureIDManager Constructor
 * ----------------------------
 * Sets up the texture pool and sampler pool for
 * registering textures/samplers with the device.
 * Before being used by a shader, a texture/sampler
 * needs to be bound to the command buffer.
 */
TextureIDManager::TextureIDManager(NVNdevice* pDevice) : m_pDevice(pDevice),
                                                         m_CurrentSamplerID(0),
                                                         m_CurrentTextureID(0)
{
    int textureSize = 0;
    int samplerSize = 0;
    int numReservedTextures = 0;
    int numReservedSamplers = 0;
    int maxTextures = 0;
    int maxSamplers = 0;
    int totalNumSamplers = 0;
    int totalNumTextures = 0;

        /* Queries the size of a single texture/sampler descriptor. */
    nvnDeviceGetInteger(m_pDevice, NVN_DEVICE_INFO_TEXTURE_DESCRIPTOR_SIZE, &textureSize);
    nvnDeviceGetInteger(m_pDevice, NVN_DEVICE_INFO_SAMPLER_DESCRIPTOR_SIZE, &samplerSize);

        /*
         * Grabs the number of texture/sampler entries that
         * are reserved for internal use by NVN.
         */
    nvnDeviceGetInteger(m_pDevice, NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &numReservedTextures);
    nvnDeviceGetInteger(m_pDevice, NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &numReservedSamplers);

        /*
         * Grabs maximum number of texture/sampler entries
         * allowed (including reserved entries)
         */
    nvnDeviceGetInteger(m_pDevice, NVN_DEVICE_INFO_MAX_TEXTURE_POOL_SIZE, &maxTextures);
    nvnDeviceGetInteger(m_pDevice, NVN_DEVICE_INFO_MAX_SAMPLER_POOL_SIZE, &maxSamplers);

        /*
         * The texture pool supports a max of 1048576 and the
         * sampler pool supports a max of 4096 descriptors.
         */
    totalNumTextures = std::min(maxTextures, numReservedTextures + g_NumPublicTextures);
    totalNumSamplers = std::min(maxSamplers, numReservedSamplers + g_NumPublicSamplers);

        /*
         * Create a memory pool with enough space for the
         * reserved texture/sampler descriptors as well as
         * the amount of descriptors that need to be supported
         * for the application.
         * The CPU_UNCACHED flag is necessary as the CPU writes
         * the descriptors to memory itself.  CPU_CACHED would
         * additionally work but would require manual flushing
         * of the needed descriptors to gpu memory.
         */
    m_DescriptorPool.Init(NULL,
                          totalNumSamplers * samplerSize + totalNumTextures * textureSize,
                          NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT,
                          m_pDevice);

        /* Initialize the sampler pool. */
    ptrdiff_t samplerPoolOffset = m_DescriptorPool.GetNewMemoryChunkOffset(totalNumSamplers * samplerSize, samplerSize);
    NVNboolean samplerPoolSuccess = nvnSamplerPoolInitialize(&m_SamplerPool, m_DescriptorPool.GetMemoryPool(), samplerPoolOffset, totalNumSamplers);
    NN_ASSERT(samplerPoolSuccess, "Failed to initialize Sampler descriptor pool\n");

        /* Initialize the texture pool. */
    ptrdiff_t texturePoolOffset = m_DescriptorPool.GetNewMemoryChunkOffset(totalNumTextures * textureSize, textureSize);
    NVNboolean texturePoolSuccess = nvnTexturePoolInitialize(&m_TexturePool, m_DescriptorPool.GetMemoryPool(), texturePoolOffset, totalNumTextures);
    NN_ASSERT(texturePoolSuccess, "Failed to initialize Texture descriptor pool\n");

        /* Grab the starting sampler and texture ID's. */
    m_CurrentSamplerID = numReservedSamplers;
    m_CurrentTextureID = numReservedTextures;
}

/*
 * TextureIDManager Destructor
 * ---------------------------
 * Clean up the memory pool and reset member values.
 */
TextureIDManager::~TextureIDManager()
{
    m_DescriptorPool.Shutdown();

    m_pDevice = NULL;

    m_CurrentSamplerID = 0;
    m_CurrentTextureID = 0;
}

/*
 * TextureIDManager::RegisterTexture
 * ---------------------------------
 * Registers the texture with the texture pool with a
 * unique ID.
 */
int TextureIDManager::RegisterTexture(NVNtexture* pTexture)
{
        /* Thread safe increment. */
    int textureID = m_CurrentTextureID.fetch_add(1);

        /* Register the texture with the texture pool with the new ID. */
    nvnTexturePoolRegisterTexture(&m_TexturePool, textureID, pTexture, NULL);

    return textureID;
}

/*
 * TextureIDManager::RegisterSampler
 * ---------------------------------
 * Registers the sampler with the sampler pool with a
 * unique ID.
 */
int TextureIDManager::RegisterSampler(NVNsampler* pSampler)
{
        /* Thread safe increment. */
    int samplerID = m_CurrentSamplerID.fetch_add(1);

        /* Register the sampler with the sampler pool with the new ID. */
    nvnSamplerPoolRegisterSampler(&m_SamplerPool, samplerID, pSampler);

    return samplerID;
}

/*
 * TextureIDManager::SetSamplerPool
 * ---------------------------------
 * Binds the sampler pool to the given command buffer.
 */
void TextureIDManager::SetSamplerPool(NVNcommandBuffer* pCommandBuffer)
{
    nvnCommandBufferSetSamplerPool(pCommandBuffer, &m_SamplerPool);
}

/*
 * TextureIDManager::SetTexturePool
 * ---------------------------------
 * Binds the texture pool to the given command buffer.
 */
void TextureIDManager::SetTexturePool(NVNcommandBuffer* pCommandBuffer)
{
    nvnCommandBufferSetTexturePool(pCommandBuffer, &m_TexturePool);
}
