﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{GraphicsObjectCube.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a class that derives from the base
 *  GraphicsObject class for drawing the graphicsObjectCube.out
 *  asset file.
 */

#include <nn/util/util_Vector.h>
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nn/nn_Assert.h>
#include <nvntutorial/GraphicsObjectCube.h>
#include <nvntutorial/UniformBufferManager.h>
#include <nvntutorial/ManagedUniformBuffer.h>
#include <nvntutorial/ShaderHeaders/GraphicsObjectCubeShaderDataHelper.h>
#include <nvntutorial/AssetFileDataHolder.h>

/*
 * GraphicsObjectCube Constructor
 * ------------------------------
 * Sets up default values for various members as well as the
 * name of the asset file that this particular derivation of
 * the GraphicsObject is associated with.
 */
GraphicsObjectCube::GraphicsObjectCube() : m_pUniformBlockVS(NULL)
{
    m_Name = "graphicsObjectCube.out";

    nn::util::MatrixIdentity(&m_WorldMatrix);
    nn::util::MatrixIdentity(&m_CameraMatrix);
    nn::util::MatrixIdentity(&m_ProjectionMatrix);
}

/*
 * GraphicsObjectCube::Init
 * ------------------------
 * Sets the data holder that the object is associated
 * with and sets up the proper render state.
 */
void GraphicsObjectCube::Init(StateType type, AssetFileDataHolder* data)
{
    switch(type)
    {
        case StateType_Opaque:
            SetOpaqueState();
            break;
        case StateType_Wire:
            SetWireState();
            break;
        default:
            NN_ASSERT(0, "Unknown State Type");
    }

    m_pAssetData = data;
}

/*
 * GraphicsObjectCube Destructor
 * -----------------------------
 * Empty destructor.
 */
GraphicsObjectCube::~GraphicsObjectCube()
{
}

/*
 * GraphicsObjectCube::SetWireState
 * --------------------------------
 * Sets up the object's render states to draw the model
 * as a wire frame.
 */
void GraphicsObjectCube::SetWireState()
{
    SetDefaultState();

    nvnDepthStencilStateSetDepthTestEnable(&m_DepthStencilState, NVN_TRUE);
    nvnDepthStencilStateSetDepthWriteEnable(&m_DepthStencilState, NVN_TRUE);
    nvnDepthStencilStateSetDepthFunc(&m_DepthStencilState, NVN_DEPTH_FUNC_LESS);

    nvnPolygonStateSetCullFace(&m_PolygonState, NVNface::NVN_FACE_NONE);
    nvnPolygonStateSetPolygonMode(&m_PolygonState, NVN_POLYGON_MODE_LINE);

    nvnColorStateSetBlendEnable(&m_ColorState, 0, NVN_TRUE);
}

/*
 * GraphicsObjectCube::SetOpaqueState
 * ----------------------------------
 * Sets up the object's render states to draw the model
 * with solid triangles.
 */
void GraphicsObjectCube::SetOpaqueState()
{
    SetDefaultState();

    nvnDepthStencilStateSetDepthTestEnable(&m_DepthStencilState, NVN_TRUE);
    nvnDepthStencilStateSetDepthWriteEnable(&m_DepthStencilState, NVN_TRUE);
    nvnDepthStencilStateSetDepthFunc(&m_DepthStencilState, NVN_DEPTH_FUNC_LESS);

    nvnPolygonStateSetFrontFace(&m_PolygonState, NVNfrontFace::NVN_FRONT_FACE_CCW);
    nvnPolygonStateSetCullFace(&m_PolygonState, NVNface::NVN_FACE_BACK);
    nvnPolygonStateSetPolygonMode(&m_PolygonState, NVN_POLYGON_MODE_FILL);

    nvnColorStateSetBlendEnable(&m_ColorState, 0, NVN_TRUE);
}

/*
 * GraphicsObjectCube::SetupUniforms
 * ---------------------------------
 * Sets up the object's uniform buffer.
 */
void GraphicsObjectCube::SetupUniforms(UniformBufferManager* uniformBufferManager)
{
    m_pUniformBlockVS = uniformBufferManager->CreateUniformBuffer(sizeof(GraphicsObjectCubeShader::BlockVSUniformBlockData));
}

/*
 * GraphicsObjectCube::UpdateUniforms
 * ----------------------------------
 * Updates the object's uniform buffer with the previously set
 * member data.
 */
void GraphicsObjectCube::UpdateUniforms()
{
    nn::util::Matrix4x3fType scale;
    nn::util::MatrixIdentity(&scale);
    nn::util::Vector3fType scaleVector;
    nn::util::VectorSet(&scaleVector, m_ScaleX, m_ScaleY, m_ScaleZ);
    nn::util::MatrixSetScale(&scale, scaleVector);

    nn::util::Matrix4x3fType rotateX;
    nn::util::MatrixIdentity(&rotateX);
    nn::util::MatrixIdentity(&rotateX);
    nn::util::Vector3fType rotateXVector;
    nn::util::VectorSet(&rotateXVector, m_RotateX * 3.1459f / 180.0f, 0.0f, 0.0f);
    nn::util::MatrixSetRotateXyz(&rotateX, rotateXVector);

    nn::util::Matrix4x3fType rotateY;
    nn::util::MatrixIdentity(&rotateY);
    nn::util::MatrixIdentity(&rotateY);
    nn::util::Vector3fType rotateYVector;
    nn::util::VectorSet(&rotateYVector, 0.0f, m_RotateY * 3.14159f / 180.0f, 0.0f);
    nn::util::MatrixSetRotateXyz(&rotateY, rotateYVector);

    nn::util::Matrix4x3fType rotateZ;
    nn::util::MatrixIdentity(&rotateZ);
    nn::util::MatrixIdentity(&rotateZ);
    nn::util::Vector3fType rotateZVector;
    nn::util::VectorSet(&rotateZVector, 0.0f, 0.0f, m_RotateZ * 3.14159f / 180.0f);
    nn::util::MatrixSetRotateXyz(&rotateZ, rotateZVector);

    nn::util::Matrix4x3fType translate;
    nn::util::MatrixIdentity(&translate);
    nn::util::MatrixIdentity(&translate);
    nn::util::Vector3fType translateVector;
    nn::util::VectorSet(&translateVector, m_TranslateX, m_TranslateY, m_TranslateZ);
    nn::util::MatrixSetTranslate(&translate, translateVector);

    nn::util::Matrix4x3fType tempMat1;
    nn::util::Matrix4x3fType tempMat2;

    nn::util::MatrixMultiply(&tempMat1, scale, rotateX);
    nn::util::MatrixMultiply(&tempMat2, tempMat1, rotateY);
    nn::util::MatrixMultiply(&tempMat1, tempMat2, rotateZ);
    nn::util::MatrixMultiply(&tempMat2, tempMat1, translate);

    nn::util::MatrixConvert(&m_WorldMatrix, tempMat2);

    GraphicsObjectCubeShader::BlockVSUniformBlockData* uniformMapVS = reinterpret_cast<GraphicsObjectCubeShader::BlockVSUniformBlockData*>(m_pUniformBlockVS->GetMappedPointer());

    nn::util::Float4x4 temp;
    nn::util::MatrixStore(&temp, m_WorldMatrix);
    uniformMapVS->SetUniform_u_modelMtx(*reinterpret_cast<float(*)[16]>(&temp));
    nn::util::MatrixStore(&temp, m_CameraMatrix);
    uniformMapVS->SetUniform_u_viewMtx(*reinterpret_cast<float(*)[16]>(&temp));
    nn::util::MatrixStore(&temp, m_ProjectionMatrix);
    uniformMapVS->SetUniform_u_projMtx(*reinterpret_cast<float(*)[16]>(&temp));
}

/*
 * GraphicsObjectCube::SetWorldMatrix
 * ----------------------------------
 * Sets the objetcts world matrix.
 */
void GraphicsObjectCube::SetWorldMatrix(const nn::util::Matrix4x4fType& world)
{
    m_WorldMatrix = world;
}

/*
 * GraphicsObjectCube::SetCameraMatrix
 * -----------------------------------
 * Sets the current camera matrix.
 */
void GraphicsObjectCube::SetCameraMatrix(const nn::util::Matrix4x4fType& camera)
{
    m_CameraMatrix = camera;
}

/*
 * GraphicsObjectCube::SetProjectionMatrix
 * ---------------------------------------
 * Sets the projection matrix for drawing the object.
 */
void GraphicsObjectCube::SetProjectionMatrix(const nn::util::Matrix4x4fType& projection)
{
    m_ProjectionMatrix = projection;
}

/*
 * GraphicsObjectCube::SetScale
 * ----------------------------
 * Sets the objects world space scale.
 */
void GraphicsObjectCube::SetScale(float x, float y, float z)
{
    m_ScaleX = x;
    m_ScaleY = y;
    m_ScaleZ = z;
}

/*
 * GraphicsObjectCube::SetTranslate
 * --------------------------------
 * Sets the objects world space translation.
 */
void GraphicsObjectCube::SetTranslate(float x, float y, float z)
{
    m_TranslateX = x;
    m_TranslateY = y;
    m_TranslateZ = z;
}

/*
 * GraphicsObjectCube::SetRotate
 * -----------------------------
 * Sets the objects world space rotation.
 */
void GraphicsObjectCube::SetRotate(float x, float y, float z)
{
    m_RotateX = x;
    m_RotateY = y;
    m_RotateZ = z;
}

/*
 * GraphicsObjectCube::Draw
 * ------------------------
 * Draws the object with the given command buffer, its data
 * holder, and the render states that have been set.
 */
void GraphicsObjectCube::Draw(NVNcommandBuffer* pCommandBuffer) const
{
    std::vector<NVNModelData*>&   modelData   = m_pAssetData->GetModelData();
    std::vector<NVNProgramData*>& programData = m_pAssetData->GetProgramData();

        /* Bind the vertex buffer data. */
    Model* pModel = &modelData[0]->m_Model;
    NVNbufferAddress vboAddr = nvnBufferGetAddress(&modelData[0]->m_VertexBuffer);
    for (size_t j = 0; j < pModel->m_VertexAttributes.size(); ++j)
    {
        VertexAttribute& attr = pModel->m_VertexAttributes[j];
        nvnCommandBufferBindVertexBuffer(pCommandBuffer, attr.m_Location, vboAddr + modelData[0]->m_VertexAttributeBufferOffsets[j], attr.m_DataSize);
    }

        /* Bind the uniform buffer. */
    nvnCommandBufferBindUniformBuffer(pCommandBuffer,
                                      NVN_SHADER_STAGE_VERTEX,
                                      GraphicsObjectCubeShader::BlockVSUniformBlockData::GetBinding(NVN_SHADER_STAGE_VERTEX),
                                      m_pUniformBlockVS->GetCurrentBufferAddress(),
                                      sizeof(GraphicsObjectCubeShader::BlockVSUniformBlockData));

        /* Bind the vertex attribute states and vertex stream states. */
    nvnCommandBufferBindVertexAttribState(pCommandBuffer, static_cast<int>(modelData[0]->m_VertexAttributeStates.size()), &modelData[0]->m_VertexAttributeStates[0]);
    nvnCommandBufferBindVertexStreamState(pCommandBuffer, static_cast<int>(modelData[0]->m_VertexStreamStates.size()), &modelData[0]->m_VertexStreamStates[0]);

        /* Bind the shader program. */
    programData[0]->BindShaderProgram(pCommandBuffer);

        /* Draw the model. */
    NVNindexType indexType = (NVNindexType)pModel->m_IndexData.m_IndexType;
    int numIndices = pModel->m_IndexData.m_DataSize / pModel->m_IndexData.m_Stride;
    nvnCommandBufferDrawElements(pCommandBuffer,
                                 (NVNdrawPrimitive)pModel->m_NvnDrawPrimitiveType,
                                 indexType,
                                 numIndices,
                                 nvnBufferGetAddress(&modelData[0]->m_IndexBuffer));
}
