﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/mbuf/mbuf_Mbuf.h>
#include "common.h"

namespace WlanTest {

void SystemInitialize() NN_NOEXCEPT
{
    // デバッグ設定書き込み
    const bool isEnabled = true;
    nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));

    nn::nifm::Initialize();
    nn::nifm::SetWirelessCommunicationEnabledForTest(false);
    // nifmのwlan利用停止を確実に待つために1秒ほどwaitを入れておく
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
}

void SystemFinalize() NN_NOEXCEPT
{
}

void DumpLine(void *ptr, uint32_t skip, uint32_t len) NN_NOEXCEPT
{
    uint8_t *p = static_cast<uint8_t*>(ptr);

    NN_SDK_LOG("%p| ", (static_cast<uint32_t>(reinterpret_cast<uintptr_t>(ptr) & 0xfffffff0))); // 64bit環境では情報落ちする
    for(uint32_t i = 0; i < skip; i++)
    {
        NN_LOG("   ");
    }
    for(uint32_t i = 0; i < len; i++)
    {
        NN_LOG("%02x ", *p);
        p ++;
    }
    for(uint32_t i = (skip + len); i < 0x10; i++)
    {
        NN_LOG("   ");
    }
    NN_LOG("|");

    for(uint32_t i = 0; i < skip; i++)
    {
        NN_LOG(" ");
    }
    p = static_cast<uint8_t*>(ptr);
    for(uint32_t i = 0; i < len; i++)
    {
        NN_LOG("%c", (((*p<0x20)||*p>=0x80)? 0x2e: *p));
        p ++;
    }
    NN_LOG("\n");
}

void DumpBuffer(void* buffp, size_t size) NN_NOEXCEPT
{
    uint8_t *buff = reinterpret_cast<uint8_t*>(buffp);
    uint32_t skip_len, disp_len;

    NN_LOG("Dump Buffer(size=%ld)\n", size);
    NN_LOG("Addr    | +0 +1 +2 +3 +4 +5 +6 +7 +8 +9 +a +b +c +d +e +f |0123456789abcdef\n");

    skip_len = static_cast<uint32_t>(reinterpret_cast<uintptr_t>(buff) & 0xf);
    disp_len = (0x10 - static_cast<uint32_t>(reinterpret_cast<uintptr_t>(buff) & 0xf));
    if(disp_len>size)    disp_len = size;

    do {
        DumpLine(buff, skip_len, disp_len);
        skip_len = 0;

        buff += disp_len;
        size -= disp_len;
        if(size > 0x10)    disp_len = 0x10;
        else            disp_len = size;
    } while(size > 0);
}

void DumpScanResult(char* pBuf) NN_NOEXCEPT
{
    uint8_t* ptr = reinterpret_cast<uint8_t*>(pBuf);

    // Scanバッファの先頭にはScanResultHeader構造体で、スキャン結果に関する情報が入っている。
    nn::wlan::ScanResultHeader* pHeader = reinterpret_cast<nn::wlan::ScanResultHeader*>(ptr);
    if( pHeader->bssCount == 0 )
    {
        NN_LOG("No BSS info!\n");
        return;
    }
    else
    {
        NN_LOG("----------------------- Scan Result ------------------------\n");
        NN_LOG("---- The number of BSS is %d\n", pHeader->bssCount);
        NN_LOG("---- Total Scaned size is %d\n", pHeader->resultLength);
    }

    // ポインタ進める
    ptr = reinterpret_cast<uint8_t*>(ptr) + sizeof(nn::wlan::ScanResultHeader);

    char macStr[nn::wlan::MacAddress::MacStringSize];
    char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
    nn::wlan::MacAddress mac;
    nn::wlan::Ssid ssid;
    nn::wlan::BssInfo* pInfo;
    for(int i=0; i < pHeader->bssCount; i++)
    {
        pInfo = reinterpret_cast<nn::wlan::BssInfo*>(ptr);

        NN_LOG("-----------------------%d BSS------------------------\n", i + 1);
        NN_LOG("BSS size : %d\n", pInfo->bssSize);
        NN_LOG("IE count : %d\n", pInfo->ieCount);
        NN_LOG("Channel : %d\n", pInfo->channel);
        NN_LOG("RSSI : %d[dBm]\n", pInfo->rssi);
        NN_LOG("LINKLEVEL : %d\n", pInfo->linkLevel);
        NN_LOG("Beacon Period : %d[ms]\n", pInfo->beaconPeriod);
        mac.Set(pInfo->bssid);
        mac.GetString(macStr);
        NN_LOG("BSSID : %s\n", macStr);
        ssid.Set(pInfo->ssid, pInfo->ssidLength);
        ssid.GetHexString(ssidStr);
        NN_LOG("SSID : %s\n", ssidStr);

        if(pInfo->ieCount > 0)
        {
            // IEが入っているところまでポインタ進める
            uint8_t* pIe = ptr; // いったんアドレス保存
            pIe = reinterpret_cast<uint8_t*>(pIe) + sizeof(nn::wlan::BssInfo);
            NN_LOG("1st IE Tag : %d (0x%02X)\n", *pIe);
            NN_LOG("1st IE Length : %d\n", *(pIe + 1));
            if( *(pIe + 1) > 0 )
            {
                NN_LOG("1st IE data : %02X:%02X:%02X\n", *(pIe + 2), *(pIe + 3), *(pIe + 4));
            }
        }
        NN_LOG("---------------------------------------------------\n");

        // 次のBSSまでポインタ進める
        ptr = reinterpret_cast<uint8_t*>(ptr) + pInfo->bssSize;
    }
}

void DumpScanResultWithReader(void* pBuf) NN_NOEXCEPT
{
    if( pBuf == NULL )
    {
        return;
    }

    nn::wlan::BeaconScanResultReader resultReader(pBuf);
    uint32_t bssCount = resultReader.GetCount();
    char macStr[nn::wlan::MacAddress::MacStringSize];
    char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
    const nn::wlan::VendorInfoElementReader* vie;
    nn::Bit8 matchingOui1[nn::wlan::MacAddress::OuiSize] = {0x00, 0x50, 0xf2};
    nn::Bit8 matchingOui2[nn::wlan::MacAddress::OuiSize] = {0x00, 0x07, 0x40};
    uint8_t matchType = 1;

    NN_LOG("\n\n\n\n ***** SCAN RESULT *****");
    NN_LOG("BSS count : %d\n", bssCount);
    NN_LOG("Scan Result Total Length : %d\n", resultReader.GetTotalResultLength());

    for(uint32_t i = 0; i < bssCount; i++)
    {
        nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();

        NN_LOG("-----------------------%d BSS------------------------\n", i + 1);
        NN_LOG("BSS size : %d\n", beacon.GetLength());
        NN_LOG("IE count : %d\n", beacon.GetIeCount());
        NN_LOG("Channel : %d\n", beacon.GetChannel());
        NN_LOG("RSSI : %d[dBm]\n", beacon.GetRssi());
        NN_LOG("LINKLEVEL : %d\n", beacon.GetLinkLevel());
        NN_LOG("Beacon Period : %d[ms]\n", beacon.GetInterval());
        NN_LOG("BSSID : %s\n", beacon.GetBssid().GetString(macStr));
        NN_LOG("SSID : %s\n", beacon.GetSsid().GetHexString(ssidStr));

        // WPA/WPA2
        if( beacon.IsWpaSupported() == true )
        {
            const nn::wlan::WpaInfoElementReader<nn::wlan::WifiInfoElementReader>* pWpa
                = beacon.GetWpaIe();

            NN_LOG("    == WPA: version[%d]\n", pWpa->GetVersion());
            for( uint8_t k = 0; k < pWpa->GetPairwiseCipherSuiteCount(); k++ )
            {
                NN_LOG("       PairCipher(%d) %s\n", k, cipherTypeString[pWpa->GetPairwiseCipherSuite(k)]);
            }
            NN_LOG("       GroupCipher %s\n", cipherTypeString[pWpa->GetGroupCipherSuite()]);
            NN_LOG("       AKM %s\n", akmTypeString[pWpa->GetAuthKeyManagementSuite(0)]);
        }
        if (beacon.IsWpa2Supported())
        {
            const nn::wlan::WpaInfoElementReader<nn::wlan::InfoElementReader>* pWpa
                = beacon.GetWpa2Ie();

            NN_LOG("    == WPA2: version[%d]\n", pWpa->GetVersion());
            for( uint8_t k = 0; k < pWpa->GetPairwiseCipherSuiteCount(); k++ )
            {
                NN_LOG("       PairCipher(%d) %s\n", k, cipherTypeString[pWpa->GetPairwiseCipherSuite(k)]);
            }
            NN_LOG("       GroupCipher %s\n", cipherTypeString[pWpa->GetGroupCipherSuite()]);
            NN_LOG("       AKM %s\n", akmTypeString[pWpa->GetAuthKeyManagementSuite(0)]);
        }

        // VIE matching
        vie = beacon.GetVendorIeWithOui(matchingOui1);
        if( vie != NULL )
        {
            NN_LOG("--- OUI matched!\n");
            NN_LOG("IE element id : %d\n", vie->GetElementId());
            NN_LOG("IE length : %d\n", vie->GetLength());
            NN_LOG("IE OUI : %02X:%02X:%02X\n", vie->GetOui()[0], vie->GetOui()[1], vie->GetOui()[2]);
        }
        else
        {
            NN_LOG("--- OUI not matched...\n");
        }
        vie = beacon.GetVendorIeWithOuiAndType(matchingOui2, matchType);
        if( vie != NULL )
        {
            NN_LOG("--- OUI and type matched!\n");
            NN_LOG("IE element id : %d\n", vie->GetElementId());
            NN_LOG("IE length : %d\n", vie->GetLength());
            NN_LOG("IE OUI : %02X:%02X:%02X\n", vie->GetOui()[0], vie->GetOui()[1], vie->GetOui()[2]);
            NN_LOG("IE type : %d\n", vie->GetVendorData()[0]);
        }
        else
        {
            NN_LOG("--- OUI and type not matched...\n");
        }
    }
}

void PrintConnectionStatus(nn::wlan::ConnectionStatus* pStatus) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pStatus);

    char macStr[nn::wlan::MacAddress::MacStringSize];
    char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];

    NN_LOG("+++ Connection Status +++\n");
    NN_LOG("STATE : %s\n", wlanConnectionStateStr[pStatus->state]);
    NN_LOG("CAUSE : %d\n", pStatus->cause);
    NN_LOG("CHANNEL : %d\n", pStatus->channel);
    NN_LOG("SSID : %s\n", pStatus->ssid.GetHexString(ssidStr));
    NN_LOG("BSSID : %s\n", pStatus->bssid.GetString(macStr));
    NN_LOG("AID : %d\n", pStatus->aid);
    NN_LOG("REASON CODE : %d\n", pStatus->statusReasonCode);
    NN_LOG("CAPABILITY INFO : 0x%04X\n", pStatus->capabilityInfo);
    NN_LOG("BEACON INTERVAL : %d\n", pStatus->beaconInterval);
    NN_LOG("+++++++++++++++++++++++++\n");
}

void PrintClientStatus(nn::wlan::ClientStatus* pStatus) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pStatus);

    char macStr[nn::wlan::MacAddress::MacStringSize];

    NN_LOG("+++++ Client Status +++++\n");
    NN_LOG("STATE : %s\n", wlanConnectionStateStr[pStatus->state]);
    NN_LOG("CAUSE : %d\n", pStatus->cause);
    NN_LOG("BSSID : %s\n", pStatus->clientMacAddress.GetString(macStr));
    NN_LOG("REASON CODE : %d\n", pStatus->statusReasonCode);
    NN_LOG("CAPABILITY INFO : 0x%04X\n", pStatus->capabilityInfo);
    NN_LOG("RSSI  : %d[dbm]\n", pStatus->rssi);
    NN_LOG("+++++++++++++++++++++++++\n");
}

void GenerateTxDataForLocal(uint8_t* pBuf, const nn::wlan::MacAddress& dst, const nn::wlan::MacAddress& src, const uint32_t dataSize) NN_NOEXCEPT
{
    using namespace nn::wlan;

    const uint32_t headerSize = MacAddress::MacAddressSize * 2;

    NN_ASSERT_NOT_NULL(pBuf);
    NN_ASSERT_RANGE(dataSize, headerSize, 1540);

    // Etherヘッダ埋め
    std::memcpy(&pBuf[0], dst.GetMacAddressData(), MacAddress::MacAddressSize);
    std::memcpy(&pBuf[MacAddress::MacAddressSize], src.GetMacAddressData(), MacAddress::MacAddressSize);

    // Payload埋め
    // LDNの仕様に合わせて、VendorSpecificなEtherType(0x88B7)の後は、OUIとProtocolIdentifierを続ける

    // EtherType
    pBuf[headerSize]     = 0x88;
    pBuf[headerSize + 1] = 0xB7;

    // OUI
    pBuf[headerSize + 2] = 0x00;
    pBuf[headerSize + 3] = 0x22;
    pBuf[headerSize + 4] = 0xAA;

    // Protocol Identifier
    pBuf[headerSize + 5] = 0x00;
    pBuf[headerSize + 6] = 0x01;

    // PI以降
    uint32_t j = 0;
    for( uint32_t i = headerSize + 7; i < dataSize; i++ )
    {
        pBuf[i] = static_cast<uint8_t>(j & 0xFF);
        j++;
    }
}


void DataPathRxThreadFunc(void* arg)
{
    nn::Result result;
    uint8_t buf[2048];

    // 受信エントリ作成
    uint32_t rxId;
    uint16_t etherTypes[] = { 0x0800, 0x0806 };
    result = nn::wlan::Socket::CreateRxEntry(&rxId, etherTypes, sizeof(etherTypes) / sizeof(uint16_t), 30); // capacityは適当。とりあえず30個は保持できるようにしておく。
    NN_ASSERT( result.IsSuccess() );
    NN_LOG("WlanTest: Created Rx Entry [%d]\n", rxId);

    while( 1 )
    {
        size_t rxSize = 0;
        result = nn::wlan::Socket::GetFrameRaw(buf, sizeof(buf), &rxSize, rxId); // ブロックされる
        if( result.IsSuccess() )
        {
            NN_LOG("[RX]To %02X:%02X:%02X:%02X:%02X:%02X, From %02X:%02X:%02X:%02X:%02X:%02X [%d]\n",
                    buf[0], buf[1], buf[2], buf[3], buf[4], buf[5],
                    buf[6], buf[7], buf[8], buf[9], buf[10], buf[11], *reinterpret_cast<int32_t*>(&buf[14]));
            //WlanTest::DumpBuffer(buf, rxSize);
        }
        else if( result.GetDescription() == nn::wlan::ResultGetFrameCancelled().GetDescription() ||
                result.GetDescription() == nn::wlan::ResultRxEntryIsMuted().GetDescription() ||
                result.GetDescription() == nn::wlan::ResultNotAuthorized().GetDescription() )
        {
            NN_LOG("GetFrame failed due to link down.\n");
            break;
        }
    }

    // 受信エントリ削除
    result = nn::wlan::Socket::DeleteRxEntry(rxId);
    NN_ASSERT( result.IsSuccess() );
}

void DataPathTxThreadFunc(void* arg)
{
    nn::Result result;

    uint8_t buf[72];
    size_t txSize = 72;
    const uint32_t headerSize = nn::wlan::MacAddress::MacAddressSize * 2;

    nn::wlan::MacAddress ownMac;
    nn::wlan::Socket::GetMacAddress(&ownMac);

    // Etherヘッダ埋め
    std::memcpy(&buf[0], nn::wlan::MacAddress::CreateBroadcastMacAddress().GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
    std::memcpy(&buf[nn::wlan::MacAddress::MacAddressSize], ownMac.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);

    // Payload埋め
    // EtherType
    buf[headerSize]     = 0x08;
    buf[headerSize + 1] = 0x00;

    uint32_t j = 0;
    for( uint32_t i = headerSize + 2; i < txSize; i++ )
    {
        buf[i] = static_cast<uint8_t>(j & 0xFF);
        j++;
    }

    uint32_t cnt = 0;
    while( 1 )
    {
        cnt++;
        std::memcpy(reinterpret_cast<uint32_t*>(&buf[headerSize + 2]), &cnt, sizeof(cnt));
        result = nn::wlan::Socket::PutFrameRaw(buf, txSize);
        if( result.GetDescription() == nn::wlan::ResultNotAuthorized().GetDescription() )
        {
            NN_LOG("Link down.\n");
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    }
}


}
