﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SceneConnecting.h"

#include <cstring>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_Ssid.h>
#include <nn/wlan/wlan_InfraApi.h>
#include <nn/wlan/wlan_Result.h>
#include <nn/bsdsocket/cfg/cfg.h>
#include <nn/bsdsocket/bsdsocket_Result.h>

namespace {
// スレッドスタック
const size_t ThreadStackSize = 8192;
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack[ ThreadStackSize ];
}

nn::os::EventType SceneConnecting::m_event;
nn::os::EventType SceneConnecting::m_cancelEvent;
nn::wlan::ConnectionStatus SceneConnecting::m_connectionStatus;
int SceneConnecting::m_failedCnt;
nn::wlan::WlanIpv4Address SceneConnecting::m_ownIp;
SceneConnecting::SceneConnecting(
        ISceneChanger* changer,
        WirelessData* pDistributor
        )
: BaseScene(changer, pDistributor)
{
    m_animCnt = 0;
    strcpy(m_animStr, ".");
    m_failedCnt = 0;
    memset(&m_ownIp, 0, sizeof(nn::wlan::WlanIpv4Address));
}

void SceneConnecting::Initialize() NN_NOEXCEPT
{
    nn::os::InitializeEvent(&m_event, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&m_cancelEvent, false, nn::os::EventClearMode_AutoClear);
    m_prevTick = nn::os::GetSystemTick();

    nn::Result result = nn::os::CreateThread(
            &m_connectThread, ConnectThreadFunc, m_distributor, g_ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create scan thread." );

    nn::os::StartThread( &m_connectThread );
}

void SceneConnecting::Finalize() NN_NOEXCEPT
{
    nn::os::FinalizeEvent(&m_event);
    nn::os::FinalizeEvent(&m_cancelEvent);
}

void SceneConnecting::Update() NN_NOEXCEPT
{
    Npad npad = Npad::GetInstance();
    npad.UpdatePadState();

    if( npad.IsTrigger(Npad::B) )
    {
        // 接続試行を止める
        nn::wlan::Infra::CancelConnect();
        nn::os::SignalEvent(&m_cancelEvent);
        nn::os::WaitThread(&m_connectThread);
        nn::os::DestroyThread(&m_connectThread);

        switch( m_SceneChanger->GetPrevScene() )
        {
        case eScene_ConnectionParam:
            m_SceneChanger->ChangeScene(eScene_ConnectionParam);
            break;
        case eScene_DirectInput:
            m_SceneChanger->ChangeScene(eScene_DirectInput);
            break;
        default:
            break;
        }
    }
    else if( nn::os::TryWaitEvent(&m_event) )
    {
        // 接続成功
        nn::os::WaitThread(&m_connectThread);
        nn::os::DestroyThread(&m_connectThread);

        m_distributor->SetOwnIp(m_ownIp);
        m_SceneChanger->ChangeScene(eScene_Connected);
    }
}

void SceneConnecting::Draw(
        GraphicsSystem* pGraphicsSystem,
        FontSystem* pFontSystem
        ) NN_NOEXCEPT
{
    BaseScene::Draw(pGraphicsSystem, pFontSystem);

    nn::gfx::util::DebugFontTextWriter&
        textWriter = pFontSystem->GetDebugFontTextWriter();

    const nn::util::Unorm8x4& textColor = Color::White;

    textWriter.SetTextColor( textColor );
    textWriter.SetScale( FONT_SCALE, FONT_SCALE );

    if( (nn::os::GetSystemTick() - m_prevTick).ToTimeSpan().GetMilliSeconds() > 500 )
    {
        m_animCnt++;
        strcat(m_animStr, ".");
        if( m_animCnt > TEXT_ANM_STR_LEN - 2 )
        {
            m_animCnt = 0;
            strcpy(m_animStr, ".");
        }
        m_prevTick = nn::os::GetSystemTick();
    }
    textWriter.SetCursor( MY_INITIAL_X, MY_INITIAL_Y );
    textWriter.Print( "Now connecting %s", m_animStr );

    textWriter.SetCursor( INITIAL_X + FONT_SIZE, INITIAL_Y + (FONT_SIZE * 20) );
    textWriter.Print( "[B]:Cancel" );

    textWriter.SetScale( 1.5, 1.5 );
    textWriter.SetCursor( 800, MY_INITIAL_Y + 40);
    textWriter.Print( "Number of failures: %d", m_failedCnt );

    pGraphicsSystem->BeginDraw();
    pFontSystem->Draw();
    pGraphicsSystem->EndDraw();

    pGraphicsSystem->Synchronize(
        nn::TimeSpan::FromNanoSeconds( 1000 * 1000 * 1000 / FRAME_RATE ) );
}

void SceneConnecting::ConnectThreadFunc(void* arg)
{
    nn::Result result;

    WirelessData* pData = reinterpret_cast<WirelessData*>(arg);

    nn::wlan::Ssid ssid;
    int16_t channel;
    nn::wlan::Security security;
    char str[64];

    pData->GetConnectionParam(&ssid, &channel, &security, str);

    while( NN_STATIC_CONDITION( true ) )
    {
        result = nn::wlan::Infra::Connect(ssid,
                nn::wlan::MacAddress::CreateBroadcastMacAddress(),
                channel, security, false);

        if( result.IsFailure() )
        {
            // 失敗
            m_failedCnt++;
            m_connectionStatus.state = nn::wlan::ConnectionState_Idle;
            break;
        }

        nn::wlan::Infra::GetConnectionStatus(&m_connectionStatus);
        if( m_connectionStatus.state == nn::wlan::ConnectionState_Connected )
        {
            // 接続成功

            // インターフェース Up
            nn::bsdsocket::cfg::IfSettings settings = {};
            settings.mode = nn::bsdsocket::cfg::IfIpAddrMode_Dhcp;
            settings.mtu = 1400;

            result = nn::bsdsocket::cfg::SetIfUp("wl0", &settings);
            if( nn::bsdsocket::ResultBusy().Includes(result) == true )
            {
                nn::bsdsocket::cfg::SetIfDown("wl0");
            }
            else if (result.IsFailure())
            {
                NN_LOG(">result_value:%08X\n", result.GetInnerValueForDebug());
                // IP取得失敗
                // 再接続してリトライ
                m_failedCnt++;
                nn::wlan::Infra::Disconnect();
                if( nn::os::TimedWaitEvent(&m_cancelEvent, nn::TimeSpan::FromSeconds(2)) == true )
                {
                    break;
                }
            }
            else
            {
                // IP取得
                nn::bsdsocket::cfg::IfState state;
                nn::bsdsocket::cfg::GetIfState("wl0", &state);
                m_ownIp.addr[0] = state.addr.S_addr & 0xff;
                m_ownIp.addr[1] = (state.addr.S_addr >> 8) & 0xff;
                m_ownIp.addr[2] = (state.addr.S_addr >> 16) & 0xff;
                m_ownIp.addr[3] = (state.addr.S_addr >> 24) & 0xff;
                NN_LOG(">ip:%d.%d.%d.%d\n",
                        m_ownIp.addr[0], m_ownIp.addr[1], m_ownIp.addr[2], m_ownIp.addr[3]
                );
                break;
            }
        }
        else
        {
            m_failedCnt++;
            // 接続失敗
            // 2秒待って再接続
            if( nn::os::TimedWaitEvent(&m_cancelEvent, nn::TimeSpan::FromSeconds(2)) == true )
            {
                break;
            }
        }
    }

    nn::os::SignalEvent(&m_event);
}
