﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include "Node.h"
#include "Scanner.h"

using namespace std;

namespace
{

uint8_t g_AfBuffer[2000];

}

namespace WlanTest {

#if 0

/*---------------------------------------------------------------------------
　　　　　NoiseMeasure
---------------------------------------------------------------------------*/
NoiseMeasurer::NoiseMeasurer()
{
    m_Param.ccaMode = CS_OR_ED_MODE;
    m_Param.edThreshold = 17;
    m_Param.measureTime = 200;
    m_Param.channelList = 0x1 | 0x1<<(6 - 1) | 0x1<<(11 - 1);
    for(int i=0; i<14; i++)
    {
        m_Result.channelBusyRatio[i] = 0;
        m_Result.numOfChannels = 0;
    }
    m_CompleteMeasureEventHandler = nullptr;
}

NoiseMeasurer::~NoiseMeasurer()
{
}

nn::Result NoiseMeasurer::Measure()
{
    nn::Result result = WlanMeasure();
    if( result.IsFailure() )
    {
        return result;
    }
    if( m_CompleteMeasureEventHandler != nullptr )
    {
        CompleteMeasureEventArgs args;
        args.Result = m_Result;
        (*m_CompleteMeasureEventHandler)(args);
    }

    return result;
}

void NoiseMeasure::Clear()
{
    for(int i=0; i<14; i++)
    {
        m_Result.channelBusyRatio[i] = 0;
    }
}

/*---------------------------------------------------------------------------
　　　　　LocalNoiseMeasurer
---------------------------------------------------------------------------*/

nn::Result LocalNoiseMeasurer::WlanMeasure()
{
    nn::Result result;
    ModeChanger::ToSapMode();

    result = Sap::MeasureChannel( &m_Result, m_Param );
    if (result.IsFailure())
    {
        NN_LOG("MeasureChannel failed\n");
        WlanTest::TPrintResult(result);
    }
    return result;
}

#endif

/*---------------------------------------------------------------------------
　　　　　AccessPoint
---------------------------------------------------------------------------*/
AccessPoint::AccessPoint() :
    m_Bssid(nn::wlan::MacAddress::CreateZeroMacAddress()),
    m_Ssid(nn::wlan::Ssid()),
    m_Channel(0),
    m_Rssi(0),
    m_LinkLevel(nn::wlan::LinkLevel_0),
    m_SecurityConfig(SecurityConfig()),
    m_BeaconInterval(0),
    m_Capability(0x0000)

{
}

AccessPoint::AccessPoint(nn::wlan::MacAddress bssid, nn::wlan::Ssid ssid, int16_t ch, int32_t rssi,
                         nn::wlan::LinkLevel linkLevel, SecurityConfig security, uint16_t interval,
                         nn::Bit16 capability) :
    m_Bssid(bssid),
    m_Ssid(ssid),
    m_Channel(ch),
    m_Rssi(rssi),
    m_LinkLevel(linkLevel),
    m_SecurityConfig(security),
    m_BeaconInterval(interval),
    m_Capability(capability)
{
}

/*---------------------------------------------------------------------------
　　　　　Scanner
---------------------------------------------------------------------------*/
Scanner::Scanner()
{
    m_Param.scanType = nn::wlan::ScanType_Active;
    std::memset(m_Param.channelList, 0, sizeof(m_Param.channelList));
    m_Param.channelCount = 0;
    m_Param.channelScanTime = 110;
    m_Param.homeChannelTime = 0;
    m_Param.ssidList = nullptr;
    m_Param.ssidCount = 0;
    m_Param.bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();

    m_BufferSize = 4096;
    m_pBuffer = new uint8_t[m_BufferSize];
    m_pReader = nullptr;
}

Scanner::~Scanner()
{
    delete[] m_pBuffer;

    if( m_pReader )
    {
        delete m_pReader;
        m_pReader = nullptr;
    }
}

void Scanner::StartScan()
{
    nn::Result result;

    // ここからスキャン
    if( m_pReader != nullptr )
    {
        delete m_pReader;
        m_pReader = nullptr;
    }

    NN_LOG("StartScan()\n");
    result = WlanStartScan();
    if( result.IsFailure() )
    {
        NN_LOG("StartScan failed\n");
        WlanTest::TPrintResult(result);
    }
    m_pReader = new nn::wlan::BeaconScanResultReader(m_pBuffer);

    // ここから解析 ひとまず、重複無く記憶
    nn::wlan::BeaconDescriptionReader bss = m_pReader->GetFirstDescription();
    for(int i=0; i<m_pReader->GetCount(); i++)
    {
        uint64_t source = 0;
        std::memcpy(&source, bss.GetBssid().GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);

        // 古い AP 情報を削除
        AccessPoint* ap = m_ApList[source];
        if( ap != nullptr )
        {
            delete ap;
        }

        // AP 情報を更新
        ap = new AccessPoint(
            bss.GetBssid(),
            bss.GetSsid(),
            bss.GetChannel(),
            bss.GetRssi(),
            bss.GetLinkLevel(),
            SecurityConfig(bss.IsWpaSupported(), bss.IsWpa2Supported(), bss.IsWpsSupported()),
            bss.GetInterval(),
            bss.GetCapabilityInformation());
        m_ApList[source] = ap;

        bss = m_pReader->GetNextDescription();
    }

    // 次に Rssi 順に記憶
    m_ApListSortedByRssi.clear();
    map<uint64_t, AccessPoint*>::iterator it = m_ApList.begin();
    while( it != m_ApList.end() )
    {
        m_ApListSortedByRssi.insert( AccessPointList::value_type( (*it).second->GetRssi(), (*it).second ));
        it++;
    }
}

void Scanner::StopScan()
{
    WlanStopScan();
}

void Scanner::Clear()
{
    m_ApList.clear();
    m_ApListSortedByRssi.clear();

    if( m_pReader != nullptr )
    {
        delete m_pReader;
        m_pReader = nullptr;
    }
}

int16_t Scanner::SearchChannel()
{
    nn::Result result;

    if( m_pReader != nullptr )
    {
        delete m_pReader;
        m_pReader = nullptr;
    }

    NN_LOG("StartScan()\n");
    result = WlanStartScan();
    if( result.IsFailure() )
    {
        NN_LOG("StartScan failed\n");
        WlanTest::TPrintResult(result);
    }
    m_pReader = new nn::wlan::BeaconScanResultReader(m_pBuffer);

    if( result.IsFailure() || GetScanResult().GetCount() == 0 )
    {
        NN_LOG("Channel Not Found.\n");
        return -1;
    }

    return GetFirstDescription().GetChannel();
}

void Scanner::PrintResult()
{
    nn::wlan::BeaconDescriptionReader reader = GetFirstDescription();
    char buf[256];
    char buf2[256];
    NN_UNUSED(buf);
    NN_UNUSED(buf2);
    NN_LOG("Count : %d\n", GetScanResult().GetCount());
    for(int i=0; i<GetScanResult().GetCount(); i++)
    {
        NN_LOG("%s %2dch %s\n", reader.GetBssid().GetString(buf), reader.GetChannel(), reader.GetSsid().GetHexString(buf2));
        reader = GetNextDescription();
    }
}

void Scanner::SetBufferSize(const uint32_t size)
{
    m_BufferSize = size;
    delete m_pBuffer;
    m_pBuffer = nullptr;
    m_pBuffer = new uint8_t[size];
}

void Scanner::SetMaxBuffer()
{
    // ひとまずバッファサイズを0にする
    SetBufferSize(0);

    uint8_t* buf = nullptr;
    size_t base = 0;
    size_t diff = 1;

    // 確保可能な最大バッファサイズを検索
    {
        while( diff != 0 )
        {
            buf = new uint8_t[base + diff];

            if( buf != nullptr )
            {
                delete [] buf;
                base += diff;
                diff *= 2;
            }
            else
            {
                diff /= 2;
            }
        }

        if( buf != nullptr )
        {
            delete [] buf;
        }
    }

    // 最大サイズを確保
    NN_LOG("Max Buffer Size : %d [Byte]\n", base);
    SetBufferSize(base);
}

void Scanner::SetEnableActiveScan(const bool active)
{
    if( active )
    {
        m_Param.scanType = nn::wlan::ScanType_Active;
    }
    else
    {
        m_Param.scanType = nn::wlan::ScanType_Passive;
    }
}

void Scanner::SetScanType(const nn::wlan::ScanType scanType)
{
    m_Param.scanType = scanType;
}

void Scanner::SetChannel(const int16_t chList[nn::wlan::WirelessChannelsCountMax], uint8_t count)
{
    if( 0 == count )
    {
        m_Param.channelCount = count;
    }
    else if( 0 < count && count < nn::wlan::WirelessChannelsCountMax )
    {
        m_Param.channelCount = count;
        std::memcpy(m_Param.channelList, chList, sizeof(m_Param.channelList));
    }
    else
    {
        NN_ASSERT(false);
    }
}

void Scanner::SetScanTime(const int32_t time)
{
    m_Param.channelScanTime = time;
}

void Scanner::SetHomeChannelTime(const int32_t time)
{
    m_Param.homeChannelTime = time;
}

void Scanner::SetSsid(const char* ssid)
{
    SetSsid(nn::wlan::Ssid(ssid));
}

void Scanner::SetSsid(const string& ssid)
{
    SetSsid(nn::wlan::Ssid(ssid.c_str()));
}

void Scanner::SetSsid(const nn::wlan::Ssid& ssid)
{
    SetSsidList(&ssid, 1);
}

void Scanner::SetSsidList(const nn::wlan::Ssid* ssidList, const uint8_t count)
{
    if( ssidList == nullptr )
    {
        m_Param.ssidList = nullptr;
    }
    else if( count > 0 )
    {
        std::memcpy(m_SsidList, ssidList, sizeof(nn::wlan::Ssid) * count);
        m_Param.ssidList = m_SsidList;
    }
    m_Param.ssidCount = count;
}

void Scanner::SetBssid(const nn::wlan::MacAddress bssid)
{
    m_Param.bssid = bssid;
}

uint32_t Scanner::GetApCount()
{
    return m_ApList.size();
}

nn::wlan::BeaconScanResultReader& Scanner::GetScanResult()
{
    return *m_pReader;
}

nn::wlan::BeaconDescriptionReader Scanner::GetFirstDescription()
{
    return GetScanResult().GetFirstDescription();
}

nn::wlan::BeaconDescriptionReader Scanner::GetNextDescription()
{
    return GetScanResult().GetNextDescription();
}

AccessPoint* Scanner::GetFirstAccessPoint()
{
    if( m_ApListSortedByRssi.size() == 0 )
    {
        return nullptr;
    }

    m_Iterator = m_ApListSortedByRssi.rbegin();
    return (*m_Iterator).second;
}

AccessPoint* Scanner::GetNextAccessPoint()
{
    if( m_ApListSortedByRssi.size() == 0 || m_Iterator == m_ApListSortedByRssi.rend() )
    {
        return nullptr;
    }

    m_Iterator++;
    return (*m_Iterator).second;
}

nn::wlan::ScanParameters Scanner::GetScanParam()
{
    return m_Param;
}

/*---------------------------------------------------------------------------
　　　　　InfraScanner
---------------------------------------------------------------------------*/
nn::Result InfraScanner::WlanStartScan()
{
    ModeChanger::ToInfraMode();
    nn::Result result;

    char buf[nn::wlan::MacAddress::MacStringSize];
    NN_LOG(" === Scan Params ===============================\n");
    NN_LOG("   Scan type         : %d\n", m_Param.scanType);
    NN_LOG("   Channel count     : %d\n", m_Param.channelCount);
    NN_LOG("   Channel list      :");
    for(int i=0; i<m_Param.channelCount; ++i)
    {
        NN_LOG(" %d", m_Param.channelList[i]);
    }
    NN_LOG("\n");
    NN_LOG("   Channel scan time : %d\n", m_Param.channelScanTime);
    NN_LOG("   Home channel time : %d\n", m_Param.homeChannelTime);
    NN_LOG("   SSID list         : %s\n", (m_Param.ssidList ? "Not null" : "Null"));
    NN_LOG("   SSID count        : %d\n", m_Param.ssidCount);
    NN_LOG("   BSSID             : %s\n", m_Param.bssid.GetString(buf));
    NN_LOG(" ===============================================\n");

    result = nn::wlan::Infra::StartScan(m_pBuffer, m_BufferSize, m_Param);

    ModeChanger::ToReadyMode();

    return result;
}

nn::Result InfraScanner::WlanStopScan()
{
    nn::Result result = nn::wlan::Infra::StopScan();
    return result;
}

/*---------------------------------------------------------------------------
　　　　　LocalScanner
---------------------------------------------------------------------------*/
LocalScanner::LocalScanner()
{
    m_AddIe = false;
    uint8_t buf[3] = {00, 0x1f, 0x32};
    m_Oui.SetRawData(buf);
    memset(reinterpret_cast<void*>(&m_MatchInfo), 0, sizeof(m_MatchInfo));
}

nn::Result LocalScanner::WlanStartScan()
{
    ModeChanger::ToLocalClientMode();
    nn::Result result;

    if( m_AddIe )
    {
        AddIe(m_Oui.GetRawData(), 3);
    }

    char buf[nn::wlan::MacAddress::MacStringSize];
    NN_LOG(" === Scan Params ===============================\n");
    NN_LOG("   Scan type         : %d\n", m_Param.scanType);
    NN_LOG("   Channel count     : %d\n", m_Param.channelCount);
    NN_LOG("   Channel list      :");
    for(int i=0; i<m_Param.channelCount; ++i)
    {
        NN_LOG(" %d", m_Param.channelList[i]);
    }
    NN_LOG("\n");
    NN_LOG("   Channel scan time : %d\n", m_Param.channelScanTime);
    NN_LOG("   Home channel time : %d\n", m_Param.homeChannelTime);
    NN_LOG("   SSID list         : %s\n", (m_Param.ssidList ? "Not null" : "Null"));
    NN_LOG("   SSID count        : %d\n", m_Param.ssidCount);
    NN_LOG("   BSSID             : %s\n", m_Param.bssid.GetString(buf));
    NN_LOG(" ===============================================\n");

    result = nn::wlan::Local::StartScan(m_pBuffer, m_BufferSize, m_Param);

    if( m_AddIe )
    {
        RemoveIe();
    }

    ModeChanger::ToReadyMode();

    return result;
}

nn::Result LocalScanner::WlanStopScan()
{
    nn::Result result = nn::wlan::Local::StopScan();
    return result;
}

void LocalScanner::EnableAddIe(const bool b)
{
    m_AddIe = b;
}

nn::Result LocalScanner::SetOui(const Oui oui)
{
    nn::Result result;
    m_Oui = oui;
    return result;
}

nn::Result LocalScanner::AddIe(const nn::Bit8 body[], const size_t length)
{
    nn::Result result;
    // result = Sap::AddIeToIesBuffer( MANAGEMENT_FRAME_TYPE_PROBE_REQ, 0, body, length );
    // result = Sap::SetIes( MANAGEMENT_FRAME_TYPE_PROBE_REQ );
    result = nn::wlan::Local::AddIe(&m_BeaconIeIdx, nn::wlan::ManagementFrameType_Beacon, body, length );

    return result;
}

nn::Result LocalScanner::RemoveIe()
{
    nn::Result result;
    // result = Sap::RemoveIeFromIesBuffer( MANAGEMENT_FRAME_TYPE_PROBE_REQ, 4 );
    // result = Sap::SetIes( MANAGEMENT_FRAME_TYPE_PROBE_REQ );
    result = nn::wlan::Local::DeleteIe(m_BeaconIeIdx);

    return result;
}

/*---------------------------------------------------------------------------
　　　　　AfTxScanner
---------------------------------------------------------------------------*/
AfTxScanner::AfTxScanner() :
    m_IsStopped(false),
    m_IsAfEnabled(true),
    m_IsScanEnabled(true),
    m_ScanWait(100)
{
    m_AfParam.payloadSize = 100;
    m_AfParam.count = 3;
    std::memset(m_AfParam.channelList, 0, sizeof(m_AfParam.channelList));
    m_AfParam.channelList[0] = nn::wlan::WirelessChannel_1ch;
    m_AfParam.channelCount = 1;
    m_AfParam.waitTime = nn::TimeSpan::FromMilliSeconds(100);
    m_AfParam.maxRandomWaitTime = 100;
    m_AfParam.interval = nn::TimeSpan::FromMilliSeconds(100);;

    ClearStatistics();
}

void AfTxScanner::GeneratePacket(uint8_t* buf, const size_t bufSize, const uint64_t sequenceNumber, const int16_t channel, const uint32_t length)
{
    NN_ASSERT(bufSize >= length, "bufSize : %u, length : %u", bufSize, length);
    NN_ASSERT(length >= sizeof(WitHeader) + sizeof(ActionFrameSearchInfoCommand));

    WitHeader header;
    GenerateWitHeader(&header, WitCommand_ActionFrameSearchInfo, sizeof(ActionFrameSearchInfoCommand));
    std::memcpy(buf, reinterpret_cast<uint8_t*>(&header), sizeof(header));

    ActionFrameSearchInfoCommand command;
    command.sequenceNumber = sequenceNumber;
    command.channel = channel;

    int64_t tsfTime = 0;
    nn::Result result = GetTsfTime(&tsfTime);
    NN_ASSERT(result.IsSuccess());
    command.tsfTime = tsfTime;

    std::memcpy(buf + sizeof(header), reinterpret_cast<uint8_t*>(&command), sizeof(command));
}

void AfTxScanner::TransmitActionFrame()
{
    std::memset(g_AfBuffer, 0x0, sizeof(g_AfBuffer));
    g_AfBuffer[0] = 0x7f;
    g_AfBuffer[1] = 0x00;
    g_AfBuffer[2] = 0x22;
    g_AfBuffer[3] = 0xAA;
    //g_AfBuffer[4] = nn::wlan::ActionFrameType_Reserved1;
    g_AfBuffer[4] = nn::wlan::ActionFrameType_Beacon;

    // アクションフレーム送信
    auto payloadSize = m_AfParam.payloadSize;
    NN_ASSERT(payloadSize <= sizeof(g_AfBuffer));
    nn::os::Tick stick, etick;
    const nn::wlan::MacAddress BroadcastMacAddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();
    for(int i=0; i<m_AfParam.channelCount && !m_IsStopped; ++i)
    {
        auto channel = m_AfParam.channelList[i];
        for(int j=0; j<m_AfParam.count && !m_IsStopped; ++j)
        {
            nn::Result result;
            GeneratePacket(g_AfBuffer + WlanActionFrameHeaderSize,
                           sizeof(g_AfBuffer) - WlanActionFrameHeaderSize,
                           m_SequenceNumber[channel], channel, payloadSize);
            stick = nn::os::GetSystemTick();
            result = nn::wlan::Local::PutActionFrameOneShot(BroadcastMacAddress,
                                                            reinterpret_cast<uint8_t*>(&g_AfBuffer),
                                                            m_AfParam.payloadSize,
                                                            m_AfParam.channelList[i],
                                                            0);
            etick = nn::os::GetSystemTick();

            if( result.IsSuccess() )
            {
                if( m_Stats.firstSendTime == nn::os::Tick(0) )
                {
                    m_Stats.firstSendTime = stick;
                }

                auto executionTime = (etick - stick).ToTimeSpan().GetMilliSeconds();
                m_Stats.sendExecutionTime.Count(executionTime);
                m_Stats.sendSize += m_AfParam.payloadSize;
                ++m_Stats.sendCount;
                m_Stats.lastSendTime = stick;
                ++m_SequenceNumber[channel];
            }
            else
            {
                ++m_Stats.sendErrorCount;
                NN_LOG("  - failed : PutActionFrameOneShot on %uch\n", m_AfParam.channelList[i]);
            }

            Sleep(m_AfParam.interval);
        }
    }
}

nn::Result AfTxScanner::WlanStartScan()
{
    nn::Result result;
    char buf[nn::wlan::MacAddress::MacStringSize];
    nn::os::Tick stick, etick;

    // NN_LOG(" === Scan Params ===============================\n");
    // NN_LOG("   Scan type         : %d\n", m_Param.scanType);
    // NN_LOG("   Channel count     : %d\n", m_Param.channelCount);
    // NN_LOG("   Channel list      :");
    // for(int i=0; i<m_Param.channelCount; ++i)
    // {
    //     NN_LOG(" %d", m_Param.channelList[i]);
    // }
    // NN_LOG("\n");
    // NN_LOG("   Channel scan time : %d\n", m_Param.channelScanTime);
    // NN_LOG("   Home channel time : %d\n", m_Param.homeChannelTime);
    // NN_LOG("   SSID list         : %s\n", (m_Param.ssidList ? "Not null" : "Null"));
    // NN_LOG("   SSID count        : %d\n", m_Param.ssidCount);
    // NN_LOG("   BSSID             : %s\n", m_Param.bssid.GetString(buf));
    // NN_LOG(" ===============================================\n");

    stick = nn::os::GetSystemTick();
    result = nn::wlan::Local::StartScan(m_pBuffer, m_BufferSize, m_Param);
    etick = nn::os::GetSystemTick();

    if( result.IsSuccess() )
    {
        auto executionTime = (etick - stick).ToTimeSpan().GetMilliSeconds();
        m_Stats.scanExecutionTime.Count(executionTime);
        ++m_Stats.scanCount;
    }
    else
    {
        ++m_Stats.scanErrorCount;
        NN_LOG("  - failed : StartScan()\n");
    }

    return result;
}

nn::Result AfTxScanner::WlanStopScan()
{
    m_IsStopped = true;
    nn::Result result = nn::wlan::Local::StopScan();
    return result;
}

void AfTxScanner::StartScan()
{
    m_IsStopped = false;
    bool isModeChanged = false;
    Mode mode = ModeChanger::GetMode();
    if( mode == READY )
    {
        isModeChanged = true;
        ModeChanger::ToLocalMasterMode();
    }

    // ランダム待ち時間
    if( m_AfParam.maxRandomWaitTime > 0 && !m_IsStopped )
    {
        auto randWaitTime = rand() % m_AfParam.maxRandomWaitTime;
        //NN_LOG(" Random Sleep : %u ms\n", randWaitTime);
        Sleep(nn::TimeSpan::FromMilliSeconds(randWaitTime));
    }

    if( m_IsAfEnabled && !m_IsStopped )
    {
        // 送信までの待ち時間
        auto fixedWaitTime = m_AfParam.waitTime.GetMilliSeconds();
        //NN_LOG(" Sleep before AF Tx : %llu ms\n", fixedWaitTime);
        Sleep(nn::TimeSpan::FromMilliSeconds(fixedWaitTime));

        TransmitActionFrame();
    }

    if( m_IsScanEnabled && !m_IsStopped )
    {
        // スキャンまでの待ち時間
        //NN_LOG(" Sleep before Scan : %llu ms\n", m_ScanWait);
        Sleep(nn::TimeSpan::FromMilliSeconds(m_ScanWait));

        WlanStartScan();
    }

    if( isModeChanged )
    {
        ModeChanger::ToReadyMode();
    }
}

void AfTxScanner::SetAfParam(const ActionFrameParams &afParam)
{
    m_AfParam = afParam;
}

void AfTxScanner::SetIsAfEnabled(const bool isEnabled)
{
    m_IsAfEnabled = isEnabled;
}

void AfTxScanner::SetIsScanEnabled(const bool isEnabled)
{
    m_IsScanEnabled = isEnabled;
}

bool AfTxScanner::IsAfEnabled()
{
    return m_IsAfEnabled;
}

bool AfTxScanner::IsScanEnabled()
{
    return m_IsScanEnabled;
}

SearchStatistics AfTxScanner::GetStatistics()
{
    return m_Stats;
}

void AfTxScanner::ClearStatistics()
{
    m_Stats.Clear();
}

ActionFrameParams AfTxScanner::GetAfParam()
{
    return m_AfParam;
}

}

