﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "LocalNode.h"
#include "Util.h"

namespace {
// OUI + PI (不要であるため適当な値)
const nn::wlan::ReceivedDataMatchInfo DummyMatchInfo = {
    { 0x00, 0x00, 0x00, 0x00, 0x00 }, 5
};

uint8_t g_AfBuffer[2000];
uint8_t g_ScanBuffer[50 * 1024];

}

namespace WlanTest {

/*---------------------------------------------------------------------------
　　　　　LocalNode
---------------------------------------------------------------------------*/

LocalNode::LocalNode() :
    m_EnablePs(false),
    m_Cs(false, 0)
{
}

LocalNode::~LocalNode()
{
    StopReceiveActionFrame();
    StopReceiveCommand();
    StopReceiveData();
}

nn::Result LocalNode::Initialize()
{
    nn::Result result = WlanInitialize();
    if(result.IsFailure())
    {
        NN_LOG(" - failed : LocalNode::WlanInitialize()\n");
        return result;
    }

    m_IsInitialized = true;

    return result;
}

nn::Result LocalNode::Finalize()
{
    nn::Result result = WlanFinalize();
    if(result.IsFailure())
    {
        NN_LOG(" - failed : LocalNode::WlanFinalize()\n");
    }

    m_IsInitialized = false;

    return result;
}

nn::Result LocalNode::WlanInitialize()
{
    nn::Result result = nn::ResultSuccess();
    return result;
}

nn::Result LocalNode::WlanFinalize()
{
    nn::Result result = nn::ResultSuccess();
    return result;
}

nn::Result LocalNode::WlanLocalCreateRxEntry(uint32_t* pRxId,  const uint16_t pProtocols[], const int32_t& count, const int32_t& capacity )
{
    nn::Result result;

    result = nn::wlan::Local::CreateRxEntry(pRxId, pProtocols, count, capacity);
    if( result.IsSuccess() )
    {
        //NN_LOG(" - success : Local::CreateRxEntry() [etherTypes = %04x, rxId = %u]\n", etherTypes[0], *pRxId);
    }
    else if( result.GetDescription() != nn::wlan::ResultDuplicateRxEntry().GetDescription() )
    {
        // 重複していたということは、さっきのをそのままつかえるということ
        // 何もしない。
        NN_LOG(" - failed : Local::CreateRxEntry() (DUPLICATE)\n");
        result = nn::ResultSuccess();
    }
    else
    {
        NN_LOG(" - failed : Local::CreateRxEntry() [etherTypes = %04x, rxId = %u]\n", pProtocols[0], *pRxId);
    }

    return result;
}

nn::Result LocalNode::WlanSocketCreateRxEntry(uint32_t* pRxId,  const uint16_t pProtocols[], const int32_t& count, const int32_t& capacity )
{
    nn::Result result;

    result = nn::wlan::Socket::CreateRxEntry(pRxId, pProtocols, count, capacity);
    if( result.IsSuccess() )
    {
        //NN_LOG(" - success : Socket::CreateRxEntry() [etherTypes = %04x, rxId = %u]\n", etherTypes[0], *pRxId);
    }
    else if( result.GetDescription() != nn::wlan::ResultDuplicateRxEntry().GetDescription() )
    {
        // 重複していたということは、さっきのをそのままつかえるということ
        // 何もしない。
        NN_LOG(" - failed : Socket::CreateRxEntry() (DUPLICATE)\n");
        result = nn::ResultSuccess();
    }
    else
    {
        NN_LOG(" - failed : Socket::CreateRxEntry() [etherTypes = %04x, rxId = %u]\n", pProtocols[0], *pRxId);
        NN_ASSERT(false);
    }

    return result;
}

nn::Result LocalNode::WlanLocalCreateRxEntryForAf(uint32_t* pRxId,  const uint16_t pProtocols[], const int32_t& count, const int32_t& capacity )
{
    nn::Result result;

    result = nn::wlan::Local::CreateRxEntryForActionFrame(pRxId, pProtocols, count, capacity);
    if( result.IsSuccess() )
    {
        //NN_LOG(" - success : Socket::CreateRxEntry() [etherTypes = %04x, rxId = %u]\n", etherTypes[0], *pRxId);
    }
    else if( result.GetDescription() != nn::wlan::ResultDuplicateRxEntry().GetDescription() )
    {
        // 重複していたということは、さっきのをそのままつかえるということ
        // 何もしない。
        NN_LOG(" - failed : Local::CreateRxEntryForActionFrame() (DUPLICATE)\n");
        result = nn::ResultSuccess();
    }
    else
    {
        NN_LOG(" - failed : Local::CreateRxEntryForActionFrame() [etherTypes = %04x, rxId = %u]\n", pProtocols[0], *pRxId);
        NN_ASSERT(false);
    }

    return result;
}

nn::Result LocalNode::WlanDetectCreateRxEntryForAf(uint32_t* pRxId,  const uint16_t pProtocols[], const int32_t& count, const int32_t& capacity )
{
    nn::Result result;

    result = nn::wlan::Detect::CreateRxEntryForActionFrame(pRxId, pProtocols, count, capacity);
    if( result.IsSuccess() )
    {
        //NN_LOG(" - success : Socket::CreateRxEntry() [etherTypes = %04x, rxId = %u]\n", etherTypes[0], *pRxId);
    }
    else if( result.GetDescription() != nn::wlan::ResultDuplicateRxEntry().GetDescription() )
    {
        // 重複していたということは、さっきのをそのままつかえるということ
        // 何もしない。
        NN_LOG(" - failed : Local::CreateRxEntryForActionFrame() (DUPLICATE)\n");
        result = nn::ResultSuccess();
    }
    else
    {
        NN_LOG(" - failed : Local::CreateRxEntryForActionFrame() [etherTypes = %04x, rxId = %u]\n", pProtocols[0], *pRxId);
        NN_ASSERT(false);
    }

    return result;
}

nn::Result LocalNode::WlanLocalDeleteRxEntry(uint32_t* pRxId)
{
    nn::Result result;

    // RxId == 0 の場合は、Rx エントリが未作成とみなし何もしない
    if( *pRxId != 0 )
    {
        result = nn::wlan::Local::DeleteRxEntry(*pRxId);
        if( result.IsFailure() )
        {
            NN_LOG("  - failed : Local::DeleteRxEntry() [rxId = %d, matchInfo = 0x", *pRxId);
            NN_ASSERT(false);
        }
        *pRxId = 0;
    }

    return nn::ResultSuccess();
}

nn::Result LocalNode::WlanSocketDeleteRxEntry(uint32_t* pRxId)
{
    nn::Result result;

    // RxId == 0 の場合は、Rx エントリが未作成とみなし何もしない
    if( *pRxId != 0 )
    {
        result = nn::wlan::Socket::DeleteRxEntry(*pRxId);
        if( result.IsFailure() )
        {
            NN_LOG("  - failed : Socket::DeleteRxEntry() [rxId = %d, matchInfo = 0x", *pRxId);
            NN_ASSERT(false);
        }
        *pRxId = 0;
    }

    return nn::ResultSuccess();
}

nn::Result LocalNode::WlanLocalDeleteRxEntryForAf(uint32_t* pRxId)
{
    nn::Result result;

    // RxId == 0 の場合は、Rx エントリが未作成とみなし何もしない
    if( *pRxId != 0 )
    {
        result = nn::wlan::Local::DeleteRxEntryForActionFrame(*pRxId);
        if( result.IsFailure() )
        {
            NN_LOG("  - failed : Socket::DeleteRxEntryForActionFrame() [rxId = %d, matchInfo = 0x", *pRxId);
            NN_ASSERT(false);
        }
        *pRxId = 0;
    }

    return nn::ResultSuccess();
}

nn::Result LocalNode::WlanDetectDeleteRxEntryForAf(uint32_t* pRxId)
{
    nn::Result result;

    // RxId == 0 の場合は、Rx エントリが未作成とみなし何もしない
    if( *pRxId != 0 )
    {
        result = nn::wlan::Detect::DeleteRxEntryForActionFrame(*pRxId);
        if( result.IsFailure() )
        {
            NN_LOG("  - failed : Socket::DeleteRxEntryForActionFrame() [rxId = %d, matchInfo = 0x", *pRxId);
            NN_ASSERT(false);
        }
        *pRxId = 0;
    }

    return nn::ResultSuccess();
}

nn::Result LocalNode::WlanLocalAddMatchingData(const uint32_t& rxId, const nn::wlan::ReceivedDataMatchInfo& matchInfo)
{
    nn::Result result;

    result = nn::wlan::Local::AddMatchingDataToRxEntry(rxId, matchInfo);
    if( result.IsFailure() )
    {
        NN_LOG(" - failed : Local::AddMatchingDataToRxEntry() [rxId = %d, matchInfo = 0x", rxId);
        for(int i=0; i<matchInfo.matchLength; ++i)
        {
            NN_LOG("%02x ", matchInfo.matchData[i]);
        }
        NN_LOG("]\n");
        NN_ASSERT(false);
    }

    return result;
}

nn::Result LocalNode::WlanLocalRemoveMatchingData(const uint32_t& rxId, const nn::wlan::ReceivedDataMatchInfo& matchInfo)
{
    nn::Result result;

    // RxId == 0 の場合は、Rx エントリが未作成とみなし何もしない
    if( rxId != 0 )
    {
        result = nn::wlan::Local::RemoveMatchingDataFromRxEntry(rxId, matchInfo);
        if( result.IsFailure() )
        {
            NN_LOG("  - failed : Local::RemoveMatchingDataFromRxEntry() [rxId = %d, matchInfo = 0x", rxId);
            for(int i=0; i<matchInfo.matchLength; ++i)
            {
                NN_LOG("%02x ", matchInfo.matchData[i]);
            }
            NN_LOG("]\n");
            NN_ASSERT(false);
        }
    }

    return nn::ResultSuccess();
}

nn::Result LocalNode::WlanGetFrame( uint32_t rxId, uint8_t pOutput[], size_t* pSize, size_t maxSize, int8_t* pRssi )
{
    if( !m_IsInitialized )
    {
        return WlanTest::ResultNotInitialized();
    }

    return nn::wlan::Socket::GetFrameRaw(pOutput, maxSize, pSize, rxId);
}

nn::Result LocalNode::WlanPutFrame( const uint8_t pInput[], size_t size, bool selfCts )
{
    return nn::wlan::Socket::PutFrameRaw(pInput, size);
}

nn::Result LocalNode::WlanGetActionFrame(nn::wlan::MacAddress* pOutSrcMac, uint8_t pOutBuf[], size_t size, size_t* pOutSize, uint32_t rxId, uint16_t* pChannel, int16_t* pRssi)
{
    return nn::wlan::Local::GetActionFrameEx(pOutSrcMac, pOutBuf, size, pOutSize, rxId, pChannel, pRssi);
}

nn::Result LocalNode::WlanPutActionFrame(const nn::wlan::MacAddress& dstMac, const uint8_t* pData, size_t size, int16_t channel, uint32_t dwellTime)
{
    return nn::wlan::Local::PutActionFrameOneShot(dstMac, pData, size, channel, dwellTime);
}

nn::Result LocalNode::WlanGetState(nn::wlan::WlanState *pState)
{
    return nn::wlan::Local::GetState(pState);
}

nn::Result LocalNode::WlanCancelGetFrame(uint32_t rxId)
{
    return nn::wlan::Socket::CancelGetFrame(rxId);
}

nn::Result LocalNode::WlanCancelGetActionFrame(uint32_t rxId)
{
    return nn::wlan::Local::CancelGetActionFrame(rxId);
}

nn::Result LocalNode::CreateRxEntry()
{
    nn::Result result;

    m_Cs.Lock();

    // RxId == 0 の場合に Rx エントリが未作成とみなし、新たに作成する
    if( m_SocketDataRxId == 0 )
    {
        uint16_t etherTypes[1] = { PID_OUI_EXT };
        result = WlanSocketCreateRxEntry(&m_SocketDataRxId, etherTypes, sizeof(etherTypes) / sizeof(etherTypes[0]), 80);
        NN_ASSERT(result.IsSuccess());
    }

    if( m_SocketControlRxId == 0 )
    {
        uint16_t etherTypes[1] = { PID_LOCAL_EXP1 };
        result = WlanSocketCreateRxEntry(&m_SocketControlRxId, etherTypes, sizeof(etherTypes) / sizeof(etherTypes[0]), 30);
        NN_ASSERT(result.IsSuccess());
    }

    if( m_LocalRxId == 0 )
    {
        // 機能的には不要だが、下記を送信可能であるかのフラグ管理として wlan が使用しているため呼ぶ
        // 呼ばないと wlan::Socket の送信もできない。 wlan::Local の送受信は使用しない。

        // 不要であるため他と重複しない適当な値
        uint16_t etherTypes[1] = { PID_LOCAL_EXP2 };
        result = WlanLocalCreateRxEntry(&m_LocalRxId, etherTypes, sizeof(etherTypes) / sizeof(etherTypes[0]), 30);
        NN_ASSERT(result.IsSuccess());

        result = WlanLocalAddMatchingData(m_LocalRxId, DummyMatchInfo);
        NN_ASSERT(result.IsSuccess());
    }

    if( m_AfRxId == 0 )
    {
        // 不要であるため他と重複しない適当な値
        //uint16_t subType = nn::wlan::ActionFrameType_Reserved1;
        uint16_t subType = nn::wlan::ActionFrameType_Beacon;
        // ##fixme capacity を 15 にすると低層で Connect()/StartScan() ではまることがある (SIGLO-70909)
        //result = WlanCreateRxEntryForAf(&m_AfRxId, &subType, 1, nn::wlan::ActionFrameRxCapacityMax);
        result = WlanLocalCreateRxEntryForAf(&m_AfRxId, &subType, 1, 5);
        NN_ASSERT(result.IsSuccess());
    }

    m_Cs.Unlock();

    return nn::ResultSuccess();
}

nn::Result LocalNode::DeleteRxEntry()
{
    if( !m_IsInitialized )
    {
        return WlanTest::ResultNotInitialized();
    }

    m_Cs.Lock();
    NN_ABORT_UNLESS_RESULT_SUCCESS( WlanSocketDeleteRxEntry(&m_SocketDataRxId) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( WlanSocketDeleteRxEntry(&m_SocketControlRxId) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( WlanLocalRemoveMatchingData(m_LocalRxId, DummyMatchInfo) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( WlanLocalDeleteRxEntry(&m_LocalRxId) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( WlanLocalDeleteRxEntryForAf(&m_AfRxId) );
    m_Cs.Unlock();

    return nn::ResultSuccess();
}

void LocalNode::StopMaintainConnection()
{
    if( !m_IsInitialized )
    {
        return;
    }

    if( !IsStop() )
    {
        m_RequestStop = true;

        nn::os::Tick tick = nn::os::GetSystemTick();
        while( !IsStop() )
        {
            int64_t time = (nn::os::GetSystemTick() - tick).ToTimeSpan().GetMilliSeconds();
            if( time > 5000 )
            {
                NN_ASSERT(false, "  - failed : cannot terminate MaintainConnectionThread\n");
            }
            Sleep(nn::TimeSpan::FromMilliSeconds(10));
        }

        Close();
        nn::os::DestroyThread(&m_MaintainConnectionThread);
    }
}

bool LocalNode::IsAfRxReady()
{
    return IsConnected();
}

}
