﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_StringUtil.h>

#include "WlanCommon.h"
#include "LcsSocket.h"
#include "Timer.h"

#include <cassert>
#include <cstring>
#include <cctype>
#include <cstdio>
#include <ctime>
#include <cstdlib>
#include <sstream>
#include <nn/os.h>
#include <nn/os/os_Config.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nnc/os.h>

namespace {

}
namespace WlanTest {

nn::TimeSpan LcsSocket::TxStatistics::UpdateInterval = nn::TimeSpan::FromSeconds(1);
nn::os::Tick LcsSocket::TxStatistics::LastUpdateTime = nn::os::Tick(0);

nn::TimeSpan LcsSocket::RxStatistics::UpdateInterval = nn::TimeSpan::FromSeconds(1);
nn::os::Tick LcsSocket::RxStatistics::LastUpdateTime = nn::os::Tick(0);

/*---------------------------------------------------------------------------
　　　　　LcsSocket
---------------------------------------------------------------------------*/

LcsSocket::LcsSocket() :
    m_TxStatsCs(false, 0),
    m_RxStatsCs(false, 0)
{
    m_IsOpened = false;

    FD_ZERO(&m_ReadFds);
    m_InterfaceName = "wl0";

    m_ListenSocket = -1;
    m_DataSocketList.clear();
    m_DataSocketId = -1;

    m_SocketMode   = SocketModeServer;
    m_SocketType   = SOCKET_TYPE_TCP;
    m_SrcIpAddress = "169.254.1.1";
    m_DstIpAddress = "0.0.0.0";
    m_SubnetMask   = "255.255.255.0";
    m_Dns1         = "0.0.0.0";
    m_Dns2         = "0.0.0.0";
    m_GwIpAddress  = "127.0.0.1";

    m_Port = 50222;
}

LcsSocket::~LcsSocket()
{
}

void LcsSocket::Open()
{
    if( !m_IsOpened )
    {
        m_IsOpened = true;
        NN_ASSERT(m_DataSocketList.size() == 0);
        NN_ASSERT(m_DataSocketId == -1);

        nn::Result result = nn::ResultSuccess();
        if( m_SocketMode == SocketModeServer )
        {
            OpenSocket(&m_ListenSocket);

            result = OpenServer(m_ListenSocket);
            NN_ASSERT(result.IsSuccess());

            SetOption(m_ListenSocket);
        }
        else if( m_SocketMode == SocketModeClient )
        {
            SocketInfo socketInfo;
            OpenSocket(&socketInfo.m_Socket);
            m_DataSocketList.push_back(socketInfo);
            m_DataSocketId = 0;

            result = OpenClient(socketInfo.m_Socket);
            NN_ASSERT(result.IsSuccess());

            // Connect 前に設定が必要
            SetOption(socketInfo.m_Socket);
        }
    }
}

void LcsSocket::Close()
{
    if( m_IsOpened )
    {
        /*
        if( m_ListenSocket >= 0 )
        {
            // struct linger soLinger = { 0 };
            // socklen_t len = sizeof(soLinger);
            // if( nn::socket::GetSockOpt(m_ListenSocket, SOL_SOCKET, SO_LINGER,
            //                            reinterpret_cast<void*>(&soLinger), &len) != 0 )
            // {
            //     NN_LOG("  - failed : GetSockOpt (errno %d)\n", errno);
            //     NN_ASSERT(false);
            // }
            // NN_LOG(" **** onoff : %d, linger : %d, len : %u\n", soLinger.l_onoff, soLinger.l_linger, len);

            // len = sizeof(soLinger);
            // soLinger.l_onoff = 1;
            // soLinger.l_linger = 0;
            // if( nn::socket::SetSockOpt(m_ListenSocket, SOL_SOCKET, SO_LINGER,
            //                            reinterpret_cast<const void*>(&soLinger), len) != 0 )
            // {
            //     NN_LOG("  - failed : SetSockOpt (errno %d)\n", errno);
            //     NN_ASSERT(false);
            // }

            // soLinger.l_onoff = 0;
            // soLinger.l_linger = 0;
            // len = sizeof(soLinger);
            // if( nn::socket::GetSockOpt(m_ListenSocket, SOL_SOCKET, SO_LINGER,
            //                            reinterpret_cast<void*>(&soLinger), &len) != 0 )
            // {
            //     NN_LOG("  - failed : GetSockOpt (errno %d)\n", errno);
            //     NN_ASSERT(false);
            // }
            // NN_LOG(" **** onoff : %d, linger : %d, len : %u\n", soLinger.l_onoff, soLinger.l_linger, len);
        }
        */

        CloseSocket(m_ListenSocket);
        for(int i=0; i<m_DataSocketList.size(); ++i)
        {
            CloseSocket(m_DataSocketList[i].m_Socket);
        }
        m_DataSocketList.clear();
        m_DataSocketId = -1;

        nn::bsdsocket::cfg::SetIfDown(const_cast<char*>(m_InterfaceName.c_str()));
        m_IsOpened = false;
    }
}

void LcsSocket::OpenSocket(int* pSocket)
{
    // ソケットの生成
    CreateSocket(pSocket);

    // notify interface linkup to BSD socket process
    struct ifreq ifr;
    nn::util::Strlcpy(ifr.ifr_name, m_InterfaceName.c_str(), sizeof(ifr.ifr_name));
    ifr.ifr_flags = IFF_UP;
    nn::socket::Ioctl(*pSocket, SIOCSIFFLAGS, &ifr, sizeof(struct ifreq));

    ConfigureInterface();
}

void LcsSocket::CloseSocket(const int socket)
{
    if( socket >= 0 )
    {
        int ret = nn::socket::Shutdown(socket, SHUT_RDWR);
        if( ret < 0 )
        {
            NN_LOG("  - failed : Shutdown (socket %d, errno %d)\n", socket, errno);
        }

        ret = nn::socket::Close(socket);
        if( ret < 0 )
        {
            NN_LOG("  - failed : CloseSocket (socket %d, errno %d)\n", socket, errno);
        }
    }
}

void LcsSocket::CreateSocket(int* pSocket)
{
    if( m_SocketType == SOCKET_TYPE_TCP )
    {
        *pSocket = nn::socket::Socket(PF_INET, SOCK_STREAM, 0);
    }
    else if( m_SocketType == SOCKET_TYPE_UDP )
    {
        *pSocket = nn::socket::Socket(PF_INET, SOCK_DGRAM, 0);
    }
    else
    {
        NN_ASSERT(false);
    }
    NN_ABORT_UNLESS(*pSocket >= 0);
}

void LcsSocket::ConfigureInterface()
{
    nn::bsdsocket::cfg::IfSettings ifcfg;
    memset(&ifcfg, 0, sizeof(ifcfg));

    // static ip
    ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Static;
    ifcfg.mtu  = 1500;

    nn::socket::InetAton(m_SrcIpAddress.c_str(), &ifcfg.u.modeStatic.addr);
    nn::socket::InetAton(m_GwIpAddress.c_str(), &ifcfg.u.modeStatic.gatewayAddr);
    nn::socket::InetAton(m_SubnetMask.c_str(), &ifcfg.u.modeStatic.subnetMask);
    ifcfg.u.modeStatic.broadcastAddr.S_addr =
        (ifcfg.u.modeStatic.addr.S_addr & ifcfg.u.modeStatic.subnetMask.S_addr) |
        ~ifcfg.u.modeStatic.subnetMask.S_addr;
    nn::socket::InetAton(m_Dns1.c_str(), &ifcfg.dnsAddrs[0]);
    nn::socket::InetAton(m_Dns2.c_str(), &ifcfg.dnsAddrs[1]);

    nn::Result result = nn::bsdsocket::cfg::SetIfUp(const_cast<char*>(m_InterfaceName.c_str()), &ifcfg);
    if( result.IsFailure() )
    {
        NN_LOG("failed to configure interface %s - %d:%d\n",
               m_InterfaceName.c_str(),
               result.GetModule(),
               result.GetDescription());
        NN_ASSERT(false);
    }

    nn::bsdsocket::cfg::IfState ifState;
    result = nn::bsdsocket::cfg::GetIfState(m_InterfaceName.c_str(), &ifState);
    if( result.IsFailure() )
    {
        NN_LOG("failed to get interface state %s - %d:%d\n",
               m_InterfaceName.c_str(),
               result.GetModule(),
               result.GetDescription());
        NN_ASSERT(false);
    }

    m_SrcIpAddress = nn::socket::InetNtoa(ifState.addr);

    NN_LOG("  [Src IP Addr : %s:%d]\n", m_SrcIpAddress.c_str(), m_Port);
    NN_LOG("  [Dst Ip Addr : %s:%u]\n", m_DstIpAddress.c_str(), m_Port);
}

nn::Result LcsSocket::OpenServer(const int socket)
{
    if( m_SocketMode != SocketModeServer )
    {
        return WlanTest::ResultFailure();
    }

    int ret;
    nn::Result result;

    // 自身の IP アドレスを設定する
    // ConfigureInterface() で Static IP を有効にしていれば IN_ANYADDR を自身の IP に設定しても
    // Static IP アドレスが有効になるはず
    sockaddr_in sockAddr = { 0 };
    in_addr outAddr = { 0 };
    nn::socket::InetAton(m_SrcIpAddress.c_str(), &outAddr);
    sockAddr.sin_addr = outAddr;
    sockAddr.sin_port = nn::socket::InetHtons(m_Port);
    sockAddr.sin_family = AF_INET;

    // TCP の仕様で Close したソケットをすぐに再利用するためには下記の設定が必要となる (参照:SIGLO-58452)
    int enabled = true;
    NN_LOG("  - test (errno %d)\n", errno);
    if( nn::socket::SetSockOpt(socket, SOL_SOCKET, SO_REUSEADDR, reinterpret_cast<const void*>(&enabled), sizeof(enabled)) != 0 )
    {
        NN_LOG("  - failed : SetSockOpt (errno %d)\n", errno);
        NN_ASSERT(false);
    }

    // ソケットへの接続
    if( nn::socket::Bind(socket, reinterpret_cast<sockaddr *>(&sockAddr), sizeof(sockAddr)) != 0 )
    {
        NN_LOG("  - failed : Bind (errno %d, %s:%u)\n", errno, m_SrcIpAddress.c_str(), m_Port);
        NN_ASSERT(false);
    }

    // パケットが届いてるか確認するために fd に結び付ける
    // fd_set の初期化
    FD_ZERO(&m_ReadFds);

    // 読み込むソケットの登録
    FD_SET(socket, &m_ReadFds);

    sockaddr_in tmpSockAddr = { 0 };
    socklen_t   saLen = sizeof(tmpSockAddr);
    if( nn::socket::GetSockName(socket, reinterpret_cast<sockaddr *>(&tmpSockAddr), &saLen) != 0 )
    {
        NN_LOG("server:  GetSockName failed (error %d)\n", nn::socket::GetLastErrno());
        return WlanTest::ResultFailure();
    }

    if( m_SocketType == SOCKET_TYPE_TCP && m_SocketMode == SocketModeServer )
    {
        // 接続可能なクライアント数に加えて Master の数も含めるので +1
        NN_ABORT_UNLESS_EQUAL(nn::socket::Listen(socket, ConnectableNodeCountMax + 1), 0);
        NN_ABORT_UNLESS_EQUAL(nn::socket::Fcntl(socket, F_SETFL, O_NONBLOCK), 0);
    }

    return nn::ResultSuccess();
}

nn::Result LcsSocket::OpenClient(const int socket)
{
    if( m_SocketMode != SocketModeClient )
    {
        return WlanTest::ResultFailure();
    }

    // 終了時に待たされることがないように NONBLOCK の設定を行う (参照:SIGLO44648)
    NN_ABORT_UNLESS_EQUAL(nn::socket::Fcntl(socket, F_SETFL, O_NONBLOCK), 0);

    return nn::ResultSuccess();
}

void LcsSocket::SetOption(const int socket)
{
    int sendBufSize = m_TxBufferSize;
    int recvBufSize = m_RxBufferSize;
    int noDelayFlag = (m_IsNegleEnabled ? 0 : 1);

    int tcpWin = recvBufSize;

    int ret = nn::socket::SetSockOpt(socket, SOL_SOCKET, SO_SNDBUF, &sendBufSize, sizeof(sendBufSize));
    if( ret != 0 )
    {
        NN_LOG("  - failed : SetSockOpt for SO_SNDBUF (errno %d)\n", errno);
        NN_ASSERT(false);
    }

    ret = nn::socket::SetSockOpt(socket, SOL_SOCKET, SO_RCVBUF, &recvBufSize, sizeof(recvBufSize));
    if( ret != 0 )
    {
        NN_LOG("  - failed : SetSockOpt for SO_RCVBUF (errno %d)\n", errno);
        NN_ASSERT(false);
    }

    ret = nn::socket::SetSockOpt(socket, SOL_TCP, TCP_NODELAY, &noDelayFlag, sizeof(noDelayFlag));
    if( ret != 0 )
    {
        NN_LOG("  - failed : SetSockOpt for TCP_NODELAY (errno %d)\n", errno);
        NN_ASSERT(false);
    }


    socklen_t size = 0;
    size = sizeof(sendBufSize);
    ret = nn::socket::GetSockOpt(socket, SOL_SOCKET, SO_SNDBUF, &sendBufSize, &size);
    if( ret != 0 )
    {
        NN_LOG("  - failed : GetSockOpt for SO_SNDBUF (errno %d)\n", errno);
        NN_ASSERT(false);
    }

    size = sizeof(recvBufSize);
    ret = nn::socket::GetSockOpt(socket, SOL_SOCKET, SO_RCVBUF, &recvBufSize, &size);
    if( ret != 0 )
    {
        NN_LOG("  - failed : GetSockOpt for SO_RCVBUF (errno %d)\n", errno);
        NN_ASSERT(false);
    }

    size = sizeof(noDelayFlag);
    ret = nn::socket::GetSockOpt(socket, SOL_TCP, TCP_NODELAY, &noDelayFlag, &size);
    if( ret != 0 )
    {
        NN_LOG("  - failed : SetSockOpt for TCP_NODELAY (errno %d)\n", errno);
        NN_ASSERT(false);
    }

    NN_LOG(" === Socket Option ====================\n");
    NN_LOG("    Socket          : %d\n", socket);
    NN_LOG("    Tx Buffer Size  : %d byte\n", sendBufSize);
    NN_LOG("    Rx Buffer Size  : %d byte\n", recvBufSize);
    NN_LOG("    No Delay        : %s\n", (noDelayFlag ? "Enabled" : "Disabled"));
    NN_LOG(" ======================================\n");
}

bool LcsSocket::Accept()
{
    if( m_SocketType != SOCKET_TYPE_TCP || m_SocketMode != SocketModeServer )
    {
        return false;
    }

    sockaddr_in sockAddr = { 0 };
    socklen_t sockLen = sizeof(sockaddr_in);

    int socket = nn::socket::Accept(m_ListenSocket, reinterpret_cast<sockaddr*>(&sockAddr), &sockLen);
    if( socket >= 0 )
    {
        string address = nn::socket::InetNtoa(sockAddr.sin_addr);
        // NN_LOG(" ****************************** Accepted!! (socket : %d, IP Address : %s)\n", socket, address.c_str());

        // 同じ情報があれば過去の統計情報などをコピーして、古い情報を削除する
        SocketInfo socketInfo;
        RemoveByAddress(address, &socketInfo);

        socketInfo.m_Socket = socket;
        socketInfo.m_IsAccepted = true;
        socketInfo.m_IpAddressStr = address;
        m_DataSocketList.push_back(socketInfo);
        if( m_DataSocketId < 0 )
        {
            m_DataSocketId = 0;
        }
        //SetOption(socket);

        return true;
    }
    else
    {
        if( errno != 11 )
        {
            NN_LOG("  - failed : Accept(errno %d)\n", errno);
        }

        return false;
    }
}

int LcsSocket::Connect()
{
    if( m_SocketType != SOCKET_TYPE_TCP || m_SocketMode != SocketModeClient )
    {
        return -1;
    }

    sockaddr_in sockAddr = { 0 };
    in_addr outAddr = { 0 };
    nn::socket::InetAton(m_DstIpAddress.c_str(), &outAddr);
    sockAddr.sin_addr = outAddr;
    sockAddr.sin_port = nn::socket::InetHtons(m_Port);
    sockAddr.sin_family = AF_INET;

    int socket = GetDataSocket();
    int ret = nn::socket::Connect(socket, reinterpret_cast<sockaddr*>(&sockAddr), sizeof(sockAddr));
    if( ret < 0 )
    {
        // 接続済み(EISCONN(106))であれば成功を返す (接続中であれば EINPROGRESS(115) で失敗を返す)
        if( errno == EISCONN )
        {
            string address = nn::socket::InetNtoa(sockAddr.sin_addr);
            // NN_LOG(" ****************************** Connected!! (socket : %d, IP Address : %s)\n", socket, address.c_str());
            m_DataSocketList[m_DataSocketId].m_IpAddressStr = address;
            m_DataSocketList[m_DataSocketId].m_IsAccepted = true;
            //SetOption(socket);

            return 0;
        }
        else
        {
            NN_LOG("  - failed : nn::socket::Connect() [socket : %d, SrcIpAddress : %s, errno : %d]\n", socket, m_SrcIpAddress.c_str(), errno);
        }
    }

    return ret;
}

bool LcsSocket::Remove(const string& ipAddress)
{
    if( m_SocketType != SOCKET_TYPE_TCP || m_SocketMode != SocketModeServer )
    {
        return false;
    }

    int target = -1;
    int i = 0;
    for(const auto& socketInfo : m_DataSocketList)
    {
        if( socketInfo.m_IpAddressStr == ipAddress )
        {
            target = i;
            break;
        }
        ++i;
    }

    if( target >= 0 )
    {
        CloseSocket(m_DataSocketList[target].m_Socket);
        m_DataSocketList.erase(m_DataSocketList.begin() + target);
    }

    return target >= 0;
}

nn::Result LcsSocket::Send(const uint8_t data[], size_t size, size_t* pSentSize)
{
    ssize_t sentSize = -1;
    int socket = GetDataSocket();
    sentSize = nn::socket::Send(socket, data, size, 0x0);
    UpdateTxStats(socket, sentSize);

    if( sentSize >= 0 )
    {
        //NN_LOG("  - Send() Success!! %d byte\n", sentSize);
        *pSentSize = sentSize;
        return nn::ResultSuccess();
    }
    else
    {
        //NN_LOG("  - failed : Send(), error : %d, %d\n", errno, sentSize);
        return WlanTest::ResultFailure();
    }
}

nn::Result LcsSocket::Receive(uint8_t pOutput[], size_t* pSize, const size_t maxSize)
{
    nn::Result result;
    sockaddr_in fromAddr = { 0 };
    result = SocketRecv(pOutput, pSize, &fromAddr, maxSize, 0);

    return result;
}

nn::Result LcsSocket::SocketRecv( uint8_t pOutput[], size_t* pSize, sockaddr_in *pFromAddr, const size_t maxSize, int flags)
{
#if 0

    int socket = -1;
    ssize_t peekSize = -1;
    sockaddr_in sockAddr = { 0 };
    socklen_t saLen = sizeof(sockAddr);

    for(int i=0; i<m_DataSocketList.size(); ++i)
    {
        socket = GetDataSocket(i);
        peekSize = nn::socket::Recv(socket, pOutput, maxSize, MSG_PEEK|MSG_DONTWAIT);

        if( peekSize > 0 )
        {
            break;
        }

        if( errno == ENOTCONN || errno == ECONNRESET )
        {
            //NN_LOG("  - failed : disconnected socket : %d, errno : %d\n", socket, errno);
        }

        if( errno != 11 )
        {
            //NN_LOG("  ****** Recv(), socket : %d, size : %llu, error : %d\n", socket, peekSize, errno);
        }
    }

    if( peekSize > 0 )
    {
        ssize_t recvSize = 0;
        ssize_t totalRecvSize = 0;
        while( true )
        {
            recvSize = nn::socket::Recv(socket, pOutput, maxSize, MSG_DONTWAIT);
            if( recvSize > 0 )
            {
                UpdateRxStats(socket, recvSize);
                totalRecvSize += recvSize;
            }
            else
            {
                break;
            }
        }

        // recvSize >= 0 でも、タイミングによっては recvSize と retValue がことなったり、終了時に 0 以下になったりすることがある
        //NN_ASSERT( recvSize == retValue );
        if( totalRecvSize <= 0 )
        {
             return WlanTest::ResultFailure();
        }

        *pSize = totalRecvSize;
        *pFromAddr = sockAddr;

        return nn::ResultSuccess();
    }
    else
    {
        //NN_LOG("  - failed : Recv(), error : %d\n", recvSize);
        return WlanTest::ResultFailure();
    }

#else

    int socket = -1;
    ssize_t recvSize = -1;
    sockaddr_in sockAddr = { 0 };
    socklen_t saLen = sizeof(sockAddr);

    *pSize = 0;
    for(int i=0; i<m_DataSocketList.size(); ++i)
    {
        socket = GetDataSocket(i);

        bool rtn = false;
        while( (recvSize = nn::socket::Recv(socket, pOutput, maxSize, MSG_DONTWAIT)) > 0 )
        {
            UpdateRxStats(socket, recvSize);
            *pSize += recvSize;
            *pFromAddr = sockAddr;
        }

        if( *pSize > 0 )
        {
            return nn::ResultSuccess();
        }
    }

    return WlanTest::ResultFailure();

#endif
}

bool LcsSocket::WaitReceive(nn::TimeSpan timeSpan)
{
    int64_t sec = timeSpan.GetSeconds();
    int64_t us  = timeSpan.GetMicroSeconds();

    if( us != 0 )
    {
        us -= nn::TimeSpan::FromSeconds(sec).GetMicroSeconds();
    }

    timeval tv;
    tv.tv_sec  = sec;
    tv.tv_usec = us;

    fd_set fds;
    // 読み込み用 fd_set の初期化
    memcpy(&fds, &m_ReadFds, sizeof(fd_set));

    // fdsに設定されたソケットが読み込み可能になるまでタイムアウトつきで待機する
    int n = nn::socket::Select(m_DataSocketList.size(), &fds, NULL, NULL, &tv);

    // タイムアウトの場合は 0 となる
    if( n == 0 )
    {
        return false;
    }

    return FD_ISSET(m_ListenSocket, &fds);
}

bool LcsSocket::RemoveByAddress(const string& address, SocketInfo* pSocketInfo)
{
    bool ret = false;

    for(auto it=m_DataSocketList.begin(); it!=m_DataSocketList.end(); )
    {
        if( it->m_IpAddressStr == address )
        {
            if( pSocketInfo )
            {
                *pSocketInfo = *it;
            }
            it = m_DataSocketList.erase(it);
            return true;
        }
        else
        {
            ++it;
        }
    }

    return false;
}

void LcsSocket::UpdateTxStats(const int32_t& socket, const ssize_t& txSize)
{
    nn::os::Tick tick = nn::os::GetSystemTick();

    m_TxStatsCs.Lock();

    for(auto& dataSocket : m_DataSocketList)
    {
        if( dataSocket.m_Socket == socket )
        {
            TxStatistics& stats = dataSocket.m_TxStats;

            if( stats.FirstTxTime == nn::os::Tick(0) )
            {
                stats.FirstTxTime = tick;
            }
            stats.LastTxTime = tick;

            if( txSize > 0 )
            {
                stats.Throughput.Count(txSize);
                stats.TxDataSize += txSize;
                ++stats.TxSuccess;

                stats.WorkThroughput.Count(txSize);
                stats.WorkTxDataSize += txSize;
                ++stats.WorkTxSuccess;
            }
            else
            {
                ++stats.TxError;

                ++stats.WorkTxError;
            }

            break;
        }
    }

    m_TxStatsCs.Unlock();
}

void LcsSocket::UpdateLastTxStats()
{
    nn::os::Tick tick = nn::os::GetSystemTick();

    m_TxStatsCs.Lock();

    if( (tick - TxStatistics::LastUpdateTime).ToTimeSpan() >= TxStatistics::UpdateInterval )
    {
        for(auto& socket : m_DataSocketList)
        {
            auto& stats = socket.m_TxStats;
            if( stats.LastTxTime != nn::os::Tick(0) )
            {
                stats.LastTxDataSize = stats.WorkTxDataSize;
                stats.LastTxSuccess = stats.WorkTxSuccess;
                stats.LastTxError = stats.WorkTxError;
                stats.LastThroughput = stats.WorkThroughput;

                stats.WorkTxDataSize = 0;
                stats.WorkTxSuccess = 0;
                stats.WorkTxError = 0;
                stats.WorkThroughput.Clear();
            }
        }
        TxStatistics::LastUpdateTime = tick;
    }
    m_TxStatsCs.Unlock();
}

void LcsSocket::UpdateRxStats(const int32_t& socket, const ssize_t& rxSize)
{
    nn::os::Tick tick = nn::os::GetSystemTick();
    for(auto& dataSocket : m_DataSocketList)
    {
        if( dataSocket.m_Socket == socket )
        {
            RxStatistics& stats = dataSocket.m_RxStats;

            if( stats.FirstRxTime == nn::os::Tick(0) )
            {
                stats.FirstRxTime = tick;
            }
            stats.LastRxTime = tick;

            if( rxSize > 0 )
            {
                stats.Throughput.Count(rxSize);
                stats.RxDataSize += rxSize;
                ++stats.RxSuccess;

                stats.WorkThroughput.Count(rxSize);
                stats.WorkRxDataSize += rxSize;
                ++stats.WorkRxSuccess;
            }
            else
            {
                ++stats.RxError;

                ++stats.WorkRxError;
            }

            break;
        }
    }
}

void LcsSocket::UpdateLastRxStats()
{
    nn::os::Tick tick = nn::os::GetSystemTick();

    m_RxStatsCs.Lock();

    if( (tick - RxStatistics::LastUpdateTime).ToTimeSpan() >= RxStatistics::UpdateInterval )
    {
        for(auto& socket : m_DataSocketList)
        {
            auto& stats = socket.m_RxStats;
            if( stats.LastRxTime != nn::os::Tick(0) )
            {
                stats.LastRxDataSize = stats.WorkRxDataSize;
                stats.LastRxSuccess = stats.WorkRxSuccess;
                stats.LastRxError = stats.WorkRxError;
                stats.LastThroughput = stats.WorkThroughput;

                stats.WorkRxDataSize = 0;
                stats.WorkRxSuccess = 0;
                stats.WorkRxError = 0;
                stats.WorkThroughput.Clear();
            }
        }
        RxStatistics::LastUpdateTime = tick;
    }
    m_RxStatsCs.Unlock();
}

bool LcsSocket::IsListIdValid()
{
    return IsListIdValid(m_DataSocketId);
}

bool LcsSocket::IsListIdValid(const int& id)
{
    return 0 <= id && id < m_DataSocketList.size();
}

string LcsSocket::GenerateIpAddress(uint8_t ip1, uint8_t ip2, uint8_t ip3, uint8_t ip4)
{
    ostringstream oss;
    oss << static_cast<int>(ip1) << '.' << static_cast<int>(ip2) << '.' << static_cast<int>(ip3) << '.' << static_cast<int>(ip4);
    return oss.str();
}

string LcsSocket::GenerateIpAddress(uint8_t ip[4])
{
    return GenerateIpAddress(ip[0], ip[1], ip[2], ip[3]);
}

}
