﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Common.h"
#include "CardWriter.h"
#include "Util.h"

namespace {

static uint8_t s_Buffer[1024 * 10];

}

namespace WlanTest {

/*---------------------------------------------------------------------------
　　　　　CardWriter
---------------------------------------------------------------------------*/

CardWriter::CardWriter() : m_Cs(false, 0)
{
    m_IsInitialized = false;
    m_BufferedSize = 0;
    memset(s_Buffer, 0, sizeof(s_Buffer));
}

CardWriter::~CardWriter()
{
    if( m_IsInitialized )
    {
        nn::fs::CloseFile(m_Handler);
    }
}

bool CardWriter::WriteCore(const char* str, const int64_t strSize)
{
    nn::Result result;
    int64_t size;

    result = nn::fs::GetFileSize(&size, m_Handler);
    if(result.IsFailure())
    {
        NN_LOG("  - failed : cannot get file size\n");
        return false;
    }

    result = nn::fs::WriteFile(m_Handler, size, str, strSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
    if(result.IsFailure())
    {
        NN_LOG("  - failed : cannot write to SD card\n");
        return false;
    }

    return true;
}

bool CardWriter::Write(const string& str)
{
    if( !m_IsInitialized )
    {
        NN_LOG("  - failed : CardWriter is not initialized\n");
        return false;
    }

    int64_t writeSize = str.size();
    m_Cs.Lock();

    if( m_BufferedSize + writeSize >= sizeof(s_Buffer) )
    {
        // 一定量に達したらバッファを書き出す
        WriteCore(reinterpret_cast<char*>(s_Buffer), m_BufferedSize);
        m_BufferedSize = 0;
    }

    if( writeSize > sizeof(s_Buffer) )
    {
        // 書き出すサイズが大きい場合は単独で書き出す
        WriteCore(str.c_str(), writeSize);
    }
    else
    {
        // 一定量になるまでバッファにプールする
        memcpy(s_Buffer + m_BufferedSize, str.c_str(), writeSize);
        m_BufferedSize += writeSize;
    }

    m_Cs.Unlock();

    return true;
}

bool CardWriter::Flush()
{
    if( !m_IsInitialized )
    {
        NN_LOG("  - failed : CardWriter is not initialized\n");
        return false;
    }

    if( m_BufferedSize == 0 )
    {
        return true;
    }

    m_Cs.Lock();

    // 強制的に書き出す
    WriteCore(reinterpret_cast<char*>(s_Buffer), m_BufferedSize);
    m_BufferedSize = 0;

    m_Cs.Unlock();

    return true;
}

bool CardWriter::Initialize(const string& dirName, const string& fileName)
{
    nn::Result result;

    if( m_IsInitialized )
    {
        return true;
    }

    result = nn::fs::MountSdCardForDebug("sd");
    if( result.IsFailure() )
    {
        NN_LOG("  - fialed : cannot mount SD card.\n");
        return false;
    }

    m_DirPath = "sd:/" + dirName;
    if( m_DirPath.back() != '/' )
    {
        m_DirPath += '/';
    }

    // ディレクトリが存在しない場合は作成する
    if( dirName.size() > 0 )
    {
        nn::fs::DirectoryEntryType directoryEntryType;
        result = nn::fs::GetEntryType(&directoryEntryType, m_DirPath.c_str());
        if( nn::fs::ResultPathNotFound().Includes(result) )
        {
            NN_LOG("Create \"%s\"\n", m_DirPath.c_str());
            result = nn::fs::CreateDirectory(m_DirPath.c_str());
            NN_ASSERT(result.IsSuccess());
        }
    }

    if( fileName.size() == 0 )
    {
        // ファイル名の指定がなければ現在の日付にする
        m_FileName = GetTodayStr() + ".txt";
    }
    else
    {
        m_FileName = fileName + ".txt";
    }

    // 書き込むファイルを作成
    string path = m_DirPath + m_FileName;
    nn::fs::DirectoryEntryType directoryEntryType;
    result = nn::fs::GetEntryType(&directoryEntryType, path.c_str());
    if( nn::fs::ResultPathNotFound().Includes(result) )
    {
        NN_LOG("Create \"%s\"\n", path.c_str());
        result = nn::fs::CreateFile(path.c_str(), 0);
        NN_ASSERT(result.IsSuccess());
    }
    else
    {
        NN_ASSERT(false);
        return false;
    }

    NN_LOG("Open \"%s\"\n", path.c_str());
    result = nn::fs::OpenFile(&m_Handler, path.c_str(), nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
    if( !result.IsSuccess() )
    {
        NN_ASSERT(false);
        return false;
    }

    m_IsInitialized = true;

    return true;
}

void CardWriter::Finalize()
{
    if( !m_IsInitialized )
    {
        return;
    }

    Flush();

    m_DirPath = "";
    m_FileName = "";
    nn::fs::CloseFile(m_Handler);
    nn::fs::Unmount("sd");
    m_IsInitialized = false;
}

}    // WlanTest
