﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/irsensor.h>
#include <nn/btm/btm.h>
#include <nn/btm/btm_Result.h>
#include <nn/os.h>

#include "WlanTestTypes.h"
#include "Counter.h"

namespace WlanTest {

int ToDebugPadButton(Button button);
int ToNPadButton(Button button);

struct BluetoothPlr
{
    nn::bluetooth::BluetoothAddress address;
    nn::os::Tick updateTime;
    double plr;
    bool isConnected;
};

class BluetoothPlrStatistics
{
private :
protected :

    static const uint16_t RECORD_COUNT = 300;

    uint32_t m_LastSamplingNumber;
    std::deque<BluetoothPlr> m_Plot[REMOTE_COUNT_MAX];

    void GetHidDeviceCondition(nn::btm::HidDeviceCondition** pHdc, const nn::bluetooth::BluetoothAddress& addr, nn::btm::DeviceConditionList* list);

public :

    static const uint32_t UPDATE_INTERVAL_MS = 1500; // bluetooth で PLR を更新する間隔
    static constexpr double INVALID_PLR = -1.0f; // 適当な負の値

    typedef std::deque<BluetoothPlr> StatisticsList[REMOTE_COUNT_MAX];

    static BluetoothPlrStatistics& GetInstance();

    void Update();
    const StatisticsList& GetStatistics();
};

class BluetoothChannelMap
{
    private:
    protected:
        uint8_t m_ChannelMap[10];
        uint8_t m_ChannelCount;
    public:
        static BluetoothChannelMap& GetInstance();

        void Update();
        const uint8_t* GetChannelMap();
        uint8_t GetChannelCount();
};

class DebugPad
{
/*---------------------------------------------------------------------------
　　　　　定数・型定義など
---------------------------------------------------------------------------*/
public:

    struct PadState
    {
        nn::hid::DebugPadButtonSet trigger;
        nn::hid::DebugPadButtonSet hold;
        nn::hid::DebugPadButtonSet release;
    };

private:

/*---------------------------------------------------------------------------
　　　　　静的変数
---------------------------------------------------------------------------*/
public:
private:

    static bool m_IsEnabled;

/*---------------------------------------------------------------------------
　　　　　静的メソッド
---------------------------------------------------------------------------*/
public:

    static DebugPad& GetInstance();
    static void EnablePad();
    static void DisablePad();

private:

/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:

    nn::TimeSpan m_HoldDetectionDelay;
    nn::TimeSpan m_HoldDetectionInterval;

protected:

    PadState m_CurrentState;
    nn::TimeSpan m_PressTime;
    nn::TimeSpan m_PreviousDetectionTime;

private:
/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual void UpdatePadState();

    bool IsTrigger(Button button);
    bool IsHold(Button button);
    bool IsRelease(Button button);
    void Clear();

protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:

    virtual ~DebugPad();

protected:

    DebugPad();

private:
/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
};


class NPad
{
/*---------------------------------------------------------------------------
　　　　　定数・型定義など
---------------------------------------------------------------------------*/
public:

    struct PadState
    {
        nn::hid::NpadButtonSet trigger;
        nn::hid::NpadButtonSet hold;
        nn::hid::NpadButtonSet release;
    };

    struct PadInfo
    {
        bool                         isIrEnabled;
        nn::hid::NpadIdType          npadId;
        nn::irsensor::IrCameraHandle irHandle;
        nn::hid::SixAxisSensorHandle leftHandle;
        nn::hid::SixAxisSensorHandle rightHandle;
        nn::hid::SixAxisSensorHandle fullKeyHandle;
        SamplingCountRecorder        recorder;
    };

protected:
private:

/*---------------------------------------------------------------------------
　　　　　静的変数
---------------------------------------------------------------------------*/
public:
protected:

    static const uint8_t NPAD_ID_TYPE_COUNT = 9;
    static const nn::hid::NpadIdType m_NpadIds[NPAD_ID_TYPE_COUNT];
    static PadInfo m_PadInfo[REMOTE_COUNT_MAX];

private:

    static bool m_IsEnabled;
    static bool m_IsIrConfigurationEnabled;

/*---------------------------------------------------------------------------
　　　　　静的メソッド
---------------------------------------------------------------------------*/
public:

    static NPad& GetInstance();
    static void EnablePad();
    static void DisablePad();
    static void GetPadInfo(const PadInfo* stats[REMOTE_COUNT_MAX], uint8_t* pCount);
    static void ClearStatistics();
    static void Finalize();

private:

/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:

    nn::TimeSpan m_HoldDetectionDelay;
    nn::TimeSpan m_HoldDetectionInterval;

protected:

    PadState m_CurrentState;
    nn::TimeSpan m_PressTime;
    nn::TimeSpan m_PreviousDetectionTime;

private:
/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    void UpdatePadState();

    bool IsTrigger(Button button);
    bool IsHold(Button button);
    bool IsRelease(Button button);
    void Clear();

    void EnableIrConfiguration(const bool& isEnabled);
    bool SetBtmWlanMode(const nn::btm::WlanMode& wlanMode);
    bool SetBtmBluetoothMode(const nn::btm::BluetoothMode& bluetoothMode);

    void DisconnectNpad(const nn::hid::NpadIdType& id);
    void DisconnectAllNpads();

protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:

    virtual ~NPad();

protected:

    NPad();

private:
/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
};

class Pad
{
/*---------------------------------------------------------------------------
　　　　　定数・型定義など
---------------------------------------------------------------------------*/
public:
private:

/*---------------------------------------------------------------------------
　　　　　静的変数
---------------------------------------------------------------------------*/
public:
protected:
private:

/*---------------------------------------------------------------------------
　　　　　静的メソッド
---------------------------------------------------------------------------*/
public:

    static Pad& GetInstance();
    static void EnablePad();
    static void DisablePad();

private:

/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:

    static DebugPad* m_pDebugPad;
    static NPad* m_pNPad;

private:
/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual void UpdatePadState();

    bool IsTrigger(Button button);
    bool IsHold(Button button);
    bool IsRelease(Button button);
    void Clear();

    void Disconnect();

protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:

    virtual ~Pad();

protected:

    Pad();

private:
/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
};



}
