﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cmath>
#include <string>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/crypto.h>

#define NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET

#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#if NN_GFX_IS_TARGET_NVN
    #include <nvn/nvn.h>
    #include <nvn/nvn_FuncPtrInline.h>
#endif

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    #include <nv/nv_MemoryManagement.h>
#endif

#include "WlanTestTypes.h"

namespace WlanTest {

using namespace std;

static const uint32_t SNOW = 0xFFFAFA;
static const uint32_t GHOSTWHITE = 0xF8F8FF;
static const uint32_t WHITE_SMOKE = 0xF5F5F5;
static const uint32_t GAINSBORO = 0xDCDCDC;
static const uint32_t FLORALWHITE = 0xFFFAF0;
static const uint32_t OLD_LACE = 0xFDF5E6;
static const uint32_t LINEN = 0xFAF0E6;
static const uint32_t ANTIQUE_WHITE = 0xFAEBD7;
static const uint32_t PAPAYAWHIP = 0xFFEFD5;
static const uint32_t BLANCHED_ALMOND = 0xFFEBCD;
static const uint32_t BISQUE = 0xFFE4C4;
static const uint32_t PEACHPUFF = 0xFFDAB9;
static const uint32_t NAVAJO_WHITE = 0xFFDEAD;
static const uint32_t MOCCASIN = 0xFFE4B5;
static const uint32_t CORNSILK = 0xFFF8DC;
static const uint32_t IVORY = 0xFFFFF0;
static const uint32_t LEMONCHIFFON = 0xFFFACD;
static const uint32_t SEASHELL = 0xFFF5EE;
static const uint32_t HONEYDEW = 0xF0FFF0;
static const uint32_t MINTCREAM = 0xF5FFFA;
static const uint32_t AZURE = 0xF0FFFF;
static const uint32_t ALICE_BLUE = 0xF0F8FF;
static const uint32_t LAVENDER = 0xE6E6FA;
static const uint32_t LAVENDER_BLUSH = 0xFFF0F5;
static const uint32_t MISTY_ROSE = 0xFFE4E1;
static const uint32_t WHITE = 0xFFFFFF;
static const uint32_t BLACK = 0x000000;
static const uint32_t DARK_SLATE_GREY = 0x2F4F4F;
static const uint32_t DIM_GREY = 0x696969;
static const uint32_t SLATE_GRAY = 0x708090;
static const uint32_t LIGHT_SLATE_GRAY = 0x778899;
static const uint32_t GRAY = 0xBEBEBE;
static const uint32_t LIGHT_GREY = 0xD3D3D3;
static const uint32_t MIDNIGHT_BLUE = 0x191970;
static const uint32_t NAVY = 0x000080;
static const uint32_t NAVY_BLUE = 0x000080;
static const uint32_t CORNFLOWER_BLUE = 0x6495ED;
static const uint32_t DARK_SLATE_BLUE = 0x483D8B;
static const uint32_t SLATE_BLUE = 0x6A5ACD;
static const uint32_t MEDIUM_SLATE_BLUE = 0x7B68EE;
static const uint32_t LIGHT_SLATE_BLUE = 0x8470FF;
static const uint32_t MEDIUM_BLUE = 0x0000CD;
static const uint32_t ROYAL_BLUE = 0x4169E1;
static const uint32_t BLUE = 0x0000FF;
static const uint32_t DODGER_BLUE = 0x1E90FF;
static const uint32_t DEEP_SKY_BLUE = 0x00BFFF;
static const uint32_t SKY_BLUE = 0x87CEEB;
static const uint32_t LIGHT_SKY_BLUE = 0x87CEFA;
static const uint32_t STEEL_BLUE = 0x4682B4;
static const uint32_t LIGHT_STEEL_BLUE = 0xB0C4DE;
static const uint32_t LIGHT_BLUE = 0xADD8E6;
static const uint32_t POWDER_BLUE = 0xB0E0E6;
static const uint32_t PALE_TURQUOISE = 0xAFEEEE;
static const uint32_t DARK_TURQUOISE = 0x00CED1;
static const uint32_t MEDIUM_TURQUOISE = 0x48D1CC;
static const uint32_t TURQUOISE = 0x40E0D0;
static const uint32_t CYAN = 0x00FFFF;
static const uint32_t LIGHT_CYAN = 0xE0FFFF;
static const uint32_t CADET_BLUE = 0x5F9EA0;
static const uint32_t MEDIUM_AQUAMARINE = 0x66CDAA;
static const uint32_t AQUAMARINE = 0x7FFFD4;
static const uint32_t DARK_GREEN = 0x006400;
static const uint32_t DARK_OLIVE_GREEN = 0x556B2F;
static const uint32_t DARK_SEA_GREEN = 0x8FBC8F;
static const uint32_t SEA_GREEN = 0x2E8B57;
static const uint32_t MEDIUM_SEA_GREEN = 0x3CB371;
static const uint32_t LIGHT_SEA_GREEN = 0x20B2AA;
static const uint32_t PALE_GREEN = 0x98FB98;
static const uint32_t SPRING_GREEN = 0x00FF7F;
static const uint32_t LAWN_GREEN = 0x7CFC00;
static const uint32_t GREEN = 0x00FF00;
static const uint32_t CHARTREUSE = 0x7FFF00;
static const uint32_t MEDIUM_SPRING_GREEN = 0x00FA9A;
static const uint32_t GREEN_YELLOW = 0xADFF2F;
static const uint32_t LIME_GREEN = 0x32CD32;
static const uint32_t YELLOW_GREEN = 0x9ACD32;
static const uint32_t FOREST_GREEN = 0x228B22;
static const uint32_t OLIVE_DRAB = 0x6B8E23;
static const uint32_t DARK_KHAKI = 0xBDB76B;
static const uint32_t KHAKI = 0xF0E68C;
static const uint32_t PALE_GOLDENROD = 0xEEE8AA;
static const uint32_t LIGHT_GOLDENROD_YELLOW = 0xFAFAD2;
static const uint32_t LIGHT_YELLOW = 0xFFFFE0;
static const uint32_t YELLOW = 0xFFFF00;
static const uint32_t GOLD = 0xFFD700;
static const uint32_t LIGHT_GOLDEN_ROD = 0xEEDD82;
static const uint32_t GOLDEN_ROD = 0xDAA520;
static const uint32_t DARKG_GLDEN_ROD = 0xB8860B;
static const uint32_t ROSY_BROWN = 0xBC8F8F;
static const uint32_t INDIAN_RED = 0xCD5C5C;
static const uint32_t SADDLE_BROWN = 0x8B4513;
static const uint32_t SIENNA = 0xA0522D;
static const uint32_t PERU = 0xCD853F;
static const uint32_t BURLYWOOD = 0xDEB887;
static const uint32_t BEIGE = 0xF5F5DC;
static const uint32_t WHEAT = 0xF5DEB3;
static const uint32_t SANDY_BROWN = 0xF4A460;
static const uint32_t TAN = 0xD2B48C;
static const uint32_t CHOCOLATE = 0xD2691E;
static const uint32_t FIREBRICK = 0xB22222;
static const uint32_t BROWN = 0xA52A2A;
static const uint32_t DARK_SALMON = 0xE9967A;
static const uint32_t SALMON = 0xFA8072;
static const uint32_t LIGHT_SALMON = 0xFFA07A;
static const uint32_t ORANGE = 0xFFA500;
static const uint32_t DARK_ORANGE = 0xFF8C00;
static const uint32_t CORAL = 0xFF7F50;
static const uint32_t LIGHT_CORAL = 0xF08080;
static const uint32_t TOMATO = 0xFF6347;
static const uint32_t ORANGE_RED = 0xFF4500;
static const uint32_t RED = 0xFF0000;
static const uint32_t HOT_PINK = 0xFF69B4;
static const uint32_t DEEP_PINK = 0xFF1493;
static const uint32_t PINK = 0xFFC0CB;
static const uint32_t LIGHT_PINK = 0xFFB6C1;
static const uint32_t PALE_VIOLET_RED = 0xDB7093;
static const uint32_t MAROON = 0xB03060;
static const uint32_t MEDIUM_VIOLET_RED = 0xC71585;
static const uint32_t VIOLET_RED = 0xD02090;
static const uint32_t MAGENTA = 0xFF00FF;
static const uint32_t VIOLET = 0xEE82EE;
static const uint32_t PLUM = 0xDDA0DD;
static const uint32_t ORCHID = 0xDA70D6;
static const uint32_t MEDIUM_ORCHID = 0xBA55D3;
static const uint32_t DARK_ORCHID = 0x9932CC;
static const uint32_t DARK_VIOLET = 0x9400D3;
static const uint32_t BLUE_VIOLET = 0x8A2BE2;
static const uint32_t PURPLE = 0xA020F0;
static const uint32_t MEDIUM_PURPLE = 0x9370DB;
static const uint32_t THISTLE = 0xD8BFD8;
static const uint32_t DARK_GRAY = 0xA9A9A9;
static const uint32_t DARK_BLUE = 0x00008B;
static const uint32_t DARK_CYAN = 0x008B8B;
static const uint32_t DARK_MAGENTA = 0x8B008B;
static const uint32_t DARK_RED = 0x8B0000;
static const uint32_t LIGHT_GREEN = 0x90EE90;
static const uint32_t INDIGO = 0x4B082;
static const uint32_t INDIGO2 = 0x218868;
static const uint32_t CRIMSON = 0xDC143C;
static const uint32_t SNOW1 = 0xFFFAFA;
static const uint32_t SNOW2 = 0xEEE9E9;
static const uint32_t SNOW3 = 0xCDC9C9;
static const uint32_t SNOW4 = 0x8B8989;
static const uint32_t SEA_SHELL1 = 0xFFF5EE;
static const uint32_t SEA_SHELL2 = 0xEEE5DE;
static const uint32_t SEA_SHELL3 = 0xCDC5BF;
static const uint32_t SEA_SHELL4 = 0x8B8682;
static const uint32_t ANTIQUE_WHITE1 = 0xFFEFDB;
static const uint32_t ANTIQUE_WHITE2 = 0xEEDFCC;
static const uint32_t ANTIQUE_WHITE3 = 0xCDC0B0;
static const uint32_t ANTIQUE_WHITE4 = 0x8B8378;
static const uint32_t BISQUE1 = 0xFFE4C4;
static const uint32_t BISQUE2 = 0xEED5B7;
static const uint32_t BISQUE3 = 0xCDB79E;
static const uint32_t BISQUE4 = 0x8B7D6B;
static const uint32_t PEACH_PUFF1 = 0xFFDAB9;
static const uint32_t PEACH_PUFF2 = 0xEECBAD;
static const uint32_t PEACH_PUFF3 = 0xCDAF95;
static const uint32_t PEACH_PUFF4 = 0x8B7765;
static const uint32_t NAVAJO_WHITE1 = 0xFFDEAD;
static const uint32_t NAVAJO_WHITE2 = 0xEECFA1;
static const uint32_t NAVAJO_WHITE3 = 0xCDB38B;
static const uint32_t NAVAJO_WHITE4 = 0x8B795E;
static const uint32_t LEMON_CHIFFON1 = 0xFFFACD;
static const uint32_t LEMON_CHIFFON2 = 0xEEE9BF;
static const uint32_t LEMON_CHIFFON3 = 0xCDC9A5;
static const uint32_t LEMON_CHIFFON4 = 0x8B8970;
static const uint32_t CORNSILK1 = 0xFFF8DC;
static const uint32_t CORNSILK2 = 0xEEE8CD;
static const uint32_t CORNSILK3 = 0xCDC8B1;
static const uint32_t CORNSILK4 = 0x8B8878;
static const uint32_t IVORY1 = 0xFFFFF0;
static const uint32_t IVORY2 = 0xEEEEE0;
static const uint32_t IVORY3 = 0xCDCDC1;
static const uint32_t IVORY4 = 0x8B8B83;
static const uint32_t HONEYDEW1 = 0xF0FFF0;
static const uint32_t HONEYDEW2 = 0xE0EEE0;
static const uint32_t HONEYDEW3 = 0xC1CDC1;
static const uint32_t HONEYDEW4 = 0x838B83;
static const uint32_t LAVENDER_BLUSH1 = 0xFFF0F5;
static const uint32_t LAVENDER_BLUSH2 = 0xEEE0E5;
static const uint32_t LAVENDER_BLUSH3 = 0xCDC1C5;
static const uint32_t LAVENDER_BLUSH4 = 0x8B8386;
static const uint32_t MISTY_ROSE1 = 0xFFE4E1;
static const uint32_t MISTY_ROSE2 = 0xEED5D2;
static const uint32_t MISTY_ROSE3 = 0xCDB7B5;
static const uint32_t MISTY_ROSE4 = 0x8B7D7B;
static const uint32_t AZURE1 = 0xF0FFFF;
static const uint32_t AZURE2 = 0xE0EEEE;
static const uint32_t AZURE3 = 0xC1CDCD;
static const uint32_t AZURE4 = 0x838B8B;
static const uint32_t SLATE_BLUE1 = 0x836FFF;
static const uint32_t SLATE_BLUE2 = 0x7A67EE;
static const uint32_t SLATE_BLUE3 = 0x6959CD;
static const uint32_t SLATE_BLUE4 = 0x473C8B;
static const uint32_t ROYAL_BLUE1 = 0x4876FF;
static const uint32_t ROYAL_BLUE2 = 0x436EEE;
static const uint32_t ROYAL_BLUE3 = 0x3A5FCD;
static const uint32_t ROYAL_BLUE4 = 0x27408B;
static const uint32_t BLUE1 = 0x0000FF;
static const uint32_t BLUE2 = 0x0000EE;
static const uint32_t BLUE3 = 0x0000CD;
static const uint32_t BLUE4 = 0x00008B;
static const uint32_t DODGER_BLUE1 = 0x1E90FF;
static const uint32_t DODGER_BLUE2 = 0x1C86EE;
static const uint32_t DODGER_BLUE3 = 0x1874CD;
static const uint32_t DODGER_BLUE4 = 0x104E8B;
static const uint32_t STEEL_BLUE1 = 0x63B8FF;
static const uint32_t STEEL_BLUE2 = 0x5CACEE;
static const uint32_t STEEL_BLUE3 = 0x4F94CD;
static const uint32_t STEEL_BLUE4 = 0x36648B;
static const uint32_t DEEP_SKY_BLUE1 = 0x00BFFF;
static const uint32_t DEEP_SKY_BLUE2 = 0x00B2EE;
static const uint32_t DEEP_SKY_BLUE3 = 0x009ACD;
static const uint32_t DEEP_SKY_BLUE4 = 0x00688B;
static const uint32_t SKY_BLUE1 = 0x87CEFF;
static const uint32_t SKY_BLUE2 = 0x7EC0EE;
static const uint32_t SKY_BLUE3 = 0x6CA6CD;
static const uint32_t SKY_BLUE4 = 0x4A708B;
static const uint32_t LIGHT_SKY_BLUE1 = 0xB0E2FF;
static const uint32_t LIGHT_SKY_BLUE2 = 0xA4D3EE;
static const uint32_t LIGHT_SKY_BLUE3 = 0x8DB6CD;
static const uint32_t LIGHT_SKY_BLUE4 = 0x607B8B;
static const uint32_t SLATE_GRAY1 = 0xC6E2FF;
static const uint32_t SLATE_GRAY2 = 0xB9D3EE;
static const uint32_t SLATE_GRAY3 = 0x9FB6CD;
static const uint32_t SLATE_GRAY4 = 0x6C7B8B;
static const uint32_t LIGHT_STEEL_BLUE1 = 0xCAE1FF;
static const uint32_t LIGHT_STEEL_BLUE2 = 0xBCD2EE;
static const uint32_t LIGHT_STEEL_BLUE3 = 0xA2B5CD;
static const uint32_t LIGHT_STEEL_BLUE4 = 0x6E7B8B;
static const uint32_t LIGHT_BLUE1 = 0xBFEFFF;
static const uint32_t LIGHT_BLUE2 = 0xB2DFEE;
static const uint32_t LIGHT_BLUE3 = 0x9AC0CD;
static const uint32_t LIGHT_BLUE4 = 0x68838B;
static const uint32_t LIGHT_CYAN1 = 0xE0FFFF;
static const uint32_t LIGHT_CYAN2 = 0xD1EEEE;
static const uint32_t LIGHT_CYAN3 = 0xB4CDCD;
static const uint32_t LIGHT_CYAN4 = 0x7A8B8B;
static const uint32_t PALE_TURQUOISE1 = 0xBBFFFF;
static const uint32_t PALE_TURQUOISE2 = 0xAEEEEE;
static const uint32_t PALE_TURQUOISE3 = 0x96CDCD;
static const uint32_t PALE_TURQUOISE4 = 0x668B8B;
static const uint32_t CADET_BLUE1 = 0x98F5FF;
static const uint32_t CADET_BLUE2 = 0x8EE5EE;
static const uint32_t CADET_BLUE3 = 0x7AC5CD;
static const uint32_t CADET_BLUE4 = 0x53868B;
static const uint32_t TURQUOISE1 = 0x00F5FF;
static const uint32_t TURQUOISE2 = 0x00E5EE;
static const uint32_t TURQUOISE3 = 0x00C5CD;
static const uint32_t TURQUOISE4 = 0x00868B;
static const uint32_t CYAN1 = 0x00FFFF;
static const uint32_t CYAN2 = 0x00EEEE;
static const uint32_t CYAN3 = 0x00CDCD;
static const uint32_t CYAN4 = 0x008B8B;
static const uint32_t DARK_SLATE_GRAY1 = 0x97FFFF;
static const uint32_t DARK_SLATE_GRAY2 = 0x8DEEEE;
static const uint32_t DARK_SLATE_GRAY3 = 0x79CDCD;
static const uint32_t DARK_SLATE_GRAY4 = 0x528B8B;
static const uint32_t AQUAMARINE1 = 0x7FFFD4;
static const uint32_t AQUAMARINE2 = 0x76EEC6;
static const uint32_t AQUAMARINE3 = 0x66CDAA;
static const uint32_t AQUAMARINE4 = 0x458B74;
static const uint32_t DARK_SEA_GREEN1 = 0xC1FFC1;
static const uint32_t DARK_SEA_GREEN2 = 0xB4EEB4;
static const uint32_t DARK_SEA_GREEN3 = 0x9BCD9B;
static const uint32_t DARK_SEA_GREEN4 = 0x698B69;
static const uint32_t SEA_GREEN1 = 0x54FF9F;
static const uint32_t SEA_GREEN2 = 0x4EEE94;
static const uint32_t SEA_GREEN3 = 0x43CD80;
static const uint32_t SEA_GREEN4 = 0x2E8B57;
static const uint32_t PALE_GREEN1 = 0x9AFF9A;
static const uint32_t PALE_GREEN2 = 0x90EE90;
static const uint32_t PALE_GREEN3 = 0x7CCD7C;
static const uint32_t PALE_GREEN4 = 0x548B54;
static const uint32_t SPRING_GREEN1 = 0x00FF7F;
static const uint32_t SPRING_GREEN2 = 0x00EE76;
static const uint32_t SPRING_GREEN3 = 0x00CD66;
static const uint32_t SPRING_GREEN4 = 0x008B45;
static const uint32_t GREEN1 = 0x00FF00;
static const uint32_t GREEN2 = 0x00EE00;
static const uint32_t GREEN3 = 0x00CD00;
static const uint32_t GREEN4 = 0x008B00;
static const uint32_t CHARTREUSE1 = 0x7FFF00;
static const uint32_t CHARTREUSE2 = 0x76EE00;
static const uint32_t CHARTREUSE3 = 0x66CD00;
static const uint32_t CHARTREUSE4 = 0x458B00;
static const uint32_t OLIVE_DRAB1 = 0xC0FF3E;
static const uint32_t OLIVE_DRAB2 = 0xB3EE3A;
static const uint32_t OLIVE_DRAB3 = 0x9ACD32;
static const uint32_t OLIVE_DRAB4 = 0x698B22;
static const uint32_t DARK_OLIVE_GREEN1 = 0xCAFF70;
static const uint32_t DARK_OLIVE_GREEN2 = 0xBCEE68;
static const uint32_t DARK_OLIVE_GREEN3 = 0xA2CD5A;
static const uint32_t DARK_OLIVE_GREEN4 = 0x6E8B3D;
static const uint32_t KHAKI1 = 0xFFF68F;
static const uint32_t KHAKI2 = 0xEEE685;
static const uint32_t KHAKI3 = 0xCDC673;
static const uint32_t KHAKI4 = 0x8B864E;
static const uint32_t LIGHT_GOLDENROD1 = 0xFFEC8B;
static const uint32_t LIGHT_GOLDENROD2 = 0xEEDC82;
static const uint32_t LIGHT_GOLDENROD3 = 0xCDBE70;
static const uint32_t LIGHT_GOLDENROD4 = 0x8B814C;
static const uint32_t LIGHT_YELLOW1 = 0xFFFFE0;
static const uint32_t LIGHT_YELLOW2 = 0xEEEED1;
static const uint32_t LIGHT_YELLOW3 = 0xCDCDB4;
static const uint32_t LIGHT_YELLOW4 = 0x8B8B7A;
static const uint32_t YELLOW1 = 0xFFFF00;
static const uint32_t YELLOW2 = 0xEEEE00;
static const uint32_t YELLOW3 = 0xCDCD00;
static const uint32_t YELLOW4 = 0x8B8B00;
static const uint32_t GOLD1 = 0xFFD700;
static const uint32_t GOLD2 = 0xEEC900;
static const uint32_t GOLD3 = 0xCDAD00;
static const uint32_t GOLD4 = 0x8B7500;
static const uint32_t GOLDENROD1 = 0xFFC125;
static const uint32_t GOLDENROD2 = 0xEEB422;
static const uint32_t GOLDENROD3 = 0xCD9B1D;
static const uint32_t GOLDENROD4 = 0x8B6914;
static const uint32_t DARK_GOLDENROD1 = 0xFFB90F;
static const uint32_t DARK_GOLDENROD2 = 0xEEAD0E;
static const uint32_t DARK_GOLDENROD3 = 0xCD950C;
static const uint32_t DARK_GOLDENROD4 = 0x8B6508;
static const uint32_t ROSY_BROWN1 = 0xFFC1C1;
static const uint32_t ROSY_BROWN2 = 0xEEB4B4;
static const uint32_t ROSY_BROWN3 = 0xCD9B9B;
static const uint32_t ROSY_BROWN4 = 0x8B6969;
static const uint32_t INDIAN_RED1 = 0xFF6A6A;
static const uint32_t INDIAN_RED2 = 0xEE6363;
static const uint32_t INDIAN_RED3 = 0xCD5555;
static const uint32_t INDIAN_RED4 = 0x8B3A3A;
static const uint32_t SIENNA1 = 0xFF8247;
static const uint32_t SIENNA2 = 0xEE7942;
static const uint32_t SIENNA3 = 0xCD6839;
static const uint32_t SIENNA4 = 0x8B4726;
static const uint32_t BURLYWOOD1 = 0xFFD39B;
static const uint32_t BURLYWOOD2 = 0xEEC591;
static const uint32_t BURLYWOOD3 = 0xCDAA7D;
static const uint32_t BURLYWOOD4 = 0x8B7355;
static const uint32_t WHEAT1 = 0xFFE7BA;
static const uint32_t WHEAT2 = 0xEED8AE;
static const uint32_t WHEAT3 = 0xCDBA96;
static const uint32_t WHEAT4 = 0x8B7E66;
static const uint32_t TAN1 = 0xFFA54F;
static const uint32_t TAN2 = 0xEE9A49;
static const uint32_t TAN3 = 0xCD853F;
static const uint32_t TAN4 = 0x8B5A2B;
static const uint32_t CHOCOLATE1 = 0xFF7F24;
static const uint32_t CHOCOLATE2 = 0xEE7621;
static const uint32_t CHOCOLATE3 = 0xCD661D;
static const uint32_t CHOCOLATE4 = 0x8B4513;
static const uint32_t FIREBRICK1 = 0xFF3030;
static const uint32_t FIREBRICK2 = 0xEE2C2C;
static const uint32_t FIREBRICK3 = 0xCD2626;
static const uint32_t FIREBRICK4 = 0x8B1A1A;
static const uint32_t BROWN1 = 0xFF4040;
static const uint32_t BROWN2 = 0xEE3B3B;
static const uint32_t BROWN3 = 0xCD3333;
static const uint32_t BROWN4 = 0x8B2323;
static const uint32_t SALMON1 = 0xFF8C69;
static const uint32_t SALMON2 = 0xEE8262;
static const uint32_t SALMON3 = 0xCD7054;
static const uint32_t SALMON4 = 0x8B4C39;
static const uint32_t LIGHT_SALMON1 = 0xFFA07A;
static const uint32_t LIGHT_SALMON2 = 0xEE9572;
static const uint32_t LIGHT_SALMON3 = 0xCD8162;
static const uint32_t LIGHT_SALMON4 = 0x8B5742;
static const uint32_t ORANGE1 = 0xFFA500;
static const uint32_t ORANGE2 = 0xEE9A00;
static const uint32_t ORANGE3 = 0xCD8500;
static const uint32_t ORANGE4 = 0x8B5A00;
static const uint32_t DARK_ORANGE1 = 0xFF7F00;
static const uint32_t DARK_ORANGE2 = 0xEE7600;
static const uint32_t DARK_ORANGE3 = 0xCD6600;
static const uint32_t DARK_ORANGE4 = 0x8B4500;
static const uint32_t CORAL1 = 0xFF7256;
static const uint32_t CORAL2 = 0xEE6A50;
static const uint32_t CORAL3 = 0xCD5B45;
static const uint32_t CORAL4 = 0x8B3E2F;
static const uint32_t TOMATO1 = 0xFF6347;
static const uint32_t TOMATO2 = 0xEE5C42;
static const uint32_t TOMATO3 = 0xCD4F39;
static const uint32_t TOMATO4 = 0x8B3626;
static const uint32_t ORANGE_RED1 = 0xFF4500;
static const uint32_t ORANGE_RED2 = 0xEE4000;
static const uint32_t ORANGE_RED3 = 0xCD3700;
static const uint32_t ORANGE_RED4 = 0x8B2500;
static const uint32_t RED1 = 0xFF0000;
static const uint32_t RED2 = 0xEE0000;
static const uint32_t RED3 = 0xCD0000;
static const uint32_t RED4 = 0x8B0000;
static const uint32_t DEEP_PINK1 = 0xFF1493;
static const uint32_t DEEP_PINK2 = 0xEE1289;
static const uint32_t DEEP_PINK3 = 0xCD1076;
static const uint32_t DEEP_PINK4 = 0x8B0A50;
static const uint32_t HOT_PINK1 = 0xFF6EB4;
static const uint32_t HOT_PINK2 = 0xEE6AA7;
static const uint32_t HOT_PINK3 = 0xCD6090;
static const uint32_t HOT_PINK4 = 0x8B3A62;
static const uint32_t PINK1 = 0xFFB5C5;
static const uint32_t PINK2 = 0xEEA9B8;
static const uint32_t PINK3 = 0xCD919E;
static const uint32_t PINK4 = 0x8B636C;
static const uint32_t LIGHT_PINK1 = 0xFFAEB9;
static const uint32_t LIGHT_PINK2 = 0xEEA2AD;
static const uint32_t LIGHT_PINK3 = 0xCD8C95;
static const uint32_t LIGHT_PINK4 = 0x8B5F65;
static const uint32_t PALE_VIOLET_RED1 = 0xFF82AB;
static const uint32_t PALE_VIOLET_RED2 = 0xEE799F;
static const uint32_t PALE_VIOLET_RED3 = 0xCD6889;
static const uint32_t PALE_VIOLET_RED4 = 0x8B475D;
static const uint32_t MAROON1 = 0xFF34B3;
static const uint32_t MAROON2 = 0xEE30A7;
static const uint32_t MAROON3 = 0xCD2990;
static const uint32_t MAROON4 = 0x8B1C62;
static const uint32_t VIOLET_RED1 = 0xFF3E96;
static const uint32_t VIOLET_RED2 = 0xEE3A8C;
static const uint32_t VIOLET_RED3 = 0xCD3278;
static const uint32_t VIOLET_RED4 = 0x8B2252;
static const uint32_t MAGENTA1 = 0xFF00FF;
static const uint32_t MAGENTA2 = 0xEE00EE;
static const uint32_t MAGENTA3 = 0xCD00CD;
static const uint32_t MAGENTA4 = 0x8B008B;
static const uint32_t ORCHID1 = 0xFF83FA;
static const uint32_t ORCHID2 = 0xEE7AE9;
static const uint32_t ORCHID3 = 0xCD69C9;
static const uint32_t ORCHID4 = 0x8B4789;
static const uint32_t PLUM1 = 0xFFBBFF;
static const uint32_t PLUM2 = 0xEEAEEE;
static const uint32_t PLUM3 = 0xCD96CD;
static const uint32_t PLUM4 = 0x8B668B;
static const uint32_t MEDIUM_ORCHID1 = 0xE066FF;
static const uint32_t MEDIUM_ORCHID2 = 0xD15FEE;
static const uint32_t MEDIUM_ORCHID3 = 0xB452CD;
static const uint32_t MEDIUM_ORCHID4 = 0x7A378B;
static const uint32_t DARK_ORCHID1 = 0xBF3EFF;
static const uint32_t DARK_ORCHID2 = 0xB23AEE;
static const uint32_t DARK_ORCHID3 = 0x9A32CD;
static const uint32_t DARK_ORCHID4 = 0x68228B;
static const uint32_t PURPLE1 = 0x9B30FF;
static const uint32_t PURPLE2 = 0x912CEE;
static const uint32_t PURPLE3 = 0x7D26CD;
static const uint32_t PURPLE4 = 0x551A8B;
static const uint32_t MEDIUM_PURPLE1 = 0xAB82FF;
static const uint32_t MEDIUM_PURPLE2 = 0x9F79EE;
static const uint32_t MEDIUM_PURPLE3 = 0x8968CD;
static const uint32_t MEDIUM_PURPLE4 = 0x5D478B;
static const uint32_t THISTLE1 = 0xFFE1FF;
static const uint32_t THISTLE2 = 0xEED2EE;
static const uint32_t THISTLE3 = 0xCDB5CD;
static const uint32_t THISTLE4 = 0x8B7B8B;


struct Color
{
    float R;
    float G;
    float B;
    float A;

    Color()
    {
        R=1.0;
        G=1.0;
        B=1.0;
        A=1.0;
    }

    Color(float r, float g, float b, float a)
    {
        R=r;
        G=g;
        B=b;
        A=a;
    }
};


inline Color ToColor(uint32_t rgb, float alpha = 1.0)
{
    Color c;
    c.R = static_cast<float>((rgb & 0x00ff0000) >> 16) / 255;
    c.G = static_cast<float>((rgb & 0x0000ff00) >> 8) / 255;
    c.B = static_cast<float>((rgb & 0x000000ff) >> 0) / 255;
    c.A = alpha;

    return c;
}

inline nn::util::Unorm8x4 ToUnormColor(uint32_t rgb, uint8_t a = 0xff)
{
    nn::util::Unorm8x4 c = { { static_cast<uint8_t>((rgb & 0x00ff0000) >> 16),
                               static_cast<uint8_t>((rgb & 0x0000ff00) >> 8),
                               static_cast<uint8_t>((rgb & 0x000000ff) >> 0),
                               a } };

    return c;
}

inline nn::util::Unorm8x4 ToUnormColor(Color color)
{
    nn::util::Unorm8x4 c = { { static_cast<uint8_t>(color.R * 255),
                               static_cast<uint8_t>(color.G * 255),
                               static_cast<uint8_t>(color.B * 255),
                               static_cast<uint8_t>(color.A * 255) } };
    return c;
}

inline Color GenerateColor(void* data, size_t size)
{
    nn::crypto::Sha1Generator sha1;
    sha1.Initialize();
    sha1.Update(data, size);
    uint8_t buf[nn::crypto::Sha1Generator::HashSize * 2];
    sha1.GetHash(buf, sizeof(buf));
    uint64_t val;
    memcpy(&val, buf, sizeof(val));

    return ToColor(val, 1.0f);
}

inline Color ToBlackOrWhite(Color c, float threshold = 0.5)
{
    float a = (c.R + c.G + c.B) / 3.0f;
    if(a < threshold)
    {
        return ToColor(BLACK);
    }
    else
    {
        return ToColor(WHITE);
    }
}

inline Color ToNegative(Color c)
{
    Color r;
    r.R = 1.0f - c.R;
    r.G = 1.0f - c.G;
    r.B = 1.0f - c.B;

    return r;
}

class IRenderSystem
{
private:
protected:

public:

    IRenderSystem() {}
    virtual ~IRenderSystem() {}

    virtual void Initialize() = 0;
    virtual void Finalize() = 0;

    virtual void SwapBuffers() = 0;
    virtual void Clear() = 0;

    virtual void SetColor(Color color) = 0;
    virtual void SetScale(float x, float y) = 0;
    virtual void SetPointSize(float size) = 0;
    virtual void SetLineWidth(float width) = 0;
    virtual void SetFixedWidth(float width) = 0;
    virtual void SetFontSize(float width, float height) = 0;

    virtual float GetFixedWidth() = 0;
    virtual float GetLineHeight() = 0;
    virtual float GetFontWidth() = 0;
    virtual float GetFontHeight() = 0;
    virtual void GetScale(float* x, float* y) = 0;
    virtual float GetPointSize() = 0;
    virtual float GetLineWidth() = 0;

    virtual uint16_t GetWidth() = 0;
    virtual uint16_t GetHeight() = 0;

    virtual void DrawPoint(float x, float y) = 0;
    virtual void DrawLine(float x0, float y0, float x1, float y1) = 0;
    virtual void DrawTriangle(float x0, float y0, float x1, float y1, float x2, float y2) = 0;
    virtual void DrawSquare(float x0, float y0, float x1, float y1, float x2, float y2, float x3, float y3) = 0;
    virtual void DrawRectangle(float x, float y, float width, float height) = 0;
    virtual void DrawText(float x, float y, const string& text) = 0;

};

class TextRenderer : public IRenderSystem
{
private:

    nn::os::Mutex m_Cs;

    nn::vi::Display* m_pDisplay;
    nn::vi::Layer* m_pLayer;
    nn::gfx::util::DebugFontTextWriter m_Writer;

    static const size_t VISIBLE_POOL_MEMORY_SIZE; // 16 * 1024 * 1024;
    static const size_t INVISIBLE_POOL_MEMORY_SIZE; // 20 * 1024 * 1024;

    void* m_pVisiblePoolMemory;
    void* m_pInvisiblePoolMemory;
    void* m_pMemoryPoolStart;
    ptrdiff_t m_MemoryPoolOffset;
    void* m_pInvisibleMemoryPoolStart;
    ptrdiff_t m_InvisibleMemoryPoolOffset;
    nn::gfx::MemoryPool m_MemoryPool;
    nn::gfx::MemoryPool m_InvisibleMemoryPool;

    nn::gfx::Device m_Device;
    nn::gfx::SwapChain m_SwapChain;
    nn::gfx::Queue m_Queue;
    nn::gfx::CommandBuffer m_CommandBuffer;
    nn::gfx::ViewportScissorState m_ViewportScissor;
    nn::gfx::DescriptorPool m_SamplerDescriptorPool;
    int m_SamplerDescriptorBaseIndex;
    nn::gfx::DescriptorPool m_TextureDescriptorPool;
    int m_TextureDescriptorBaseIndex;
    nn::util::BytePtr m_pMemory;
    nn::util::BytePtr m_pMemoryHeap;
    nn::util::BytePtr m_pDebugFontHeap;


    void InitializeDevice();
    void InitializeMemoryPool();
    void InitializeInvisibleMemoryPool();
    void InitializeSwapChain();
    void InitializeQueue();
    void InitializeCommandBuffer();
    void InitializeViewport();
    void InitializeSamplerDescriptorPool();
    void InitializeTextureDescriptorPool();
    void InitializeLayer();
    void InitializeGfxObjects();
    void FinalizeGfxObjects();

protected:

    float m_FixedWidth;
    float m_FontWidth;
    float m_FontHeight;
    float m_ScaleX;
    float m_ScaleY;
    int m_Width;
    int m_Height;

public:

    TextRenderer();
    virtual ~TextRenderer();

    void Initialize();
    void Finalize();

    virtual void SwapBuffers();
    virtual void Clear() {}

    virtual void SetColor(Color color);
    virtual void SetScale(float x, float y);
    virtual void SetPointSize(float) {}
    virtual void SetLineWidth(float) {}
    virtual void SetFixedWidth(float width);
    virtual void SetFontSize(float width, float height);

    virtual float GetFixedWidth() { return m_FixedWidth; }
    virtual float GetLineHeight() { return m_FontHeight - 2; }
    virtual float GetFontWidth() { return m_FontWidth; }
    virtual float GetFontHeight() { return m_FontHeight; }
    virtual void GetScale(float* x, float* y) { *x = m_ScaleX, *y = m_ScaleY; }
    virtual float GetPointSize() { return 0; }
    virtual float GetLineWidth() { return 0; }
    virtual uint16_t GetWidth() { return m_Width; }
    virtual uint16_t GetHeight() { return m_Height; }

    virtual void DrawText(float x, float y, const string& text);
    // 以下、非対応
    virtual void DrawPoint(float, float) {}
    virtual void DrawLine(float, float, float, float) {}
    virtual void DrawTriangle(float, float, float, float, float, float) {}
    virtual void DrawSquare(float, float, float, float, float, float, float, float) {}
    virtual void DrawRectangle(float, float, float, float) {}

    virtual void SetCursor(float x, float y);
    void VPrint(const char* format, va_list formatArg);
    void Print(float x, float y, const char* format, ...);
    void Print(const char* format, ...);

};

class Display
{
/*---------------------------------------------------------------------------
　　　　　定数・型定義など
---------------------------------------------------------------------------*/
public:
private:

/*---------------------------------------------------------------------------
　　　　　静的変数
---------------------------------------------------------------------------*/
public:
private:

    static Display* m_pDisplay;
    IRenderSystem* m_pRenderSystem;

/*---------------------------------------------------------------------------
　　　　　静的メソッド
---------------------------------------------------------------------------*/
public:

    static Display& GetInstance();

private:

/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:

    static bool m_EnableConsoleOutput;
    static bool m_EnableDisplayOutput;

private:

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual void SwapBuffer()
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->SwapBuffers();
    }

    virtual void Clear()
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->Clear();
    }

    virtual void SetColor(Color color)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->SetColor(color);
    }

    virtual void SetColor(uint32_t color, float alpha = 1.0)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->SetColor(ToColor(color, alpha));
    }

    virtual void SetPointSize(float size)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->SetPointSize(size);
    }

    virtual void SetLineWidth(float width)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->SetLineWidth(width);
    }

    virtual void SetFixedWidth(float width)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->SetFixedWidth(width);
    }

    virtual void SetFontSize(float width, float height)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->SetFontSize(width, height);
    }

    virtual void SetScale(float x, float y)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->SetScale(x, y);
    }

    virtual float GetFixedWidth()
    {
        if( !m_EnableDisplayOutput )
        {
            return 0;
        }
        return m_pRenderSystem->GetFixedWidth();
    }

    virtual float GetLineHeight()
    {
        if( !m_EnableDisplayOutput )
        {
            return 0;
        }
        return m_pRenderSystem->GetLineHeight();
    }

    virtual float GetFontWidth()
    {
        if( !m_EnableDisplayOutput )
        {
            return 0;
        }
        return m_pRenderSystem->GetFontWidth();
    }

    virtual float GetFontHeight()
    {
        if( !m_EnableDisplayOutput )
        {
            return 0;
        }
        return m_pRenderSystem->GetFontHeight();
    }

    uint16_t GetWidth()
    {
        if( !m_EnableDisplayOutput )
        {
            return 0;
        }
        return m_pRenderSystem->GetWidth();
    }

    uint16_t GetHeight()
    {
        if( !m_EnableDisplayOutput )
        {
            return 0;
        }
        return m_pRenderSystem->GetHeight();
    }

    virtual void DrawPoint(float x, float y)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->DrawPoint(x, y);
    }

    virtual void DrawLine(float x0, float y0, float x1, float y1)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->DrawLine(x0, y0, x1, y1);
    }

    virtual void DrawTriangle(float x0, float y0, float x1, float y1, float x2, float y2)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->DrawTriangle(x0, y0, x1, y1, x2, y2);
    }

    virtual void DrawSquare(float x0, float y0, float x1, float y1, float x2, float y2, float x3, float y3)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->DrawSquare(x0, y0, x1, y1, x2, y2, x3, y3);
    }

    virtual void DrawRectangle(float x, float y, float width, float height)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->DrawRectangle(x, y, width, height);
    }

    virtual void DrawText(float x, float y, const string& text)
    {
        if( m_EnableDisplayOutput )
        {
            m_pRenderSystem->DrawText(x, y, text.c_str());
        }

        if( m_EnableConsoleOutput )
        {
            NN_LOG(text.c_str());
            NN_LOG("\n");
        }
    }

    // ~Display() で delete するため new したものを引数とする
    virtual void SetRenderSystem(IRenderSystem* pRenderSystem)
    {
        if( m_pRenderSystem )
        {
            m_pRenderSystem->Finalize();
            delete m_pRenderSystem;
        }
        m_pRenderSystem = pRenderSystem;
        m_pDisplay->m_pRenderSystem->Initialize();
        m_pDisplay->m_pRenderSystem->SetColor(ToColor(WHITE));
    }

    static void EnableConsoleOutput()
    {
        m_EnableConsoleOutput = true;
    }

    static void DisableConsoleOutput()
    {
        m_EnableConsoleOutput = false;
    }

    static bool IsEnabledConsoleOutput()
    {
        return m_EnableConsoleOutput;
    }

    static void EnableDisplayOutput()
    {
        m_EnableDisplayOutput = true;
    }

    static void DisableDisplayOutput()
    {
        m_EnableDisplayOutput = false;
    }

    static bool IsEnabledDisplayOutput()
    {
        return m_EnableDisplayOutput;
    }

protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
private:

    Display(const Display&);
    Display& operator=(const Display&);

    Display();
    virtual ~Display();

protected:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:

};

}    // WlanTest
