﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SceneManager.h"

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_LocalApi.h>
#include <nn/wlan/wlan_Result.h>

#include "SceneTitle.h"
#include "SceneMeasureAllCh.h"
#include "SceneLoggingCh.h"

SceneManager::SceneManager() :
        m_nextScene(eScene_None)
{
    m_scene = (BaseScene*) new SceneTitle(this);
    m_prevScene = eScene_Title;
    m_currScene = eScene_Title;
    m_isExit = false;
}

void SceneManager::Initialize() NN_NOEXCEPT
{
    // wlanの初期化
    nn::wlan::InitializeLocalManager();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Local::OpenClientMode());

    m_scene->Initialize();
}

void SceneManager::Finalize() NN_NOEXCEPT
{
    if( m_scene != nullptr )
    {
        m_scene->Finalize();
        delete m_scene;
    }

    // wlanの終了処理
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Local::CloseClientMode());
    nn::wlan::FinalizeLocalManager();
}

void SceneManager::Update() NN_NOEXCEPT
{
    if(m_nextScene != eScene_None)  // 次のシーンがセットされているか
    {
        // 現在のシーンの終了処理
        m_scene->Finalize();
        delete m_scene;

        switch( m_nextScene )
        {
        case eScene_Title:
            m_scene = (BaseScene*) new SceneTitle(this);
            break;
        case eScene_MeasureAllChannels:
            m_scene = (BaseScene*) new SceneMeasureAllCh(this);
            break;
        case eScene_LoggingChannel:
            m_scene = (BaseScene*) new SceneLoggingCh(this);
            break;
        case eScene_Exit:
            m_scene = (BaseScene*) new SceneTitle(this);
            m_isExit = true;
            break;
        default:
            break;
        }

        //次のシーン情報をクリア
        m_prevScene = m_currScene;
        m_currScene = m_nextScene;
        m_nextScene = eScene_None;
        //シーンを初期化
        m_scene->Initialize();
    }

    //シーンの更新
    m_scene->Update();
}

void SceneManager::Draw(GraphicTools* pTools) NN_NOEXCEPT
{
    //シーンの描画
    m_scene->Draw(pTools);
}

bool SceneManager::IsExit() NN_NOEXCEPT
{
    return m_isExit;
}

// 引数 nextScene にシーンを変更する
void SceneManager::ChangeScene(eScene NextScene) NN_NOEXCEPT
{
    //次のシーンをセットする
    m_nextScene = NextScene;
}

eScene SceneManager::GetPrevScene() NN_NOEXCEPT
{
    return m_prevScene;
}
