﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiCommunicationControlForTest.h>

#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_Ssid.h>
#include <nn/wlan/wlan_InfraApi.h>
#include <nn/wlan/wlan_ScanResultReader.h>
#include <nn/wlan/wlan_BssDescriptionReader.h>
#include <nn/wlan/wlan_Result.h>

namespace {

// スキャンバッファ
const size_t  buffSize = 100 * 1024; // 100KB
NN_ALIGNAS(4096) char  g_scanBuffer[ buffSize ];

}

void AntennaRxTest(int antenna)
{
    nn::Result result;

    // インフラ通信用APIを使うためにIPCクライアントマネージャーを初期化
    nn::wlan::InitializeInfraManager();

    // インフラ通信モードのオープン
    result = nn::wlan::Infra::OpenMode();
    if( result.IsFailure() )
    {
        NN_LOG("OpenMode failure\n");
        nn::wlan::FinalizeInfraManager();
        return;
    }
    NN_LOG("OpenMode done\n");


    // 使用するアンテナの選択
    if( antenna == 1 )
    {
        NN_LOG("Use ANT_0 antenna for rx\n");
        // アンテナ0のみを使用する場合
        nn::wlan::Infra::ChangeRxAntenna(nn::wlan::RxAntennaPattern_0);
    }
    else if( antenna == 2 )
    {
        NN_LOG("Use ANT_1 antenna for rx\n");
        // アンテナ1のみを使用する場合
        nn::wlan::Infra::ChangeRxAntenna(nn::wlan::RxAntennaPattern_1);
    }

    // どのチャンネルを何秒スキャンするか、といったスキャンパラメータの設定
    nn::wlan::ScanParameters scanParam = {
            nn::wlan::ScanType_Passive,
            {1, 6, 11, 36, 44, 48},
            6,  // 0で全チャンネルスキャン
            200,
            0,
            NULL,
            0,
            nn::wlan::MacAddress::CreateBroadcastMacAddress()
    };

    // スキャン開始
    NN_LOG("Do scan\n");
    nn::wlan::Infra::StartScan(g_scanBuffer, buffSize, scanParam);
    NN_LOG("Scan finished.\n");

    // スキャン結果解析 -----------------------------------------------

    // スキャン結果解析用のオブジェクトにスキャン結果を食わせる
    nn::wlan::BeaconScanResultReader resultReader(g_scanBuffer);

    // 見つかったAPの数
    uint32_t bssCount = resultReader.GetCount();
    // SSID表示用バッファ
    char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];

    NN_LOG("\n\n\n\n ***** SCAN RESULT *****\n");
    NN_LOG("The number of found APs : %d\n", bssCount);

    for( uint32_t i = 0; i < bssCount; i++ )
    {
        // スキャン結果からさらにビーコン解析用のオブジェクトに切り出す
        nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();
        // 見つかったAPのSSIDとRSSI値を表示
        NN_LOG("[%d]%s:%d[dBm]\n", i, beacon.GetSsid().GetHexString(ssidStr), beacon.GetRssi());
    }
    // ----------------------------------------------------------------

    // インフラ通信モードのクローズ
    nn::wlan::Infra::CloseMode();
    NN_LOG("CloseMode done\n");

    // インフラ通信用のIPCクライアントマネージャーの終了
    nn::wlan::FinalizeInfraManager();
}

extern "C" void nnMain()
{
    NN_LOG("\n\n Start ManufacturingTestSample\n\n");

    // デバッグ設定書き込み
    const bool isEnabled = true;
    nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));

    nn::nifm::Initialize();
    nn::nifm::SetWirelessCommunicationEnabledForTest(false);
    // nifmのwlan利用停止を確実に待つために1秒ほどwaitを入れておく
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    AntennaRxTest(3); // アンテナ設定を何も行わないパターン(両方のアンテナを使うことになる)
    AntennaRxTest(1); // アンテナ0を検査
    AntennaRxTest(2); // アンテナ1を検査
    NN_LOG("\n\n End ManufacturingTestSample \n\n");
}

