﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Network.h"
#include <nn/os/os_Thread.h>
#include <nn/socket/socket_Types.h>
#include <vector>


namespace ApConnectivityTest
{

Network* Network::g_Instance = nullptr;
NN_OS_ALIGNAS_THREAD_STACK uint8_t Network::g_WorkerThreadStackMemory[65536];


// インスタンスを取得
Network& Network::GetInstance()
{
    return *g_Instance;
}


// 初期化
void Network::Initialize()
{
    if (g_Instance)
    {
        return;
    }

    g_Instance = new Network();
}


// 終了
void Network::Finalize()
{
    if (!g_Instance)
    {
        return;
    }

    delete g_Instance;
    g_Instance = nullptr;
}


// AP への接続を試みる
bool Network::ConnectNetwork(const nn::nifm::NetworkProfileData& profile, bool mixedMode, bool permanentProfile, const std::function<void(bool, const nn::TimeSpan&)>& callback)
{
    // コマンド発行
    m_CommandQueue.Push(std::shared_ptr<Task>(new ConnectNetworkTask(profile, mixedMode, permanentProfile, callback)));

    return true;
}


// AP から切断する
bool Network::DisconnectNetwork()
{
    // コマンド発行
    m_CommandQueue.Push(std::shared_ptr<Task>(new DisconnectNetworkTask()));

    return true;
}


// WPS PBC 接続を試行する
bool Network::WpsPbc(nn::nifm::IpSettingData profile, const std::function<void(bool, const nn::TimeSpan&)>& callback)
{
    auto dhcp = profile.ip.isAuto;

    nn::bsdsocket::cfg::IfSettings ifSettings = {};
    ifSettings.mode = dhcp ? nn::bsdsocket::cfg::IfIpAddrMode_Dhcp : nn::bsdsocket::cfg::IfIpAddrMode_Static;
    ifSettings.metric = 0;
    ifSettings.mtu = 1400;
    if (!dhcp)
    {
        ifSettings.duplicateIpWaitTime = 1000;
        ifSettings.dnsAddrs[0] = reinterpret_cast<nn::socket::InAddr&>(profile.dns.preferredDns);
        ifSettings.dnsAddrs[1] = reinterpret_cast<nn::socket::InAddr&>(profile.dns.alternateDns);
        ifSettings.u.modeStatic.addr = reinterpret_cast<nn::socket::InAddr&>(profile.ip.ipAddress);
        ifSettings.u.modeStatic.subnetMask = reinterpret_cast<nn::socket::InAddr&>(profile.ip.subnetMask);
        ifSettings.u.modeStatic.gatewayAddr = reinterpret_cast<nn::socket::InAddr&>(profile.ip.defaultGateway);
        ifSettings.u.modeStatic.broadcastAddr.S_addr = reinterpret_cast<nn::socket::InAddr&>(profile.ip.ipAddress).S_addr | ~reinterpret_cast<nn::socket::InAddr&>(profile.ip.subnetMask).S_addr;
    }

    // コマンド発行
    m_CommandQueue.Push(std::shared_ptr<Task>(new WpsTask(nn::wlan::WpsMethod_Pbc, ifSettings, callback)));

    return true;
}


// WPS PIN 接続を試行する
bool Network::WpsPin(nn::nifm::IpSettingData profile, const std::function<void(bool, const nn::TimeSpan&)>& callback)
{
    auto dhcp = profile.ip.isAuto;

    nn::bsdsocket::cfg::IfSettings ifSettings = {};
    ifSettings.mode = dhcp ? nn::bsdsocket::cfg::IfIpAddrMode_Dhcp : nn::bsdsocket::cfg::IfIpAddrMode_Static;
    ifSettings.metric = 0;
    ifSettings.mtu = 1400;
    ifSettings.duplicateIpWaitTime = 1000;
    if (!dhcp)
    {
        ifSettings.dnsAddrs[0] = reinterpret_cast<nn::socket::InAddr&>(profile.dns.preferredDns);
        ifSettings.dnsAddrs[1] = reinterpret_cast<nn::socket::InAddr&>(profile.dns.alternateDns);
        ifSettings.u.modeStatic.addr = reinterpret_cast<nn::socket::InAddr&>(profile.ip.ipAddress);
        ifSettings.u.modeStatic.subnetMask = reinterpret_cast<nn::socket::InAddr&>(profile.ip.subnetMask);
        ifSettings.u.modeStatic.gatewayAddr = reinterpret_cast<nn::socket::InAddr&>(profile.ip.defaultGateway);
        ifSettings.u.modeStatic.broadcastAddr.S_addr = reinterpret_cast<nn::socket::InAddr&>(profile.ip.ipAddress).S_addr | ~reinterpret_cast<nn::socket::InAddr&>(profile.ip.subnetMask).S_addr;
    }

    // コマンド発行
    m_CommandQueue.Push(std::shared_ptr<Task>(new WpsTask(nn::wlan::WpsMethod_Pin, ifSettings, callback)));

    return true;
}


// Ping の送受信
Network::PingController Network::Ping(const std::string& dest, size_t length, size_t count, const std::function<void()>& callback)
{
    auto task = new PingTask(dest, length, count, callback);

    // コマンド発行
    m_CommandQueue.Push(std::shared_ptr<Task>(task));

    return task->GetContoller();
}


bool Network::IsRunningPing()
{
    // [stub]
    return false;
}


// iperf を開始する
bool Network::StartIperf(const std::string& commandline, const std::function<void()>& callback)
{
    m_IperfCommandline = commandline;
    m_IperfExitCallback = callback;

    // コマンド発行
    m_CommandQueue.Push(std::shared_ptr<Task>(new IperfStartTask()));

    return true;
}


// iperf を終了する
bool Network::StopIperf()
{
    // コマンド発行
    m_CommandQueue.Push(std::shared_ptr<Task>(new IperfStopTask()));

    return true;
}


// iperf が実行中かどうかを返す
bool Network::IsRunningIperf()
{
    // [stub]

    return m_RunningIperf;
}

bool Network::IsConnectionNifm() const
{
    return m_IsNifmConnection;
}


// コンストラクタ
Network::Network() :
    m_CommandQueue(5),
    m_pConnection(nullptr),
    m_RunningIperf(false)
{
    // ネットワーク系処理ワーカースレッドを作成
    void(Network::*pTheadFunc)() = &Network::WorkerThread;
    nn::os::CreateThread(&m_WorkerThread,
        reinterpret_cast<nn::os::ThreadFunction>(reinterpret_cast<void(*)(void*)>(*reinterpret_cast<void**>(&pTheadFunc))), this,
        g_WorkerThreadStackMemory, sizeof(g_WorkerThreadStackMemory),
        nn::os::DefaultThreadPriority);
    nn::os::SetThreadName(&m_WorkerThread, "NetworkWorkerThread");
    nn::os::StartThread(&m_WorkerThread);
}


// デストラクタ
Network::~Network()
{
    // ワーカースレッド終了
    m_CommandQueue.Push(std::shared_ptr<Task>(new QuitTask()));

    nn::os::DestroyThread(&m_WorkerThread);

    // 接続解除
    if (m_pConnection)
    {
        delete m_pConnection;
    }
}


// ワーカースレッド
void Network::WorkerThread()
{
    m_WorkerExitFlag = false;
    m_CommandQueue.Clear();

    while (!m_WorkerExitFlag)
    {
        // コマンド待ち
        std::shared_ptr<Task> command = m_CommandQueue.Pop();

        // 各処理
        command->Run();
    }
}

}
