﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "BlockingQueue.h"
#include <nn/nn_Result.h>
#include <nn/bsdsocket/cfg/cfg_Types.h>
#include <nn/nifm/nifm_TypesNetworkProfile.h>
#include <nn/nifm/nifm_TemporaryNetworkProfile.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/socket/socket_Types.h>
#include <nn/wlan/wlan_InfraApi.h>
#include <functional>
#include <memory>
#include <string>
#include <vector>


namespace ApConnectivityTest {

class Network
{
public:
    class PingController
    {
    public:
        PingController();
        NN_IMPLICIT PingController(const std::shared_ptr<int>& socket);
        void Cancel();

    private:
        std::weak_ptr<int> m_Socket;
    };

private:
    enum TaskType
    {
        TaskType_Quit,
        TaskType_ConnectNetwork,
        TaskType_DisconnectNetwork,
        TaskType_Wps,
        TaskType_Ping,
        TaskType_StopPing,
        TaskType_StartIperf,
        TaskType_StopIperf,
        TaskType_IperfExitCallback,
    };

    class Task
    {
    public:
        NN_IMPLICIT Task(TaskType type) : m_Type(type) {};
        virtual ~Task() {}

        virtual void Run() = 0;

        TaskType GetType() const { return m_Type; };

    private:
        TaskType m_Type;
    };

    class QuitTask : public Task
    {
    public:
        QuitTask() : Task(TaskType_Quit) {};
        virtual ~QuitTask() {}

        virtual void Run() { Network::GetInstance().m_WorkerExitFlag = true; };
    };

    class ConnectNetworkTask : public Task
    {
    public:
        ConnectNetworkTask(const nn::nifm::NetworkProfileData& profile, bool mixedMode, bool permanentProfile, const std::function<void(bool, const nn::TimeSpan&)>& callback);
        virtual ~ConnectNetworkTask();

        virtual void Run();

    private:
        nn::nifm::NetworkProfileData m_Profile;
        bool m_MixedMode;
        bool m_PermanentProfile;
        std::function<void(bool, const nn::TimeSpan&)> m_Callback;
    };

    class DisconnectNetworkTask : public Task
    {
    public:
        DisconnectNetworkTask();
        virtual ~DisconnectNetworkTask();

        virtual void Run();
    };

    class WpsTask : public Task
    {
    public:
        WpsTask(nn::wlan::WpsMethod method, nn::bsdsocket::cfg::IfSettings ifSettings, const std::function<void(bool, const nn::TimeSpan&)>& callback);
        virtual ~WpsTask();

        virtual void Run();

    private:
        nn::wlan::WpsMethod m_WpsMethod;
        nn::bsdsocket::cfg::IfSettings m_IfSettings;
        std::string m_Pin;
        std::function<void(bool, const nn::TimeSpan&)> m_Callback;
    };

    class PingTask : public Task
    {
    private:
        enum IcmpResultType
        {
            IcmpResultType_CriticalError = 0x1000,
            IcmpResultType_Timeout = 0x1001,
            IcmpResultType_EchoReply = 0,
            IcmpResultType_DestinationUnreachable = 3,
            IcmpResultType_SourceQuench = 4,
            IcmpResultType_Redirect = 5,
            IcmpResultType_TimeExceeded = 11,
            IcmpResultType_ParameterProblem = 12
        };

        struct IcmpResult
        {
            IcmpResultType type;
            nn::TimeSpan elapsedTime;
            nn::socket::InAddr responceFrom;
            uint8_t ttl;
            uint16_t sequence;
            nn::socket::SockLenT dataLength;
        };

    public:
        PingTask(const::std::string& dest, size_t length, size_t count, const std::function<void()>& callback);
        virtual ~PingTask();

        virtual void Run();

        PingController GetContoller();

    private:
        int Receive(std::vector<uint8_t>* pOutBuffer, int timeout);
        void PrintResponce(IcmpResult& result);
        void PrintStatistics();
        void PrintSocketError();

    private:
        std::string m_Dest;
        nn::socket::InAddr m_DestIp;
        std::string m_DestHost;
        size_t m_DataLength;
        size_t m_TotalCount;
        std::function<void()> m_Callback;

        std::shared_ptr<int> m_Socket;
        std::shared_ptr<nn::os::TimerEvent> m_pSendTimerEvent;
        uint16_t m_IcmpId;

        std::vector<uint8_t> m_SendBuffer;

        size_t m_CurrentCount;
        size_t m_SuccessCount;
    };

    class IperfStartTask : public Task
    {
    public:
        IperfStartTask();
        virtual ~IperfStartTask();

        virtual void Run();
    };

    class IperfStopTask : public Task
    {
    public:
        IperfStopTask();
        virtual ~IperfStopTask();

        virtual void Run();
    };

    class IperfExitCallbackTask : public Task
    {
    public:
        IperfExitCallbackTask();
        virtual ~IperfExitCallbackTask();

        virtual void Run();
    };

    class Connection
    {
    public:
        virtual ~Connection() {};

        virtual bool Connect() = 0;
        virtual void Disconnect() = 0;
    };

    class NifmConnection : public Connection
    {
    public:
        NifmConnection(const nn::nifm::NetworkProfileData& profileData, bool mixedMode, bool permanentProfile);
        virtual ~NifmConnection();

        virtual bool Connect();
        virtual void Disconnect();

    private:
        void NifmWatchThread();

    private:
        nn::nifm::NetworkConnection* m_pNetworkConnection;
        nn::nifm::TemporaryNetworkProfile* m_pTemporaryNetworkProfile;
        nn::util::Uuid m_ProfileUuid;

        nn::nifm::NetworkProfileData m_ProfileData;
        bool m_MixedMode;
        bool m_PermanentProfile;

        nn::os::Event m_NifmWatchThreadExitEvent;
        nn::os::ThreadType m_NifmWatchThread;
        char *m_NifmWatchThreadStack;
    };

    class WlanConnection : public Connection
    {
    public:
        WlanConnection(nn::wlan::WpsMethod method, nn::bsdsocket::cfg::IfSettings bsdSettings);
        virtual ~WlanConnection();

        virtual bool Connect();
        virtual void Disconnect();

    private:
        nn::wlan::WpsMethod m_WpsMethod;
        nn::bsdsocket::cfg::IfSettings m_BsdSettings;
        std::string m_Pin;
    };

public:
    static Network& GetInstance();
    static void Initialize();
    static void Finalize();

    bool ConnectNetwork(const nn::nifm::NetworkProfileData& profile, bool mixedMode, bool permanentProfile, const std::function<void(bool, const nn::TimeSpan&)>& callback);
    bool DisconnectNetwork();

    bool WpsPbc(nn::nifm::IpSettingData profile, const std::function<void(bool, const nn::TimeSpan&)>& callback);
    bool WpsPin(nn::nifm::IpSettingData profile, const std::function<void(bool, const nn::TimeSpan&)>& callback);

    PingController Ping(const std::string& dest, size_t length, size_t count, const std::function<void()>& callback);
    bool IsRunningPing();

    bool StartIperf(const std::string& commandline, const std::function<void()>& callback);
    bool StopIperf();
    bool IsRunningIperf();

    bool IsConnectionNifm() const;

private:
    Network();
    virtual ~Network();

    void WorkerThread();
    void IperfWorkerThread();

private:
    static Network* g_Instance;
    static NN_OS_ALIGNAS_THREAD_STACK uint8_t g_WorkerThreadStackMemory[];
    static NN_OS_ALIGNAS_THREAD_STACK uint8_t g_IperfThreadStackMemory[];

    nn::os::ThreadType m_WorkerThread;
    bool m_WorkerExitFlag;

    nn::os::ThreadType m_IperfThread;

    BlockingQueue<std::shared_ptr<Task>> m_CommandQueue;

    Connection* m_pConnection;
    bool m_IsNifmConnection;

    std::string m_IperfCommandline;
    std::function<void()> m_IperfExitCallback;
    bool m_RunningIperf;
};

}
