﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/gfx.h>
#include <nn/vi.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

class BackgroundDrawer
{
    NN_DISALLOW_COPY(BackgroundDrawer);
    NN_DISALLOW_MOVE(BackgroundDrawer);
public:
    // コンストラクタ
    BackgroundDrawer() NN_NOEXCEPT;
    // デストラクタ
    ~BackgroundDrawer() NN_NOEXCEPT;

    // 初期化
    // ファイルシステムと Contents がマウントされている必要があります。
    // 本関数内でグラフィックスシステムのためのメモリ周りの初期化を行っています。
    void Initialize() NN_NOEXCEPT;
    // 終了処理
    void Finalize() NN_NOEXCEPT;

    // 背景色の設定(0 ～ 255 の範囲で設定してください)
    void SetBackgroundColor(int r, int g, int b) NN_NOEXCEPT;

    // 背景の描画を行います。描画フレームワーク Immediate で描画しています。
    void Draw() NN_NOEXCEPT;

    // デバッグフォント描画クラスの取得
    nn::gfx::util::DebugFontTextWriter* GetTextWriter() NN_NOEXCEPT;
private:
    static const int ScanBufferCount = 2;
private:
    void InitializeLayer() NN_NOEXCEPT;
    void InitializeDevice() NN_NOEXCEPT;
    void InitializeMemoryPool() NN_NOEXCEPT;
    void InitializeCompressedMemoryPool() NN_NOEXCEPT;
    void InitializeSwapChain() NN_NOEXCEPT;
    void InitializeQueue() NN_NOEXCEPT;
    void InitializeCommandBuffer() NN_NOEXCEPT;
    void InitializeViewport() NN_NOEXCEPT;
    void InitializeSamplerDescriptorPool() NN_NOEXCEPT;
    void InitializeTextureDescriptorPool() NN_NOEXCEPT;
    void InitializeFence() NN_NOEXCEPT;
    void InitializeSemaphore() NN_NOEXCEPT;
    void InitializeTextWriter() NN_NOEXCEPT;
    void ResetCommandBuffer() NN_NOEXCEPT;
private:
    nn::gfx::ClearColorValue m_BackgroundColor;               //!< 背景色
    nn::util::BytePtr m_pMemoryHeap;                          //!< ヒープ
    nn::util::BytePtr m_pMemory;                              //!< メモリ
    nn::gfx::Device        m_Device;                          //!< デバイス
    void*                  m_pMemoryPoolMemory;               //!< メモリプール用メモリ
    void*                  m_pMemoryPoolStart;                //!< メモリプールの開始位置
    ptrdiff_t              m_MemoryPoolOffset;                //!< メモリプールのオフセット
    nn::gfx::MemoryPool    m_MemoryPool;                      //!< メモリプール
    void*                  m_pCompressedMemoryPoolMemory;     //!< 圧縮メモリプール用メモリ
    void*                  m_pCompressedMemoryPoolStart;      //!< 圧縮メモリプールの開始位置
    ptrdiff_t              m_CompressedMemoryPoolOffset;      //!< 圧縮メモリプールのオフセット
    nn::gfx::MemoryPool    m_CompressedMemoryPool;            //!< 圧縮メモリプール
    nn::vi::Display*       m_pDisplay;                        //!< ディスプレイ
    nn::vi::Layer*         m_pLayer;                          //!< レイヤ
    nn::gfx::SwapChain     m_SwapChain;                       //!< スワップチェーン
    nn::gfx::Queue         m_Queue;                           //!< キュー
    nn::gfx::CommandBuffer m_CommandBuffer;                   //!< コマンドバッファ
    void*                  m_pCommandBufferControlMemory;     //!< コマンドバッファのコントロールメモリ
    ptrdiff_t              m_CommandBufferMemoryPoolOffset;   //!< コマンドバッファのメモリプールオフセット
    nn::gfx::ViewportScissorState m_ViewportScissor;          //!< ビューポートシザー
    nn::gfx::DescriptorPool m_SamplerDescriptorPool;          //!< サンプラデスクリプタプール
    int                     m_SamplerDescriptorBaseIndex;     //!< サンプラデスクリプタスロットのインデックス
    nn::gfx::DescriptorPool m_TextureDescriptorPool;          //!< テクスチャデスクリプタプール
    int                     m_TextureDescriptorBaseIndex;     //!< テクスチャデスクリプタスロットのインデックス
    nn::gfx::Fence          m_DisplayFence;                   //!< スキャンバッファ用のフェンス
    nn::gfx::Semaphore      m_DisplaySemaphore;               //!< スキャンバッファ用のセマフォ
    nn::gfx::Fence          m_GpuDoneFence;                   //!< DRAW 待ち用のフェンス
    nn::gfx::ColorTargetView* m_pScanBufferViews[ScanBufferCount]; //!< スキャンバッファへのポインタ
    nn::gfx::util::DebugFontTextWriter m_DebugFontTextWriter; //!< デバッグ用テキスト
    int                     m_NextScanBufferIndex;            //!< 次フレームのスキャンバッファのインデックス
    int                     m_FrameCounter;                   //!< フレーム数
};
