﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "WebOfflineSimple_BackgroundDrawer.hpp"

#include <cstdlib>

#include <nn/nn_Assert.h>
#include <nn/fs.h>

#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
#include <nv/nv_MemoryManagement.h>
#endif

namespace {
// メモリのサイズ
const size_t MemorySize = 16 * 1024 * 1024;
// メモリプールサイズ
const size_t MemoryPoolMemorySize = 16 * 1024 * 1024;
// ディスプレイの幅と高さ
const int DisplayWidth = 1280;
const int DisplayHeight = 720;
// スワップチェーンのフォーマット
const nn::gfx::ImageFormat SwapChainFormat
    = nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm;
// コマンドバッファのメモリサイズ
const size_t CommandBufferControlMemorySize = 256;
const size_t CommandBufferMemoryPoolSize = 1024 * 1024;
// 一度に表示できる最大文字数
const int CharCountMax = 2048;
// デスクリプタスロットの数
const size_t TextureDescriptorPoolSlotCount = 256;


#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
//------------------------------------------------------------------------------
// グラフィックスシステム用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* NvAllocateFunction(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}
static void NvFreeFunction(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}
static void* NvReallocateFunction(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif
}

BackgroundDrawer::BackgroundDrawer() NN_NOEXCEPT
: m_BackgroundColor()
, m_pMemoryHeap(nullptr)
, m_pMemory(nullptr)
, m_Device()
, m_pMemoryPoolMemory(nullptr)
, m_pMemoryPoolStart(nullptr)
, m_MemoryPoolOffset(0)
, m_MemoryPool()
, m_pCompressedMemoryPoolMemory(nullptr)
, m_pCompressedMemoryPoolStart(nullptr)
, m_CompressedMemoryPoolOffset(0)
, m_CompressedMemoryPool()
, m_pDisplay(nullptr)
, m_pLayer(nullptr)
, m_SwapChain()
, m_Queue()
, m_CommandBuffer()
, m_pCommandBufferControlMemory(nullptr)
, m_CommandBufferMemoryPoolOffset(0)
, m_ViewportScissor()
, m_SamplerDescriptorPool()
, m_SamplerDescriptorBaseIndex(0)
, m_TextureDescriptorPool()
, m_TextureDescriptorBaseIndex(0)
, m_DisplayFence()
, m_DisplaySemaphore()
, m_GpuDoneFence()
, m_DebugFontTextWriter()
, m_NextScanBufferIndex(0)
, m_FrameCounter(0)
{
    for( int idx = 0; idx < ScanBufferCount; ++idx )
    {
        m_pScanBufferViews[idx] = nullptr;
    }
    // 背景色は白にしておく
    SetBackgroundColor(255, 255, 255);
}

BackgroundDrawer::~BackgroundDrawer() NN_NOEXCEPT
{
}

void BackgroundDrawer::Initialize() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    {
        const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
        nv::SetGraphicsAllocator(
            NvAllocateFunction, NvFreeFunction, NvReallocateFunction, nullptr);
        nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);
    }
#endif

    // メモリの準備
    m_pMemoryHeap.Reset(malloc(MemorySize));
    m_pMemory = m_pMemoryHeap;

    // レイヤの初期化

    nn::vi::Initialize();
    InitializeLayer();

    // gfxライブラリを初期化
    nn::gfx::Initialize();

    InitializeDevice();

    // Nvn ではデスクリプタスロットのシステムによる予約分がある
    nn::gfx::Device::DataType& deviceData = nn::gfx::AccessorToData(m_Device);
    nvnDeviceGetInteger(deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &m_TextureDescriptorBaseIndex);
    nvnDeviceGetInteger(deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &m_SamplerDescriptorBaseIndex);

    InitializeMemoryPool();
    InitializeCompressedMemoryPool();

    InitializeSwapChain();
    InitializeQueue();

    InitializeCommandBuffer();
    InitializeViewport();

    InitializeSamplerDescriptorPool();
    InitializeTextureDescriptorPool();

    InitializeFence();
    InitializeSemaphore();

    InitializeTextWriter();

    NN_ASSERT(m_pMemoryHeap.Distance(m_pMemory.Get()) < MemorySize);
    NN_ASSERT(m_MemoryPoolOffset < MemoryPoolMemorySize);
    NN_ASSERT(m_CompressedMemoryPoolOffset < MemoryPoolMemorySize);
}

void BackgroundDrawer::Finalize() NN_NOEXCEPT
{
    m_Queue.Sync();

    m_DebugFontTextWriter.Finalize();
    m_DisplaySemaphore.Finalize(&m_Device);
    m_GpuDoneFence.Finalize(&m_Device);
    m_DisplayFence.Finalize(&m_Device);
    m_TextureDescriptorPool.Finalize(&m_Device);
    m_SamplerDescriptorPool.Finalize(&m_Device);
    m_ViewportScissor.Finalize(&m_Device);
    m_CommandBuffer.Finalize(&m_Device);
    m_SwapChain.Finalize(&m_Device);
    m_Queue.Finalize(&m_Device);
    m_CompressedMemoryPool.Finalize(&m_Device);
    m_MemoryPool.Finalize(&m_Device);
    m_Device.Finalize();

    // ライブラリを終了
    nn::gfx::Finalize();

    nn::vi::DestroyLayer(m_pLayer);
    nn::vi::CloseDisplay(m_pDisplay);
    nn::vi::Finalize();

    free(m_pMemoryHeap.Get());
    free(m_pMemoryPoolMemory);
    free(m_pCompressedMemoryPoolMemory);
}

void BackgroundDrawer::InitializeLayer() NN_NOEXCEPT
{
    nn::Result result = nn::vi::OpenDefaultDisplay(&m_pDisplay);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_UNUSED(result);

    result = nn::vi::CreateLayer(&m_pLayer, m_pDisplay);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::vi::SetLayerScalingMode(m_pLayer, nn::vi::ScalingMode_FitToLayer);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void BackgroundDrawer::InitializeDevice() NN_NOEXCEPT
{
    nn::gfx::Device::InfoType info;
    info.SetDefault();
    info.SetApiVersion(nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion);
    m_Device.Initialize(info);
}

void BackgroundDrawer::InitializeMemoryPool() NN_NOEXCEPT
{
    nn::gfx::MemoryPool::InfoType info;
    info.SetDefault();
    info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached
        | nn::gfx::MemoryPoolProperty_GpuCached);

    size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(&m_Device, info);
    m_pMemoryPoolMemory = malloc(MemoryPoolMemorySize + alignment);

    m_pMemoryPoolStart = nn::util::BytePtr(m_pMemoryPoolMemory).AlignUp(alignment).Get();
    info.SetPoolMemory(m_pMemoryPoolStart, nn::util::align_down(MemoryPoolMemorySize,
        nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&m_Device, info)));
    m_MemoryPool.Initialize(&m_Device, info);

    m_MemoryPoolOffset = 0;
}

void BackgroundDrawer::InitializeCompressedMemoryPool() NN_NOEXCEPT
{
    nn::gfx::MemoryPool::InfoType info;
    info.SetDefault();
    info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuCached |
        nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_Compressible);

    size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(&m_Device, info);
    m_pCompressedMemoryPoolMemory = malloc(MemoryPoolMemorySize + alignment);

    m_pCompressedMemoryPoolStart =
        nn::util::BytePtr(m_pCompressedMemoryPoolMemory).AlignUp(alignment).Get();
    info.SetPoolMemory(m_pCompressedMemoryPoolStart, nn::util::align_down(MemoryPoolMemorySize,
        nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&m_Device, info)));
    m_CompressedMemoryPool.Initialize(&m_Device, info);

    m_CompressedMemoryPoolOffset = 0;
}

void BackgroundDrawer::InitializeSwapChain() NN_NOEXCEPT
{
    nn::gfx::SwapChain::InfoType info;

    info.SetDefault();
    info.SetLayer(m_pLayer);
    info.SetWidth(DisplayWidth);
    info.SetHeight(DisplayHeight);
    info.SetFormat(SwapChainFormat);
    info.SetBufferCount(ScanBufferCount);
    if( NN_STATIC_CONDITION(nn::gfx::SwapChain::IsMemoryPoolRequired) )
    {
        size_t size = m_SwapChain.CalculateScanBufferSize(&m_Device, info);
        m_CompressedMemoryPoolOffset = nn::util::align_up(m_CompressedMemoryPoolOffset,
            nn::gfx::SwapChain::GetScanBufferAlignment(&m_Device, info));
        m_SwapChain.Initialize(
            &m_Device, info, &m_CompressedMemoryPool, m_CompressedMemoryPoolOffset, size);
        m_CompressedMemoryPoolOffset += size;
    }
    else
    {
        m_SwapChain.Initialize(&m_Device, info, nullptr, 0, 0);
    }
    NN_ASSERT(nn::gfx::IsInitialized(m_SwapChain));

    m_SwapChain.GetScanBufferViews(m_pScanBufferViews, ScanBufferCount);
}

void BackgroundDrawer::InitializeQueue() NN_NOEXCEPT
{
    nn::gfx::Queue::InfoType info;
    info.SetDefault();
    info.SetCapability(nn::gfx::QueueCapability_Graphics);
    m_Queue.Initialize(&m_Device, info);
    NN_ASSERT(nn::gfx::IsInitialized(m_Queue));
}

void BackgroundDrawer::InitializeCommandBuffer() NN_NOEXCEPT
{
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability(nn::gfx::QueueCapability_Graphics);
    info.SetCommandBufferType(nn::gfx::CommandBufferType_Direct);
    m_CommandBuffer.Initialize(&m_Device, info);

    // コントロールメモリを確
    const size_t alignment = 256;
    m_pMemory.AlignUp(alignment);
    m_pCommandBufferControlMemory = m_pMemory.Get();
    m_pMemory.Advance(CommandBufferControlMemorySize);

    // メモリプール領域確保
    m_MemoryPoolOffset = nn::util::align_up(m_MemoryPoolOffset,
        nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&m_Device));
    m_CommandBufferMemoryPoolOffset = m_MemoryPoolOffset;
    m_MemoryPoolOffset += CommandBufferMemoryPoolSize;
}

void BackgroundDrawer::InitializeViewport() NN_NOEXCEPT
{
    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled(true);
    nn::gfx::ViewportStateInfo viewportInfo;
    {
        viewportInfo.SetDefault();
        viewportInfo.SetWidth(static_cast< float >(DisplayWidth));
        viewportInfo.SetHeight(static_cast< float >(DisplayHeight));
    }
    nn::gfx::ScissorStateInfo scissorInfo;
    {
        scissorInfo.SetDefault();
        scissorInfo.SetWidth(DisplayWidth);
        scissorInfo.SetHeight(DisplayHeight);
    }
    info.SetViewportStateInfoArray(&viewportInfo, 1);
    info.SetScissorStateInfoArray(&scissorInfo, 1);
    m_ViewportScissor.Initialize(&m_Device, info);
    NN_ASSERT(nn::gfx::IsInitialized(m_ViewportScissor));
}

void BackgroundDrawer::InitializeSamplerDescriptorPool() NN_NOEXCEPT
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_Sampler);
    info.SetSlotCount(m_SamplerDescriptorBaseIndex + 1);
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&m_Device, info);
    m_MemoryPoolOffset = nn::util::align_up(m_MemoryPoolOffset,
        nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&m_Device, info));
    m_SamplerDescriptorPool.Initialize(&m_Device, info,
        &m_MemoryPool, m_MemoryPoolOffset, size);
    m_MemoryPoolOffset += size;
}

void BackgroundDrawer::InitializeTextureDescriptorPool() NN_NOEXCEPT
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_TextureView);
    info.SetSlotCount(m_TextureDescriptorBaseIndex + TextureDescriptorPoolSlotCount);
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&m_Device, info);
    m_MemoryPoolOffset = nn::util::align_up(m_MemoryPoolOffset,
        nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&m_Device, info));
    m_TextureDescriptorPool.Initialize(&m_Device, info, &m_MemoryPool, m_MemoryPoolOffset, size);
    m_MemoryPoolOffset += size;
}

void BackgroundDrawer::InitializeFence() NN_NOEXCEPT
{
    nn::gfx::Fence::InfoType info;
    info.SetDefault();
    m_DisplayFence.Initialize(&m_Device, info);
    m_GpuDoneFence.Initialize(&m_Device, info);
}

void BackgroundDrawer::InitializeSemaphore() NN_NOEXCEPT
{
    nn::gfx::Semaphore::InfoType info;
    info.SetDefault();
    m_DisplaySemaphore.Initialize(&m_Device, info);
}

void BackgroundDrawer::InitializeTextWriter() NN_NOEXCEPT
{
    nn::gfx::util::DebugFontTextWriterInfo info;
    info.SetDefault();
    info.SetCharCountMax(CharCountMax);
    info.SetUserMemoryPoolEnabled(true);

    const size_t memorySize =
        nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(&m_Device, info);
    const size_t memoryPoolSize =
        nn::gfx::util::DebugFontTextWriter::GetRequiredMemoryPoolSize(&m_Device, info);

    m_DebugFontTextWriter.Initialize(&m_Device,
        info, m_pMemory.Get(), memorySize, &m_MemoryPool, m_MemoryPoolOffset,
        memoryPoolSize);
    m_DebugFontTextWriter.SetDisplayWidth(DisplayWidth);
    m_DebugFontTextWriter.SetDisplayHeight(DisplayHeight);

    m_pMemory.Advance(memorySize);
    m_MemoryPoolOffset += memoryPoolSize;

    m_DebugFontTextWriter.SetTextureDescriptor(
        &m_TextureDescriptorPool,
        m_TextureDescriptorBaseIndex);
    ++m_TextureDescriptorBaseIndex;
    m_DebugFontTextWriter.SetSamplerDescriptor(
        &m_SamplerDescriptorPool,
        m_SamplerDescriptorBaseIndex);
    ++m_SamplerDescriptorBaseIndex;
}

void BackgroundDrawer::ResetCommandBuffer() NN_NOEXCEPT
{
    m_CommandBuffer.Reset();
    m_CommandBuffer.AddCommandMemory(
        &m_MemoryPool, m_CommandBufferMemoryPoolOffset, CommandBufferMemoryPoolSize);
    m_CommandBuffer.AddControlMemory(
        m_pCommandBufferControlMemory, CommandBufferControlMemorySize);
}

void BackgroundDrawer::SetBackgroundColor(int r, int g, int b) NN_NOEXCEPT
{
    // RGBを設定する（アルファは1で固定）
    static const float s_MaxValue = 255.0f;
    m_BackgroundColor.valueFloat[0] = float(r) / s_MaxValue;
    m_BackgroundColor.valueFloat[1] = float(g) / s_MaxValue;
    m_BackgroundColor.valueFloat[2] = float(b) / s_MaxValue;
    m_BackgroundColor.valueFloat[3] = 1.0f;
}

nn::gfx::util::DebugFontTextWriter* BackgroundDrawer::GetTextWriter() NN_NOEXCEPT
{
    return &m_DebugFontTextWriter;
}

void BackgroundDrawer::Draw() NN_NOEXCEPT
{
    // 描画フレームワーク Immediate で描画しています。
    if( m_FrameCounter == 0 )
    {
        // 0 フレーム目はフレームの頭で次のフレームのスキャンバッファの取得しておく
        nn::gfx::AcquireScanBufferResult acquireResult = m_SwapChain.AcquireNextScanBufferIndex(
            &m_NextScanBufferIndex, &m_DisplaySemaphore, &m_DisplayFence);
        NN_ASSERT(acquireResult == nn::gfx::AcquireScanBufferResult_Success);
        NN_UNUSED(acquireResult);
        // スキャンバッファの取得を同期（GPU）は不要
        // m_Queue.SyncSemaphore(&m_DisplaySemaphore);
        nn::gfx::SyncResult syncResult = m_DisplayFence.Sync(nn::TimeSpan::FromSeconds(1));
        NN_ASSERT(syncResult == nn::gfx::SyncResult_Success);
        NN_UNUSED(syncResult);
    }
    nn::gfx::ColorTargetView* pTarget = m_pScanBufferViews[m_NextScanBufferIndex];

    ResetCommandBuffer();

    m_CommandBuffer.Begin();
    m_CommandBuffer.ClearColorTarget(pTarget, m_BackgroundColor, nullptr);
    m_CommandBuffer.SetRenderTargets(1, &pTarget, nullptr);
    m_CommandBuffer.SetViewportScissorState(&m_ViewportScissor);

    m_CommandBuffer.SetDescriptorPool(&m_TextureDescriptorPool);
    m_CommandBuffer.SetDescriptorPool(&m_SamplerDescriptorPool);

    m_DebugFontTextWriter.Draw(&m_CommandBuffer);

    m_CommandBuffer.End();

    // コマンドの実行
    m_Queue.ExecuteCommand(&m_CommandBuffer, &m_GpuDoneFence);
    m_Queue.Present(&m_SwapChain, 1);

    // 次のフレームのレンダリング対象となるスキャンバッファを取得する
    const nn::gfx::AcquireScanBufferResult acquireResult =
        m_SwapChain.AcquireNextScanBufferIndex(
        &m_NextScanBufferIndex, &m_DisplaySemaphore, &m_DisplayFence);
    NN_ASSERT(acquireResult == nn::gfx::AcquireScanBufferResult_Success);
    NN_UNUSED(acquireResult);
    // スキャンバッファの取得を同期（GPU）は不要
    // m_Queue.SyncSemaphore(&m_DisplaySemaphore);
    // 次のフレームでレンダリング対象となるスキャンバッファが使用可能になるまで待つ
    nn::gfx::SyncResult syncResult = m_DisplayFence.Sync(nn::TimeSpan::FromSeconds(1));
    NN_ASSERT(syncResult == nn::gfx::SyncResult_Success);
    // 現在行っているレンダリングの待ち
    syncResult = m_GpuDoneFence.Sync(nn::TimeSpan::FromSeconds(1));
    NN_ASSERT(syncResult == nn::gfx::SyncResult_Success);

    // Vsync
    ++m_FrameCounter;
}
