(function () {
  class Sound {
    constructor() {
      var AudioContext = window.AudioContext || window.webkitAudioContext;
      this.audioContext_ = new AudioContext();
      this.buffers_ = {};     
      this.playingCount_ = 0; 
      this.onAllEnded_ = null;
      this.playingList_ = {}; 
      this.isLoading_ = false;
      this.suspendedCallbackList_ = [];
      this.resumedCallbackList_ = [];
      var sound = this;
      this.audioContext_.onstatechange = function() {
        sound.onStateChange_();
      }
    }
    /**
     * ファイルをロードする
     * ロードに失敗したファイルはスキップして次のデータを読み込みます。
     * @public
     * @param {!Object} hash      ラベル名とURLの組み合わせで指定します({'se': 'se.wav',...})
     * @param {function()=} opt_onLoadEnd すべてのファイルのロードが完了したときに呼び出されるコールバック
     */
    load(hash, opt_onLoadEnd) {
      if (this.isLoading_) {
        console.log('[Sound]failed to load(already loading)');
        return;
      }
      this.isLoading_ = true;
      this.loadingHash_ = hash;
      this.onLoadEnd_ = opt_onLoadEnd;
      this.loadImpl_();
    }
    /**
     * 指定したラベルの音を再生します。
     * 存在しないラベルを指定した場合は、何も行いません。
     * @public
     * @param {!string}  label   ラベル名
     * @param {function()=} opt_onEnded 再生が完了したときに呼び出されるコールバック
     */
    play(label, opt_onEnded) {
      /// ロードが終わっていないか・エラーで読み込めなかった場合。
      if (!this.buffers_[label]) {
        console.log('[Sound]failed to play(label:' + label.toString() + ')');
        return;
      }
      var bufferSource = this.audioContext_.createBufferSource();
      bufferSource.buffer = this.buffers_[label];
      bufferSource.connect(this.audioContext_.destination);
      
      /// 消えないように連想配列に保存
      this.playingList_[label].push(bufferSource);
      
      var sound = this;
      sound.playingCount_++; ///< 再生数を加算
      bufferSource.onended = function() {
        sound.playingCount_--; ///< 再生が終わったら減算
        sound.playingList_[label].shift(); ///< 参照を消す
        /// コールバックが指定されていたら呼び出す
        if (opt_onEnded) {
          opt_onEnded();
        }
        /// すべての再生が終わっていたらコールバックを呼び出す。
        if (sound.playingCount_ === 0 && sound.onAllEnded_) {
          sound.onAllEnded_();
        }
      }
      bufferSource.start(0);
    }
    
    /**
     * 指定したラベルの音を再生し、再生が終了したら suspend() を実行します。
     * 存在しないラベルを指定した場合は、何も行いません。
     * ページキャッシュを有効にしたまま遷移する場合に用います。
     * @public
     * @param {!string}  label   ラベル名
     * @param {function()=} opt_onEnded サスペンド状態に遷移したときに呼び出されるコールバック
     */
    playWithSuspend(label, opt_onEnded) {
      /// 再生する
      this.play(label, function() {
        /// サスペンドを呼び出す
        sound.suspend(function() {
          /// サスペンド状態に遷移が完了したらコールバックを呼び出す
          if (opt_onEnded) {
            opt_onEnded();
          }
        });
      });
    }
    /**
     * @brief すべての再生が完了したときに呼び出されるコールバックを設定
     * 再生数が0になったタイミングで呼び出されます。
     * 呼び出されるタイミングは、play()のopt_onEndedの後です。
     * @public
     * @param {function()=} callback コールバック
     */
    setOnAllEndedCallback(callback) {
      this.onAllEnded_ = callback;
    }
    /**
     * @brief AudioContext の サスペンド を行います。
     * ページキャッシュを使用する場合は、ページ遷移前に呼び出してください。
     * suspend() の状態遷移は非同期に実行されます。
     * 現在の状態を確認するときは、 getState() を使用してください。
     * getState() == 'suspended' ではない状態でページ遷移すると該当ページがページキャッシュに保存されません。
     * @public
     * @param {function()=} opt_onStateChanged 状態変更が完了したときに呼び出されるコールバック
     */
    suspend(opt_onStateChanged) {
      this.audioContext_.suspend();
      if (opt_onStateChanged) {
        this.suspendedCallbackList_.push(opt_onStateChanged);
        this.onStateChange_();
      }
    }
    /**
     * @brief AudioContext の レジューム を行います。
     * resume() の状態遷移は非同期に実行されます。
     * 現在の状態を確認するときは、 getState() を使用してください。
     * ページキャッシュを使用する場合は、onpageshowイベントのpersisted=trueのときに呼び出してください。
     * @public
     * @param {function()=} opt_onStateChanged 状態変更が完了したときに呼び出されるコールバック
     */
    resume(opt_onStateChanged) {
      this.audioContext_.resume();
      if (opt_onStateChanged) {
        this.resumedCallbackList_.push(opt_onStateChanged);
        this.onStateChange_();
      }
    }
    /**
     * @brief AudioContext の state を取得します。
     * @public
     */
    getState() {
      return this.audioContext_.state;
    }
    
    /**
     * ロードとデコードの内部処理
     * @private
     */
    loadImpl_() {
      /// 全部の処理が終わったか
      if (Object.keys(this.loadingHash_).length <= 0) {
        this.isLoading_ = false;
        if (this.onLoadEnd_) {
          this.onLoadEnd_();
        }
        return;
      }
      /// キーを取り出す
      var key = Object.keys(this.loadingHash_)[0];
      var url = this.loadingHash_[key];
      /// 使ったキーを削除
      delete this.loadingHash_[key];
      
      var sound = this;
      
      /// 読み込み実行
      var request = new XMLHttpRequest();
      request.open('GET', url);
      request.responseType = 'arraybuffer';
      request.onloadend = function(e) {
        var status = request.status;
        var responseData = request.response;
        request = null; ///< メモリ解放対策
        
        /// ロードに失敗
        if ((status !== 200 && status !== 0) || responseData === null || responseData.byteLength === 0) {
          console.log('[Sound]failed to load(label:' + key.toString() + ')');
          /// ロードに失敗した場合は、次のファイルのロードを行う
          sound.loadImpl_();
          return;
        }
        
        /// デコード
        sound.audioContext_.decodeAudioData(responseData, function(decodedData) {
          /// データを保存
          sound.buffers_[key] = decodedData;
          /// プレイングリストを追加
          sound.playingList_[key] = [];
          /// 次のデータを処理する
          sound.loadImpl_();
        }, function() {
          console.log('[Sound]failed to decode(label:' + key.toString() + ')');
          /// 次のデータを処理する
          sound.loadImpl_();
        });
      };
      request.send();
    }
    /**
     * AudioContextのステート変更が発生したときに呼び出される。
     * @private
     */
    onStateChange_() {
      var state = this.audioContext_.state;
      var list = null;
      switch (state) {
      case 'running':
        list = this.resumedCallbackList_;
        break;
      case 'suspended':
        list = this.suspendedCallbackList_;
        break;
      case 'closed':
        break;
      }
      /// 登録されているコールバックを実行する
      if (list !== null) {
        while (list.length > 0) {
          var callback = list.shift();
          callback();
        }
      }
    }
  }
  /// windowのメンバーとして生成する。
  window['sound'] = new Sound();
})();
