﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{WebDemo_Main.cpp,PageSampleWebDemo}
 *
 * @brief
 *  Web アプレット・オフライン Web アプレットの機能の活用方法を示すサンプルプログラム
 */

/**
 * @page PageSampleWebDemo WebDemo
 * @tableofcontents
 *
 * @brief
 *  Web アプレット・オフライン Web アプレットの機能の活用方法を示すサンプルプログラム
 *
 * @section PageSampleWebDemo_SectionBrief 概要
 *  Web アプレット・オフライン Web アプレットの機能の活用方法を示すサンプルプログラムです。
 *
 * @section PageSampleWebDemo_SectionFileStructure ファイル構成
 *  本サンプルプログラムは
 *  @link
 *  ../../../Samples/Sources/Applications/WebDemo Samples/Sources/Applications/WebDemo
 *  @endlink 以下にあります。
 *
 * @section PageSampleWebDemo_SectionNecessaryEnvironment 必要な環境
 *  本サンプルプログラムは NX 実機環境のみサポートしています。
 *
 * @section PageSampleWebDemo_SectionHowToOperate 操作方法
 *  操作方法は、各サンプル Web ページの解説書を参照してください。
 *
 * @section PageSampleWebDemo_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleWebDemo_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleWebDemo_SectionDetail 解説
 *  本サンプルはアプリケーション、複数のサンプル Web ページ、
 *  サンプル Web ページを選択するメニューで構成されます。@n
 *  サンプル Web ページとメニューは、オフライン Web アプレットで表示する
 *  オフライン HTML として用意されています。
 *
 *  本サンプルの基本的な流れは下記の通りです。
 *  - アプリケーションは、先ず画面を灰色で描画します。
 *  - アプリケーションは、オフライン Web アプレットを起動してメニューを表示します。
 *  - メニューで Web ページが選ばれると、選ばれた Web ページの種類をアプリケーションに伝えます。
 *  - アプリケーションは、選ばれた Web ページの種類に応じて、
 *    オフライン Web アプレットを起動して Web ページを表示します。
 *  - Web ページの表示を終えると、オフライン Web アプレットからアプリケーションに戻ります。
 *  - アプリケーションは、再びオフライン Web アプレットを起動してメニューを表示します。
 *
 *  各サンプル Web ページの処理は、アプリケーション部分と Web ページ部分で構成されます。@n
 *  処理の内容は、各サンプル Web ページの解説書を参照してください。
 *
 *  アプリケーションから Web アプレット・オフライン Web アプレットを起動して
 *  Web ページを表示しているとき、
 *  ユーザーが HOME ボタンを押して HOME メニューからシステムアプリを起動すると、
 *  アプリケーションから起動していたアプレットはシステムによって終了してしまいます。@n
 *  アプリケーションは nn::web::OfflineHtmlPageReturnValue::GetOfflineExitReason() を使って
 *  オフライン Web アプレットの終了理由を取得します。@n
 *  システムによって終了していた場合、
 *  再びオフライン Web アプレットを起動して、Web ページを復元します。
 */

#include <cstdint>
#include <cstdio>
#include <cstdlib>

#include <nn/fs.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/web.h>
#include <nn/util/util_Color.h>

#include "WebDemo_GraphicsSystem.hpp"
#include "WebDemo_Pad.hpp"

namespace {

enum SampleId {
    SampleId_InstructionManual,
    SampleId_PauseMenu,
    SampleId_QuickHelp,
    SampleId_RaceResult,
    SampleId_ShootingMovie,
    SampleId_DirectPlayback,
    SampleId_KeyboardEvents,
    SampleId_GamepadApi,
    SampleId_SpatialNavigation
};

enum State {
    State_PrepareToLaunchMenu,
    State_LaunchMenu,
    State_WaitForInput,
    State_PrepareToLaunchSample,
    State_LaunchSample
};

const int PathMaxLength = nn::web::ShowOfflineHtmlPageArg::DocumentPathMaxLength;
const int PrintUsageAnimationFrameLength = 5;

SamplePad g_SamplePad;
GraphicsSystem g_GraphicsSystem;
State g_State = State_PrepareToLaunchMenu;
SampleId g_SampleId = SampleId_InstructionManual;
int g_PrintUsageAnimationFrame = 0;
void* g_pFsCacheBuffer = nullptr;

void* Allocate(size_t size) NN_NOEXCEPT
{
    return malloc(size);
}

void Deallocate(void* ptr, size_t) NN_NOEXCEPT
{
    free(ptr);
}

void InitializeFs() NN_NOEXCEPT
{
    nn::Result result;

    nn::fs::SetAllocator(Allocate, Deallocate);

    /// キャッシュ用のバッファを確保
    size_t cacheSize;
    result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    g_pFsCacheBuffer = malloc(cacheSize);
    /// ROMをマウント
    result = nn::fs::MountRom("Contents", g_pFsCacheBuffer, cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_UNUSED(result);
}

void FinalizeFs() NN_NOEXCEPT
{
    if( g_pFsCacheBuffer )
    {
        free(g_pFsCacheBuffer);
        g_pFsCacheBuffer = nullptr;
    }
    nn::fs::Unmount("Contents");
}

}

void Initialize() NN_NOEXCEPT
{
    srand(static_cast<unsigned int>(time(NULL)));

    InitializeFs();

    /// 背景描画システムの初期化
    g_GraphicsSystem.Initialize();
    /// 背景色の設定（RGB の各値を 0～255 までの範囲で設定できます）
    g_GraphicsSystem.SetBackgroundColor(226, 226, 226);
    /// 背景の種類の設定（背景表示なし。背景色がそのまま表示されます。）
    g_GraphicsSystem.SetBackgroundKind(GraphicsSystem::BackgroundKind_None);
}

void Finalize() NN_NOEXCEPT
{
    g_GraphicsSystem.Finalize();
    FinalizeFs();
}

bool NeedsInputToLaunch() NN_NOEXCEPT
{
    switch( g_SampleId )
    {
    case SampleId_PauseMenu:
    case SampleId_QuickHelp:
        return true;

    default:
        return false;
    }
    return false;
}

void LaunchMenu() NN_NOEXCEPT
{
    const char* documentPathFormat = "menu.htdocs/index.html#item-%d";
    const char* strCallbackUrlStartsWith = "http://localhost/item-";

    bool shouldRestore = false;
    do
    {
        nn::web::OfflineHtmlPageReturnValue menuReturnValue;

        char documentPath[PathMaxLength];
        snprintf(documentPath, sizeof(documentPath), documentPathFormat, g_SampleId);

        nn::web::ShowOfflineHtmlPageArg menuArg(documentPath);
        menuArg.SetBootDisplayKind(nn::web::OfflineBootDisplayKind_CallerCapture);
        menuArg.SetFooterEnabled(false);
        menuArg.SetPointerEnabled(false);
        menuArg.SetJsExtensionEnabled(true);

        nn::Result result;
        result = nn::web::ShowOfflineHtmlPage(&menuReturnValue, menuArg);
        NN_ABORT_UNLESS(result.IsSuccess());

        nn::web::OfflineExitReason exitReason = menuReturnValue.GetOfflineExitReason();
        shouldRestore =
            (exitReason == nn::web::OfflineExitReason_ExitMessage ||
            exitReason == nn::web::OfflineExitReason_BackButtonPressed);

        if( exitReason == ::nn::web::OfflineExitReason_CallbackUrlReached )
        {
            g_SampleId = static_cast<SampleId>(atoi(
                menuReturnValue.GetLastUrl() + strnlen(strCallbackUrlStartsWith, PathMaxLength)
                ));
        }
    } while ( shouldRestore );
}

void LaunchInstructionManual() NN_NOEXCEPT
{
    const char* documentPath = "sample/instruction_manual.htdocs/index.html";
    const char* strWebJumpUrlStartsWith = "http://localhost/?href=";
    int strlenWebJumpUrlStartsWith = strnlen(strWebJumpUrlStartsWith, PathMaxLength);

    bool shouldRestore = false;
    do
    {
        nn::web::OfflineHtmlPageReturnValue sampleReturnValue;

        nn::web::ShowOfflineHtmlPageArg sampleArg(documentPath);
        sampleArg.SetBootDisplayKind(nn::web::OfflineBootDisplayKind_CallerCapture);
        sampleArg.SetFooterEnabled(false);
        sampleArg.SetPointerEnabled(false);
        sampleArg.SetJsExtensionEnabled(true);

        nn::Result result;
        result = nn::web::ShowOfflineHtmlPage(&sampleReturnValue, sampleArg);
        NN_ABORT_UNLESS(result.IsSuccess());

        nn::web::OfflineExitReason exitReason = sampleReturnValue.GetOfflineExitReason();


        if( exitReason == nn::web::OfflineExitReason_CallbackUrlReached &&
            strncmp(
            sampleReturnValue.GetLastUrl(),
            strWebJumpUrlStartsWith,
            strlenWebJumpUrlStartsWith
            ) == 0 )
        {
            // Webアプレットの呼び出し
            nn::web::WebPageReturnValue webReturnValue;
            const char* url = sampleReturnValue.GetLastUrl() + strlenWebJumpUrlStartsWith;
            nn::web::ShowWebPageArg webArg(url);
            nn::web::ShowWebPage(&webReturnValue, webArg);
            shouldRestore = true;
        }
        else
        {
            shouldRestore = (sampleReturnValue.GetOfflineExitReason()
                == nn::web::OfflineExitReason_ExitMessage);
        }
    } while ( shouldRestore );
}

bool LaunchPauseMenu() NN_NOEXCEPT
{
    const char* documentPath = "sample/pause_menu.htdocs/index.html";
    const char* strExitUrlStartsWith = "http://localhost/?id=";
    const int strlenExitUrlStartsWith = strnlen(strExitUrlStartsWith, PathMaxLength);

    bool shouldRestore = false;
    do
    {
        nn::web::OfflineHtmlPageReturnValue sampleReturnValue;

        nn::web::ShowOfflineHtmlPageArg sampleArg(documentPath);
        sampleArg.SetBackgroundKind(nn::web::OfflineBackgroundKind_ApplicationCaptureBlur);
        sampleArg.SetFooterEnabled(false);
        sampleArg.SetPointerEnabled(false);
        sampleArg.SetJsExtensionEnabled(true);
        sampleArg.SetWebAudioEnabled(true);

        nn::Result result;
        result = nn::web::ShowOfflineHtmlPage(&sampleReturnValue, sampleArg);
        NN_ABORT_UNLESS(result.IsSuccess());

        nn::web::OfflineExitReason exitReason = sampleReturnValue.GetOfflineExitReason();

        if( exitReason == nn::web::OfflineExitReason_CallbackUrlReached &&
            strncmp(sampleReturnValue.GetLastUrl(),
                strExitUrlStartsWith, strlenExitUrlStartsWith) == 0 )
        {
            int id = atoi(
                sampleReturnValue.GetLastUrl() + strnlen(strExitUrlStartsWith, PathMaxLength)
                );
            switch( id )
            {
            case 0:
                {
                    NN_LOG("0: Exit Stage\n");
                    return true;
                }

            case 1:
                {
                    NN_LOG("1: Exit Game\n");
                    return true;
                }

            default: NN_UNEXPECTED_DEFAULT;
            }
        }

        shouldRestore =
            (sampleReturnValue.GetOfflineExitReason() == nn::web::OfflineExitReason_ExitMessage);

    } while ( shouldRestore );

    return false;
}

void LaunchQuickHelp() NN_NOEXCEPT
{
    const char* documentPath =
        "sample/quick_help.htdocs/index.html"
        "?data=["
        "[0,0,140,392,60,50],"
        "[0,1,290,122,60,50],"
        "[0,2,28,604,264,72],"
        "[0,3,316,604,264,72],"
        "[1,0,140,442,114,134],"
        "[1,1,290,176,128,142],"
        "[2,0,462,212,140,108],"
        "[2,1,806,334,302,242],"
        "[3,0,640,190,160,20],"
        "[3,1,1134,432,98,144],"
        "[4,0,1126,26,78,70]"
        "]";

    bool shouldRestore = false;
    do
    {
        nn::web::OfflineHtmlPageReturnValue sampleReturnValue;

        nn::web::ShowOfflineHtmlPageArg sampleArg(documentPath);
        sampleArg.SetBackgroundKind(nn::web::OfflineBackgroundKind_ApplicationCapture);
        sampleArg.SetFooterEnabled(false);
        sampleArg.SetPointerEnabled(false);
        sampleArg.SetJsExtensionEnabled(true);

        nn::Result result;
        result = nn::web::ShowOfflineHtmlPage(&sampleReturnValue, sampleArg);
        NN_ABORT_UNLESS(result.IsSuccess());

        shouldRestore =
            (sampleReturnValue.GetOfflineExitReason() == nn::web::OfflineExitReason_ExitMessage);

    } while ( shouldRestore );
}

void LaunchRaceResult() NN_NOEXCEPT
{
    const char* documentPathFormat =
        "sample/race_result.htdocs/index.html"
        "?stage=1&course=0&car=0"
        "&player=%d"""
        "&ranking="
        "Speed King,10:09.412|"
        "Ultra G,11:48.025|"
        "Mr. Topgear,13:56.301|"
        "Octane 300,14:07.214|"
        "AAAaa,14:52.322|"
        "Taro0423,15:14.411|"
        "Mike56,16:48.546"
        "%s";
    const char* flagmentForRestore = "#skip-intro";

    bool shouldRestore = false;
    int playerId = rand() % 4;
    if( playerId == 3 )
    {
        playerId = 3 + rand() % 4;
    }
    do
    {
        nn::web::OfflineHtmlPageReturnValue sampleReturnValue;

        char documentPath[PathMaxLength];
        snprintf(
            documentPath,
            sizeof(documentPath),
            documentPathFormat,
            playerId,
            shouldRestore ? flagmentForRestore : ""
            );
        nn::web::ShowOfflineHtmlPageArg sampleArg(documentPath);
        sampleArg.SetBootDisplayKind(nn::web::OfflineBootDisplayKind_CallerCapture);
        sampleArg.SetFooterEnabled(false);
        sampleArg.SetJsExtensionEnabled(true);
        sampleArg.SetPointerEnabled(false);

        nn::Result result;
        result = nn::web::ShowOfflineHtmlPage(&sampleReturnValue, sampleArg);
        NN_ABORT_UNLESS(result.IsSuccess());

        shouldRestore =
            (sampleReturnValue.GetOfflineExitReason() == nn::web::OfflineExitReason_ExitMessage);

    } while ( shouldRestore );
}

void LaunchShootingMovie() NN_NOEXCEPT
{
    const char* documentPath = "sample/shooting_movie.htdocs/index.html";

    bool shouldRestore = false;
    do
    {
        nn::web::OfflineHtmlPageReturnValue sampleReturnValue;

        nn::web::ShowOfflineHtmlPageArg sampleArg(documentPath);
        sampleArg.SetBootDisplayKind(nn::web::OfflineBootDisplayKind_CallerCapture);
        sampleArg.SetFooterEnabled(false);
        sampleArg.SetJsExtensionEnabled(true);
        sampleArg.SetPointerEnabled(false);
        sampleArg.SetMediaPlayerAutoCloseEnabled(true);

        nn::Result result;
        result = nn::web::ShowOfflineHtmlPage(&sampleReturnValue, sampleArg);
        NN_ABORT_UNLESS(result.IsSuccess());

        shouldRestore =
            (sampleReturnValue.GetOfflineExitReason() == nn::web::OfflineExitReason_ExitMessage);

    } while ( shouldRestore );
}

void LaunchDirectPlayback() NN_NOEXCEPT
{
    const char* documentPath = "sample/shooting_movie.htdocs/img/shooting_movie.mp4";

    bool shouldRestore = false;
    do
    {
        nn::web::OfflineHtmlPageReturnValue sampleReturnValue;

        nn::web::ShowOfflineHtmlPageArg sampleArg(documentPath);
        sampleArg.SetBootAsMediaPlayer(true);
        sampleArg.SetMediaPlayerAutoCloseEnabled(true);
        sampleArg.SetBootDisplayKind(nn::web::OfflineBootDisplayKind_CallerCapture);

        nn::Result result;
        result = nn::web::ShowOfflineHtmlPage(&sampleReturnValue, sampleArg);
        NN_ABORT_UNLESS(result.IsSuccess());

        shouldRestore =
            (sampleReturnValue.GetOfflineExitReason() == nn::web::OfflineExitReason_ExitMessage);

    } while ( shouldRestore );
}

void LaunchKeyboardEvents() NN_NOEXCEPT
{
    const char* documentPath = "sample/keyboard_events.htdocs/index.html";

    bool shouldRestore = false;
    do
    {
        nn::web::OfflineHtmlPageReturnValue sampleReturnValue;

        nn::web::ShowOfflineHtmlPageArg sampleArg(documentPath);
        sampleArg.SetBootDisplayKind(nn::web::OfflineBootDisplayKind_CallerCapture);
        sampleArg.SetPointerEnabled(false);
        sampleArg.SetJsExtensionEnabled(true);

        nn::Result result;
        result = nn::web::ShowOfflineHtmlPage(&sampleReturnValue, sampleArg);
        NN_ABORT_UNLESS(result.IsSuccess());

        shouldRestore =
            (sampleReturnValue.GetOfflineExitReason() == nn::web::OfflineExitReason_ExitMessage);

    } while ( shouldRestore );
}

void LaunchGamepadApi() NN_NOEXCEPT
{
    const char* documentPath = "sample/gamepad_api.htdocs/index.html";

    bool shouldRestore = false;
    do
    {
        nn::web::OfflineHtmlPageReturnValue sampleReturnValue;

        nn::web::ShowOfflineHtmlPageArg sampleArg(documentPath);
        sampleArg.SetBootDisplayKind(nn::web::OfflineBootDisplayKind_CallerCapture);
        sampleArg.SetPointerEnabled(false);
        sampleArg.SetJsExtensionEnabled(true);

        nn::Result result;
        result = nn::web::ShowOfflineHtmlPage(&sampleReturnValue, sampleArg);
        NN_ABORT_UNLESS(result.IsSuccess());

        shouldRestore =
            (sampleReturnValue.GetOfflineExitReason() == nn::web::OfflineExitReason_ExitMessage);

    } while ( shouldRestore );
}

void LaunchSpatialNavigation() NN_NOEXCEPT
{
    const char* documentPath = "sample/spatial_navigation.htdocs/index.html";

    bool shouldRestore = false;
    do
    {
        nn::web::OfflineHtmlPageReturnValue sampleReturnValue;

        nn::web::ShowOfflineHtmlPageArg sampleArg(documentPath);
        sampleArg.SetBootDisplayKind(nn::web::OfflineBootDisplayKind_CallerCapture);
        sampleArg.SetPointerEnabled(false);
        sampleArg.SetTouchEnabledOnContents(false);

        nn::Result result;
        result = nn::web::ShowOfflineHtmlPage(&sampleReturnValue, sampleArg);
        NN_ABORT_UNLESS(result.IsSuccess());

        shouldRestore =
            (sampleReturnValue.GetOfflineExitReason() == nn::web::OfflineExitReason_ExitMessage);

    } while ( shouldRestore );
}

bool LaunchSample() NN_NOEXCEPT
{
    switch( g_SampleId )
    {
    case SampleId_InstructionManual:
        {
            LaunchInstructionManual();
        }
        break;

    case SampleId_PauseMenu:
        {
            const bool result = LaunchPauseMenu();
            return result;
        }

    case SampleId_QuickHelp:
        {
            LaunchQuickHelp();
            return false;
        }

    case SampleId_RaceResult:
        {
            LaunchRaceResult();
        }
        break;


    case SampleId_ShootingMovie:
        {
            LaunchShootingMovie();
        }
        break;

    case SampleId_DirectPlayback:
        {
            LaunchDirectPlayback();
        }
        break;

    case SampleId_KeyboardEvents:
        {
            LaunchKeyboardEvents();
        }
        break;

    case SampleId_GamepadApi:
        {
            LaunchGamepadApi();
        }
        break;

    case SampleId_SpatialNavigation:
        {
            LaunchSpatialNavigation();
        }
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    return true;
}

void ChangeBackgroundToDefault() NN_NOEXCEPT
{
    g_GraphicsSystem.SetBackgroundKind(GraphicsSystem::BackgroundKind_None);
}

void ChangeBackgroundForSample() NN_NOEXCEPT
{
    GraphicsSystem::BackgroundKind backgroundKind = GraphicsSystem::BackgroundKind_None;

    switch( g_SampleId )
    {
    case SampleId_PauseMenu:
        {
            backgroundKind = GraphicsSystem::BackgroundKind_PauseGame;
        }
        break;

    case SampleId_QuickHelp:
        {
            backgroundKind = GraphicsSystem::BackgroundKind_HelpGame;
        }
        break;

    default:
        {
            backgroundKind = GraphicsSystem::BackgroundKind_None;
        }
        break;
    }

    g_GraphicsSystem.SetBackgroundKind(backgroundKind);
}

void PrintUsage(int r, int g, int b) NN_NOEXCEPT
{
    NN_ASSERT(NeedsInputToLaunch());

    const char* strFirstLine = "You are playing this game.";
    const char* strSecondLine = nullptr;

    switch( g_SampleId )
    {
    case SampleId_PauseMenu:
        {
            strSecondLine = "[B] Exit  [+/-] Pause";
        }
        break;

    case SampleId_QuickHelp:
        {
            strSecondLine = "[B] Exit  [+/-] Help";
        }
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    if( 0 < g_PrintUsageAnimationFrame )
    {
        float rate = float(g_PrintUsageAnimationFrame) / float(PrintUsageAnimationFrameLength);
        rate = -2.f * rate * rate + 3.f * rate;
        rate = 0.8f + 0.2f * rate;

        nn::gfx::util::DebugFontTextWriter* pWriter = g_GraphicsSystem.GetTextWriter();
        NN_ASSERT_NOT_NULL(pWriter);

        pWriter->SetFontSize(36.f * rate);

        const float centerX = 640.f;
        const float centerY = 60.0f;

        float firstLineWidth = pWriter->CalculateStringWidth(strFirstLine);
        float firstLineTop = centerY - (10.f + pWriter->CalculateStringHeight(strFirstLine)) * 0.5f;
        float secondLineWidth = pWriter->CalculateStringWidth(strSecondLine);
        float secondLineTop = centerY + (10.f + pWriter->CalculateStringHeight(strSecondLine)) * 0.5f;
        float secondLineBottom = secondLineTop + pWriter->CalculateStringHeight(strSecondLine);

        float rectWidth =
            (firstLineWidth > secondLineWidth ? firstLineWidth : secondLineWidth) + 40.f * rate;
        float rectHeight = (secondLineBottom - firstLineTop) + 20.f * rate;
        const float rectOffsetY = 20.f;

        GraphicsSystem::RectanglePositions positions;
        g_GraphicsSystem.SetRectangleVisible(true);
        g_GraphicsSystem.SetRectangleColor(r, g, b, 96 + int(64.f * rate));
        positions.leftTop = NN_UTIL_FLOAT_2_INITIALIZER(
            centerX - rectWidth * 0.5f, centerY - rectHeight * 0.5f + rectOffsetY);
        positions.rightTop = NN_UTIL_FLOAT_2_INITIALIZER(
            centerX + rectWidth * 0.5f, centerY - rectHeight * 0.5f + rectOffsetY);
        positions.leftBottom = NN_UTIL_FLOAT_2_INITIALIZER(
            centerX - rectWidth * 0.5f, centerY + rectHeight * 0.5f + rectOffsetY);
        positions.rightBottom = NN_UTIL_FLOAT_2_INITIALIZER(
            centerX + rectWidth * 0.5f, centerY + rectHeight * 0.5f + rectOffsetY);
        g_GraphicsSystem.SetRectanglePositions(positions);

        pWriter->SetTextColor(nn::util::Color4u8(255, 255, 255, 255));
        pWriter->SetCursor(centerX - firstLineWidth * 0.5f, firstLineTop);
        pWriter->Print(strFirstLine);

        pWriter->SetCursor(centerX - secondLineWidth * 0.5f, secondLineTop);
        pWriter->Print(strSecondLine);
    }
    else
    {
        g_GraphicsSystem.SetRectangleVisible(false);
    }
}

void Update() NN_NOEXCEPT
{
    g_SamplePad.Update();

    switch( g_State )
    {
    case State_PrepareToLaunchMenu:
        {
            ChangeBackgroundToDefault();
            g_GraphicsSystem.SetRectangleVisible(false);

            g_State = State_LaunchMenu;
        }
        break;

    case State_LaunchMenu:
        {
            LaunchMenu();

            if( NeedsInputToLaunch() )
            {
                g_PrintUsageAnimationFrame = 0;

                g_State = State_WaitForInput;
            }
            else
            {
                g_State = State_PrepareToLaunchSample;
            }
        }
        break;

    case State_WaitForInput:
        {
            ChangeBackgroundForSample();
            PrintUsage(0, 0, 0);

            if( g_SamplePad.IsTrigger<nn::hid::DebugPadButton::Start>() ||
                g_SamplePad.IsTrigger<nn::hid::DebugPadButton::Select>() )
            {
                g_State = State_PrepareToLaunchSample;
            }
            else if( g_SamplePad.IsTrigger<nn::hid::DebugPadButton::B>() )
            {
                g_State = State_PrepareToLaunchMenu;
            }

            if( g_PrintUsageAnimationFrame < PrintUsageAnimationFrameLength ) {
                ++g_PrintUsageAnimationFrame;
            }
        }
        break;

    case State_PrepareToLaunchSample:
        {
            ChangeBackgroundForSample();

            if( NeedsInputToLaunch() ) {
                PrintUsage(0, 128, 200);

                if( 0 < g_PrintUsageAnimationFrame )
                {
                    --g_PrintUsageAnimationFrame;
                }
                else if( g_PrintUsageAnimationFrame == 0 )
                {
                    g_State = State_LaunchSample;
                }
            }
            else {
                g_State = State_LaunchSample;
            }
        }
        break;

    case State_LaunchSample:
        {
            bool shouldExit = LaunchSample();
            if( shouldExit )
            {
                g_State = State_PrepareToLaunchMenu;
            }
            else
            {
                g_PrintUsageAnimationFrame = 0;

                g_State = State_WaitForInput;
            }
        }
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    g_GraphicsSystem.Draw();
}

//
//  Main Function
//  メイン関数です。
//

extern "C" void nnMain() NN_NOEXCEPT
{
    Initialize();

    for(;;)
    {
        Update();
    }

    Finalize();
}
