﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/vi.private.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/ae.h>
#include "Controller.h"
#include "MenuManager.h"
#include "RgbMenu.h"
#include "ModeMenu.h"
#include "UnderscanMenu.h"
#include "CmuModeMenu.h"
#include "ContrastRatioMenu.h"
#include "GammaMenu.h"

/**
 * @examplesource{ViMode.cpp,PageSampleViMode}
 *
 * @brief This sample demonstrates changing display modes with the nn::vi library.
 */

/**
 * @page PageSampleViMode ViMode
 * @tableofcontents
 *
 * @brief Sample program to instruct how to change the external display modes.
 *
 * @section ViMode_SectionBrief Summary
 *  The nn::vi library exposes the ability to change various display modes.  Please note
 *  that the exact behavior will depend on the display being used.
 *
 *  Currently, this sample has the ability to change resolution, RGB range, and the underscan used by
 *  an external display.
 *
 * @section ViMode_SectionFileStructure File List
 *  Please refer to @link ../../../Samples/Sources/Applications/ViMode Samples/Sources/Applications/ViMode @endlink
 *
 * @section ViMode_SectionNecessaryEnvironment Necessary Environment
 *  Please connect a HDMI-compliant display to the NX via the cradle.  This sample also uses a debug pad
 *  to navigate the menu.
 *
 * @section ViMode_SectionHowToOperate Operating Procedure
 *  Please launch this sample with TargetManager or Visual Studio.
 *
 *  A menu will appear in the output log.  Use the Up and Down buttons on the directional pad and the
 *  A button to confirm the selection.  To switch between the various menus, use the L and R buttons.
 *
 * @section ViMode_SectionPrecaution Precautions
 *  The display driver does not support gamma settings at this time.  The menu is functional, but
 *  no operations will occur.
 *
 * @section ViMode_SectionHowToExecute How To Execute
 *  Please build and run the sample.
 *
 * @section ViMode_SectionDetail Details
 *
 * @subsection ViMode_SectionSampleProgram Sample Program
 *  The main sample source code is linked below.
 *
 *  ViMode.cpp
 *  @includelineno ViMode.cpp
 *
 * @subsection ViMode_SectionSampleDetail Sample Program Details
 *
 *  This sample will proceed through a slightly different initialization sequence for nn::vi:
 *  - Explicitly initialize nn::vi for private functionality
 *  - Open the External display
 *  - Enumerate the current and available modes on the connected display
 *
 *  For NX, the Default display will either attempt to change the mode of either the LCD or HDMI display.
 *  This behavior is dependent on the whether the NX device is set to console or handheld
 *  mode.
 *
 *  A new External display is available that will always attempt to target a display connected
 *  via HDMI.  It is recommended to use this display for changing modes on NX.
 *
 *  From a design standpoint, nn::vi uses the following naming conventions regarding display modes:
 *
 *  - nn::vi::List* returns the capabilities of the display
 *  - nn::vi::Get* returns the current setting of the display
 *  - nn::vi::Set* changes the current setting of the display
 *
 *  The sample has evolved a bit since its first incarnation.  The nn::vi function calls are scattered
 *  throughout the source code.  For examples of the nn::vi::List* and nn::vi::Get* APIs, please check these files:
 *
 *  - CmuModeMenu.cpp @includelineno CmuModeMenu.cpp
 *  - ContrastRatioMenu.cpp @includelineno ContrastRatioMenu.cpp
 *  - GammaMenu.cpp @includelineno GammaMenu.cpp
 *  - ModeMenu.cpp @includelineno ModeMenu.cpp
 *  - RgbMenu.cpp @includelineno RgbMenu.cpp
 *  - UnderscanMenu.cpp @includelineno UnderscanMenu.cpp
 *
 *  For examples of the nn::vi::Set* APIs, please check these files:
 *
 *  - CmuModeMenuItem.cpp @includelineno CmuModeMenuItem.cpp
 *  - ContrastRatioMenuItem.cpp @includelineno ContrastRatioMenuItem.cpp
 *  - GammaMenuItem.cpp @includelineno GammaMenuItem.cpp
 *  - ModeMenuItem.cpp @includelineno ModeMenuItem.cpp
 *  - RgbMenuItem.cpp @includelineno ModeMenuItem.cpp
 *  - UnderscanMenuItem.cpp @includelineno UnderscanMenuItem.cpp
 *
 */

namespace
{
    struct SwitchOperationModeData
    {
        nn::vi::Display* pDefaultDisplay;
        nn::vi::Display* pExternalDisplay;
    };

    char NN_OS_ALIGNAS_THREAD_STACK g_AeMessageThreadStack[16384];

    void HandleAeMessage(void* args) NN_NOEXCEPT
    {
        MenuManager* pManager = static_cast<MenuManager*>(args);

        nn::os::SystemEventType messageEvent;
        nn::ae::InitializeNotificationMessageEvent(&messageEvent);

        for(;;)
        {
            nn::ae::Message message = nn::ae::WaitForNotificationMessage(&messageEvent);

            switch (message)
            {
            case nn::ae::Message_OperationModeChanged:
                pManager->SetOperationMode(nn::ae::GetOperationMode());
                break;
            default:
                break;
            }
        }

        nn::os::DestroySystemEvent(&messageEvent);
    }

    void SwitchOperationMode(nn::ae::OperationMode mode, MenuManager::AddMenuHandler handler, void* userData) NN_NOEXCEPT
    {
        SwitchOperationModeData* data = static_cast<SwitchOperationModeData*>(userData);

        handler(std::make_unique<CmuModeMenu>(data->pDefaultDisplay));
        handler(std::make_unique<ContrastRatioMenu>(data->pDefaultDisplay));

        switch( mode )
        {
        case nn::ae::OperationMode_Console:
            handler(std::make_unique<GammaMenu>(data->pExternalDisplay));
            handler(std::make_unique<ModeMenu>(data->pExternalDisplay));
            handler(std::make_unique<RgbMenu>(data->pExternalDisplay));
            handler(std::make_unique<UnderscanMenu>(data->pExternalDisplay));
            break;
        case nn::ae::OperationMode_Handheld:
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

void ViModeMain(nn::ae::OverlayAppletParameters* param)
{
    NN_UNUSED(param);
    nn::vi::Initialize();

    nn::vi::DisplayInfo displays[3];
    // Currently three named displays on HOS.
    int displayCount = nn::vi::ListDisplays(displays, sizeof(displays) / sizeof(displays[0]));
    NN_ASSERT(3 == displayCount, "Failed to enumerate displays.");

    // There are additional named displays.  They have different policies from the Default
    // display.
    for( int i = 0; i < displayCount; ++i )
    {
        NN_LOG("Display Name    : %s\n", displays[i].name);

        if( displays[i].hasLayerLimit )
        {
            // Process can only create this many layers on this display.
            NN_LOG("Layer Limit     : %d\n", displays[i].layerCountMax);
        }

        // Max dimensions of a layer (in pixels) on the default display.
        NN_LOG("Layer Max Width : %d\n", displays[i].layerWidthPixelCountMax);
        NN_LOG("Layer Max Height: %d\n", displays[i].layerHeightPixelCountMax);

        NN_LOG("\n");
    }

    nn::vi::Display* pDefaultDisplay;
    {
        nn::Result result = nn::vi::OpenDefaultDisplay(&pDefaultDisplay);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    nn::vi::Display* pExternalDisplay;
    // This is the new display type that will always attempt to target a display connected
    // via HDMI.  The External display can always be opened regardless of the current hotplug
    // state, but certain functions may fail if the display isn't connected.
    // It is important to note that this display cannot be used to create layers.
    {
        nn::Result result = nn::vi::OpenDisplay(&pExternalDisplay, "External");
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    SwitchOperationModeData data;
    data.pExternalDisplay = pExternalDisplay;
    data.pDefaultDisplay  = pDefaultDisplay;
    MenuManager menus(SwitchOperationMode, &data);

    nn::os::ThreadType aeMessageThread;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&aeMessageThread,
                                                        HandleAeMessage,
                                                        &menus,
                                                        g_AeMessageThreadStack,
                                                        sizeof(g_AeMessageThreadStack),
                                                        nn::os::DefaultThreadPriority));
    nn::os::StartThread(&aeMessageThread);

    Controller controller;

    nn::ae::BeginOverlayUserInteraction();
    while( true )
    {
        controller.Update();

        menus.Update(controller);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }
    nn::ae::EndOverlayUserInteraction();

    nn::vi::Finalize();
}

extern "C" void nnMain()
{
    nn::ae::InvokeOverlayAppletMain(nn::ae::AppletId_OverlayApplet, ViModeMain);
}
