﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include "MenuManager.h"
#include "Controller.h"
#include "IMenu.h"

MenuManager::AddMenuHandler::AddMenuHandler(MenuManager* manager)
    : m_Manager(manager)
{
}

void MenuManager::AddMenuHandler::operator()(std::unique_ptr<IMenu>&& menu) NN_NOEXCEPT
{
    m_Manager->Add(std::move(menu));
}

MenuManager::MenuManager(OperationModeHandler handler, void* pUserData) NN_NOEXCEPT
    : m_MenuIndex(0)
    , m_Lock(false)
    , m_Handler(handler)
    , m_pUserData(pUserData)
{
    SetOperationMode(nn::ae::GetOperationMode());
}

void MenuManager::Update(const Controller& input) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);

    if( m_Menus.size() != 0 )
    {
        if( input.IsPressed<nn::hid::DebugPadButton::L>() )
        {
            // not using modulus due to unsigned underflow potential
            if( m_MenuIndex != 0 )
            {
                --m_MenuIndex;
            }
            else
            {
                m_MenuIndex = m_Menus.size() - 1;
            }

            RenderUnsafe();
        }

        if( input.IsPressed<nn::hid::DebugPadButton::R>() )
        {
            m_MenuIndex = (m_MenuIndex + 1) % m_Menus.size();

            RenderUnsafe();
        }

        m_Menus[m_MenuIndex]->Update(input);
    }
}

void MenuManager::Render() const NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);
    RenderUnsafe();
}

void MenuManager::RenderUnsafe() const NN_NOEXCEPT
{
    if( m_Menus.size() != 0 )
    {
        m_Menus[m_MenuIndex]->Refresh();
    }
}

void MenuManager::Add(std::unique_ptr<IMenu>&& menu) NN_NOEXCEPT
{
    m_Menus.push_back(std::move(menu));
}

void MenuManager::SetOperationMode(nn::ae::OperationMode mode) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Lock);

    m_Menus.clear();
    m_MenuIndex = 0;

    if( m_Handler != nullptr )
    {
        AddMenuHandler addHandler(this);
        m_Handler(mode, addHandler, m_pUserData);
    }

    RenderUnsafe();
}
