﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 *  社内向けにカスタム（アクション・シェーダ・フィールド）で作る機能の検証環境用サンプル。
 *  非公開サンプルなのでこの部分の Doxygen コメントは書かない。
 */

#pragma once

#include "Resources/CustomAction/CustomAction1.h"
#include "Resources/CustomShader/CustomShader1.h"

namespace {

inline float Clamp( float x, float low, float high )
{
    if( x < low ) { return low; }
    else if( x > high ) { return high; }
    return x;
}

inline float ClampAbs1( float x )
{
    return Clamp( x, -1.0f, 1.0f );
}

nn::util::Vector3f g_GlobalWindVec( 0.1f, 0, 0 );

//------------------------------------------------------------------------------
//  「風の影響を受ける」の CPU 実装
//------------------------------------------------------------------------------
inline bool CustomFieldCallback(
    nn::util::Vector3fType* pOutPos,
    nn::util::Vector3fType* pOutVec,
    float* pOutLife,
    float* pOutBirthTime,
    nn::vfx::Emitter* pEmitter,
    const nn::vfx::detail::ParticleProperty* pParticleProperty,
    const nn::vfx::detail::ResFieldCustom* pCustomFieldData,
    int particleIndex )
{
    NN_UNUSED( pOutVec );
    NN_UNUSED( pOutLife );
    NN_UNUSED( pOutBirthTime );
    NN_UNUSED( pEmitter );
    NN_UNUSED( pParticleProperty );
    NN_UNUSED( particleIndex );

    // カスタムフィールドサンプル：グローバル風
    if( ( pCustomFieldData->flags & 0x1 ) == 0x1 )
    {
        // ゲーム内の風速をパーティクルの移動量に適用する。
        // TODO: ゲーム内の風速をグローバルで持って ComputeShader にも送る
        nn::util::VectorMultiply( &g_GlobalWindVec, g_GlobalWindVec, pCustomFieldData->value0 );
        nn::util::VectorAdd( pOutPos, *pOutPos, g_GlobalWindVec );
    }
    return true;
}

//------------------------------------------------------------------------------
//  パーティクル放出コールバック
//------------------------------------------------------------------------------
inline bool ParticleEmitCallback( nn::vfx::ParticleEmitArg& arg )
{
    const ResCustomActionSample* res = reinterpret_cast< const ResCustomActionSample* >( arg.pEmitter->GetCustomActionParameter() );
    if( ( res->choices & 0x4 ) == 0x4 || ( res->choices & 0x1 ) == 0x1 )
    {
        //------------------------------------------------------------------------------
        // 「粒の初速を向きとして保持する」:
        // ここで設定した初期値を反映させるためには、以下の 3項目の設定が必要です。
        // * 「回転」タブの「回転軸 Y, Z」 のチェックをONに設定
        // * 「回転」タブの「回転順序」を「Z -> X -> Y」 に設定
        // * 「パーティクル」タブのパーティクルタイプを「XY平面のポリゴン」に設定
        //------------------------------------------------------------------------------
        const nn::util::Vector3f axisX( 1.0f, 0.0f, 0.0f );
        const nn::util::Vector3f axisY( 0.0f, 1.0f, 0.0f );
        const nn::util::Vector3f axisZ( 0.0f, 0.0f, 1.0f );
        const float Pi = nn::util::FloatPi;
        const float _2Pi = 2.0f * Pi;
        const float epsilon = 0.0001f;

        // パーティクルの初速を正規化して取得
        nn::util::Vector3f velInit;
        arg.GetLocalVec( &velInit );

        if( velInit.LengthSquared() < epsilon )
        {
            // 初速が無い場合、エミッタローカル位置を向きとする
            arg.GetLocalPos( &velInit );
            if( velInit.LengthSquared() < epsilon )
            {
                // 初期位置すら原点であれば、上を向かせる。
                velInit = nn::util::Vector3f( 0, 1, 0 );
            }
        }
        velInit.Normalize();

        // Z -> X -> Y 回転を想定して、X 軸、Y 軸の回転量を決定
        nn::util::Vector3f velXZ( velInit.GetX(), 0, velInit.GetZ() );
        nn::util::Vector3f rot( 0, 0, 0 );

        if( velXZ.LengthSquared() < epsilon )
        {
            // Y 軸方向が進行方向
            const float d = nn::util::VectorDot( axisY, velInit );
            const float rX = ( d > 0.0f ) ? 0 : -Pi;
            rot.Set( rX, 0, 0 );
        }
        else
        {
            velXZ.Normalize();

            // 進行方向は Y 軸に重ならない
            const float cosX = ClampAbs1( nn::util::VectorDot( axisX, velXZ ) );
            const float cosZ = ClampAbs1( nn::util::VectorDot( axisZ, velXZ ) );
            const float cosY = ClampAbs1( nn::util::VectorDot( axisY, velInit ) );

            float rY = Pi - nn::util::AcosEst( cosZ );
            if( cosX > 0.0f )
            {
                // 回転方向を逆にする
                rY = _2Pi - rY;
            }
            float rX = nn::util::AcosEst( cosY );
            rot.Set( -rX, rY, 0 );
        }

        // .eset に設定した初期回転は無視されるが加えると都合が悪いので捨てる
        // ※上記で決めた初期値＋初期回転が丸ごと回転方向ランダムにかけられる

        // 初期回転として上書き
        arg.SetInitialRotateRadian( rot );
    }
    else if( ( res->choices & 0x8 ) == 0x8 )
    {
        {
            //------------------------------------------------------------------------------
            // 「剣ブラー（手動）」:
            //  粒の位置を手動で制御して、ストライプで剣ブラーを作成
            //  MEMO: 単に手動制御のパーティクルととらえることもできる
            //  MEMO: いらないかも
            //------------------------------------------------------------------------------
            // とりあえず右手っぽい所に初期位置として置く
            nn::util::Vector3f pos( -6, 2.5, 0 );
            arg.SetLocalPos( pos );
        }
    }
    return true;
} //NOLINT(impl/function_size)

//------------------------------------------------------------------------------
//  パーティクル計算コールバック
//------------------------------------------------------------------------------
inline void ParticleCalculate( nn::vfx::ParticleCalculateArg& arg )
{
    const ResCustomActionSample* res = reinterpret_cast< const ResCustomActionSample* >( arg.pEmitter->GetCustomActionParameter() );
    if( ( res->choices & 0x2 ) == 0x2 )
    {
        float time = arg.GetTime();
        if( time < res->gazingFrame )
        {
            //------------------------------------------------------------------------------
            // 「指定期間エミッタ位置を向き続ける」:
            // ここで設定した初期値を反映させるためには、以下の 3項目の設定が必要です。
            // * 「回転」タブの「回転軸 Y, Z」 のチェックをONに設定
            // * 「回転」タブの「回転順序」を「Z -> X -> Y」 に設定
            // * 「パーティクル」タブのパーティクルタイプを「XY平面のポリゴン」に設定
            //------------------------------------------------------------------------------
            const nn::util::Vector3f axisX( 1.0f, 0.0f, 0.0f );
            const nn::util::Vector3f axisY( 0.0f, 1.0f, 0.0f );
            const nn::util::Vector3f axisZ( 0.0f, 0.0f, 1.0f );
            const float Pi = nn::util::FloatPi;
            const float _2Pi = 2.0f * Pi;
            const float epsilon = 0.0001f;

            // エミッタ方向を初速を正規化して取得
            nn::util::Vector4f ptclWorldPos;
            arg.GetWorldPos( &ptclWorldPos );
            nn::util::Matrix4x3f emat = arg.pEmitter->GetMatrixRt();
            nn::util::Vector3f ePos;
            nn::util::MatrixGetAxisW( &ePos, emat );
            nn::util::Vector3f velInit;
            nn::util::VectorSubtract( &velInit, ePos, nn::util::Vector3f( ptclWorldPos.GetX(), ptclWorldPos.GetY(), ptclWorldPos.GetZ() ) );

            if( velInit.LengthSquared() < epsilon )
            {
                // 完全にエミッタ位置と重なった場合は上を向く
                velInit = nn::util::Vector3f( 0, 1, 0 );
            }
            velInit.Normalize();
            arg.TransformToLocalVec( &velInit, velInit );

            // Z -> X -> Y 回転を想定して、X 軸、Y 軸の回転量を決定
            nn::util::Vector3f velXZ( velInit.GetX(), 0, velInit.GetZ() );
            nn::util::Vector3f rot( 0, 0, 0 );

            if( velXZ.LengthSquared() < epsilon )
            {
                // Y 軸方向が進行方向
                const float d = nn::util::VectorDot( axisY, velInit );
                const float rX = ( d > 0.0f ) ? 0 : -Pi;
                rot.Set( rX, 0, 0 );
            }
            else
            {
                velXZ.Normalize();

                // 進行方向は Y 軸に重ならない
                const float cosX = ClampAbs1( nn::util::VectorDot( axisX, velXZ ) );
                const float cosZ = ClampAbs1( nn::util::VectorDot( axisZ, velXZ ) );
                const float cosY = ClampAbs1( nn::util::VectorDot( axisY, velInit ) );

                float rY = Pi - nn::util::AcosEst( cosZ );
                if( cosX > 0.0f )
                {
                    // 回転方向を逆にする
                    rY = _2Pi - rY;
                }
                float rX = nn::util::AcosEst( cosY );
                rot.Set( -rX, rY, 0 );
            }

            // .eset に設定した初期回転は無視されるが加えると都合が悪いので捨てる
            // ※上記で決めた初期値＋初期回転が丸ごと回転方向ランダムにかけられる

            // 初期回転として上書き
            arg.SetInitialRotateRadian( rot );
        }
    }
    else if( ( res->choices & 0x4 ) == 0x4 )
    {
        float time = arg.GetTime();
        if( time < res->pulledFrame )
        {
            //------------------------------------------------------------------------------
            // 「指定期間エミッタ移動に引っ張られる」:
            // ここで設定した初期値を反映させるためには、以下の 3項目の設定が必要です。
            // * 「回転」タブの「回転軸 Y, Z」 のチェックをONに設定
            // * 「回転」タブの「回転順序」を「Z -> X -> Y」 に設定
            // * 「パーティクル」タブのパーティクルタイプを「XY平面のポリゴン」に設定
            //------------------------------------------------------------------------------

            // 今の回転量を取得してベクトルに変換
            nn::util::Vector3f rotate;
            arg.GetInitialRotateRadian( &rotate );

            nn::util::Matrix4x3f matrix = nn::util::Matrix4x3f::Identity();
            matrix.SetRotate( rotate );

            nn::util::Vector3f v( 0, 1, 0 );
            nn::util::VectorTransform( &v, v, matrix );
            v.Normalize();
            arg.TransformToWorldVec( &v, v );

            // ベクトルを補正
            nn::util::Vector3f prevPos, nowPos, vAdd;
            prevPos = arg.pEmitter->GetEmitterPrevPos();
            nn::util::Matrix4x3f emat = arg.pEmitter->GetMatrixSrt();
            nn::util::MatrixGetAxisW( &nowPos, emat );
            nn::util::VectorSubtract( &vAdd, nowPos, prevPos );
            nn::util::VectorMultiply( &vAdd, vAdd, res->pulledPower );

            v -= vAdd;
            v.Normalize();
            arg.TransformToLocalVec( &v, v );

            // 再度回転量に戻して保存
            const nn::util::Vector3f axisX( 1.0f, 0.0f, 0.0f );
            const nn::util::Vector3f axisY( 0.0f, 1.0f, 0.0f );
            const nn::util::Vector3f axisZ( 0.0f, 0.0f, 1.0f );
            const float Pi = nn::util::FloatPi;
            const float _2Pi = 2.0f * Pi;
            const float epsilon = 0.0001f;

            // Z -> X -> Y 回転を想定して、X 軸、Y 軸の回転量を決定
            nn::util::Vector3f velInit = v;
            nn::util::Vector3f velXZ( velInit.GetX(), 0, velInit.GetZ() );
            nn::util::Vector3f rot( 0, 0, 0 );

            if( velXZ.LengthSquared() < epsilon )
            {
                // Y 軸方向が進行方向
                const float d = nn::util::VectorDot( axisY, velInit );
                const float rX = ( d > 0.0f ) ? 0 : -Pi;
                rot.Set( rX, 0, 0 );
            }
            else
            {
                velXZ.Normalize();

                // 進行方向は Y 軸に重ならない
                const float cosX = ClampAbs1( nn::util::VectorDot( axisX, velXZ ) );
                const float cosZ = ClampAbs1( nn::util::VectorDot( axisZ, velXZ ) );
                const float cosY = ClampAbs1( nn::util::VectorDot( axisY, velInit ) );

                float rY = Pi - nn::util::AcosEst( cosZ );
                if( cosX > 0.0f )
                {
                    // 回転方向を逆にする
                    rY = _2Pi - rY;
                }
                float rX = nn::util::AcosEst( cosY );
                rot.Set( -rX, rY, 0 );
            }

            // .eset に設定した初期回転は無視されるが加えると都合が悪いので捨てる
            // ※上記で決めた初期値＋初期回転が丸ごと回転方向ランダムにかけられる

            // 初期回転として上書き
            arg.SetInitialRotateRadian( rot );
        }
    }
    else if( ( res->choices & 0x8 ) == 0x8 )
    {
        {
            //------------------------------------------------------------------------------
            // 「剣ブラー（手動）」:
            //  粒の位置を手動で制御して、ストライプで剣ブラーを作成
            //  MEMO: 単に手動制御のパーティクルととらえることもできる
            //------------------------------------------------------------------------------
            // キャラクタを模したエミッタは150fでY軸回転しているので、その右手の位置に粒の位置を動かす

            const float r = 6.0f;
            const float time = arg.GetTime();
            const float maxFrame = 150.0f;
            const float t = ::std::fmodf( time / maxFrame, 1.0f );
            const float initPhase = nn::util::FloatPi;

            nn::util::Vector3f rightHandPos;
            rightHandPos.SetX( nn::util::CosEst( t * nn::util::FloatPi * 2.0f + initPhase ) * r );
            rightHandPos.SetY( nn::util::SinEst( t * ( nn::util::FloatPi * 2.0f ) * 4.0f ) * 0.5f + 2.5f );
            rightHandPos.SetZ( nn::util::SinEst( t * nn::util::FloatPi * 2.0f + initPhase ) * r );
            arg.SetLocalPos( rightHandPos );
        }
    }
} //NOLINT(impl/function_size)

inline void SetCallbacks( nn::vfx::System* pSystem )
{
    nn::vfx::CallbackSet callbackSet;
    callbackSet.particleEmit = ParticleEmitCallback;
    callbackSet.particleCalculate = ParticleCalculate;
    pSystem->SetCallback( nn::vfx::CallbackId_CustomAction1, callbackSet );
}
}
