﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{UtilDecompression.cpp,PageSampleUtilDecompression}
 *
 * @brief
 *  伸長ライブラリのサンプルプログラム
 */

/**
 * @page PageSampleUtilDecompression 一括伸長
 * @tableofcontents
 *
 * @brief
 *  伸長ライブラリを用いたサンプルプログラムの解説です。
 *
 * @section PageSampleUtilDecompression_SectionBrief 概要
 *  ここでは、伸長ライブラリを用いて圧縮データの一括伸長を行うサンプルプログラムの説明をします。
 *
 *  伸長ライブラリの使い方については、
 *  @ref nn::util "ユーティリティライブラリの関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleUtilDecompression_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/UtilDecompression
 *  Samples/Sources/Applications/UtilDecompression @endlink 以下にあります。
 *
 * @section PageSampleUtilDecompression_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleUtilDecompression_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleUtilDecompression_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleUtilDecompression_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleUtilDecompression_SectionDetail 解説
 *
 * @subsection PageSampleUtilDecompression_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  UtilDecompression.h
 *  @includelineno UtilDecompression.h
 *
 *  UtilDecompression.cpp
 *  @includelineno UtilDecompression.cpp
 *
 * @subsection PageSampleUtilDecompression_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムではある文字列を gzip 形式で圧縮したデータである SampleData
 *  の一括伸長を行います。@n
 *  伸長ライブラリが対応している他の圧縮形式には zlib 形式と raw deflate 形式があります。
 *
 *  サンプルプログラムの処理の流れは以下の通りです。
 *  - 伸長元データである SampleData のサイズを表示
 *      - SampleData の圧縮元データは @ref PageSampleUtilCompression で圧縮した文字列
 *  - 伸長ライブラリの nn::util::GetGzipDecompressedSize() によって伸長後のデータサイズを取得
 *      - 伸長を行う前に圧縮データから伸長後のデータサイズを取得可能
 *      - この機能は gzip 形式でのみサポート
 *  - 伸長データ用のバッファである pDecompressedBuffer のメモリ領域確保
 *      - 必要なサイズは nn::util::GetGzipDecompressedSize() から取得
 *  - 一括伸長に必要なワークバッファ pWorkBuffer のメモリ領域確保
 *      - 必要なサイズは nn::util::DecompressGzipWorkBufferSize の値から取得
 *  - 伸長ライブラリの nn::util::DecompressGzip() を呼び出して一括伸長
 *      - pOutDecompressedBuffer に伸長データを格納
 *  - 伸長して得られた文字列を表示
 *
 *  このサンプルプログラムの実行結果を以下に示します。@n
 *  ログに表示される伸長データの文字列は @ref PageSampleUtilCompression の文字列 SampleData と一致します。
 *
 *  @verbinclude  UtilDecompression_OutputExample.txt
 *
 */

#include <algorithm>
#include <memory>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>

#include <nn/util/util_Decompression.h>

#include "UtilDecompression.h"

//-----------------------------------------------------------------------------

namespace {

//
// gzip 形式の一括伸長を行うサンプル関数
//
void SampleDecompressGzip(void* pOutDecompressedBuffer, size_t decompressedBufferSize,
                          const uint8_t* pComressedData, size_t compressedDataSize)
{
    // gzip 形式一括伸長に必要なワークバッファのサイズ
    const size_t workBufferSize = nn::util::DecompressGzipWorkBufferSize;

    // ワークバッファをヒープから取得
    std::unique_ptr<char[]> pWorkBuffer(new char[workBufferSize]);
    NN_ABORT_UNLESS_NOT_NULL(pWorkBuffer.get());

    // gzip 形式で一括伸長
    bool isSuccess = nn::util::DecompressGzip(pOutDecompressedBuffer, decompressedBufferSize,
                pComressedData, compressedDataSize, pWorkBuffer.get(), workBufferSize);

    // ライブラリからの返り値をチェック
    NN_LOG("Decompression (gzip): %s\n", (isSuccess) ? "OK." : "NG.");
    if(!isSuccess)
    {
        // ライブラリが false を返した場合は入力データが壊れているか圧縮形式が異なる
        NN_LOG(" (the input data is corrupted or no supported format.)\n");
    }
}

} // namespace
//-----------------------------------------------------------------------------

//
//  メイン関数
//
extern "C" void nnMain()
{
    // 伸長対象の圧縮データのサイズを表示
    NN_LOG("\nCompressed data size ( %d Bytes )\n", SampleDataSize);

    // gzip 形式の圧縮データから圧縮前のデータサイズを取得
    size_t originalDataSize = nn::util::GetGzipDecompressedSize(SampleData, SampleDataSize);
    NN_LOG("Get pre-compressed data size ( %d Bytes )\n", originalDataSize);

    // 伸長データを格納するバッファの確保
    std::unique_ptr<char[]> pDecompressedBuffer(new char[originalDataSize]);
    NN_ABORT_UNLESS_NOT_NULL(pDecompressedBuffer.get());

    // gzip 形式での一括伸長
    SampleDecompressGzip(pDecompressedBuffer.get(), originalDataSize, SampleData, SampleDataSize);

    // 伸長データの文字列をログ出力
    NN_LOG("\n< Decompressed Data >\n");
    NN_LOG("%s\n", pDecompressedBuffer.get());
}

