﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
 * @examplesource{TmSimple.cpp,PageSampleTmSimple}
 *
 * @brief
 *  TM ライブラリサンプルプログラム
 */

/**
 * @page PageSampleTmSimple TmSimple
 * @tableofcontents
 *
 * @brief
 *  TM ライブラリの使用方法を例示したサンプルです。
 *
 * @section PageSampleTmSimple_SectionBrief 概要
 *  ここでは、TM ライブラリを使用した、ホスト側ツールの作成方法を説明します。
 *
 *  TM ライブラリを使用したプロジェクトの作成方法については、
 *  @ref nn::tm "TM ライブラリ の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleTmSimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/TmSimple Samples/Sources/Applications/TmSimple @endlink 以下にあります。
 *
 * @section PageSampleTmSimple_SectionNecessaryEnvironment 必要な環境
 *  あらかじめ Target Manager をインストールしておく必要があります。
 *
 * @section PageSampleTmSimple_SectionHowToOperate 操作方法
 *  1. 追加可能な機材の一覧が出力されるので、追加したい機材に対応する数値を入力してください
 *  2. サンプルを実行可能な機材の一覧が出力されるので、サンプルを実行したい機材に対する数値を入力してください
 *  数値入力の際は、-1 などの範囲外の数値を入力することで、その操作をスキップすることができます
 *
 * @section PageSampleTmSimple_SectionPrecaution 注意事項
 *
 * @section PageSampleTmSimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleTmSimple_SectionDetail 解説
 *
 * @subsection PageSampleTmSimple_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  TmSimple.cpp
 *  @includelineno TmSimple.cpp
 *
 * @subsection PageSampleTmSimple_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - この PC で動作する Target Manager に登録可能なターゲットのリストアップを行う
 *  - この Target Manager にすでに登録されているターゲットのリストアップを行う
 *  - 登録可能なターゲットを提示し、指定されたターゲットを Target Manager に登録する
 *  - 登録されているターゲットを提示し、指定されたターゲット上でサンプルを実行する
 *
 */

#include <memory>
#include <windows.h>
#include <string>
#include "C:\Program Files\Nintendo\Oasis\include\tm.h"

#define RETURN_IF_FAILURE(operation) \
do { \
    nn::tm::Error MACRO_ERROR = operation; \
    if (MACRO_ERROR != nn::tm::Error_Ok) \
    { \
        nn::tm::ErrorText MACRO_ERROR_STRING; \
        nn::tm::GetErrorTextEn(&MACRO_ERROR_STRING, MACRO_ERROR); \
        printf("%s failed because %s(%d).\n", #operation, MACRO_ERROR_STRING.text, MACRO_ERROR); \
        return; \
    } \
} while (0)

#define PRINT_PARAMETER(error, print, param) \
do { \
    if (error != nn::tm::Error_Ok) \
    { \
        printf(print, "Error"); \
    } \
    else \
    { \
        printf(print, param); \
    } \
} while (0)

const char* ConnectionTypeName[] =
{
    "Unknown",
    "Usb",
    "Ethernet",
    "Future",
};

const char* HardwareTypeName[] =
{
    "Unknown",
    "Edev",
    "Sdev",
    "Sim",
    "Future",
};

const char* TargetStatusName[] =
{
    "Unknown",
    "Disconnected",
    "Disconnecting",
    "Connected",
    "Connecting",
    "Asleep",
    "PoweringOn",
    "PoweringOff",
};

const char* ProgramStateName[] =
{
    "NoProgram",
    "Loading",
    "Loaded",
    "Running",
    "Halted",
};

void ShowRegisteredTargetsWithoutDetails(nn::tm::TargetHandle targetHandles[], int targetCount)
{
    for (int32_t iTarget = 0; iTarget < targetCount; iTarget++)
    {
        nn::tm::Error error;
        printf("%d) ", iTarget);

        nn::tm::TargetName nameBuffer = { 0 };
        error = nn::tm::GetTargetName(&nameBuffer, targetHandles[iTarget]);
        PRINT_PARAMETER(error, "\tName: %s", nameBuffer.name);

        nn::tm::TargetIpAddress ipAddress;
        error = GetTargetIpAddress(&ipAddress, targetHandles[iTarget]);
        PRINT_PARAMETER(error, "\tIPAddress: %15s", ipAddress.ipAddress);

        nn::tm::TargetSerialNumber serialNumber;
        error = nn::tm::GetTargetSerialNumber(&serialNumber, targetHandles[iTarget]);
        PRINT_PARAMETER(error, "\tSerialNumber: %s\n", serialNumber.serialNumber);
    }
}

int GetInputNumber()
{
    int ret;
    char buffer[128];
    char* check;
    while (true)
    {
        fgets(buffer, sizeof(buffer), stdin);
        ret = strtol(buffer, &check, 10);
        if (*check == '\n')
        {
            break;
        }
        printf("Please input number: ");
    }
    return ret;
}

void ShowDiscovoeredTargets()
{
    const int WaitSec = 2;
    RETURN_IF_FAILURE(nn::tm::StartDiscovery(WaitSec * 1000));

    for (int i = 0; i < WaitSec; i++)
    {
        Sleep(1000);
        printf("This is a list of currently found targets\n");

        {
            int32_t targetCount;
            RETURN_IF_FAILURE(nn::tm::GetDiscoveryTargetCount(&targetCount));

            std::unique_ptr<nn::tm::DiscoveryHandle[]> pDiscoveryHandles(new nn::tm::DiscoveryHandle[targetCount]);
            RETURN_IF_FAILURE(nn::tm::GetDiscoveryTargets(pDiscoveryHandles.get(), &targetCount, targetCount));

            for (int32_t iTarget = 0; iTarget < targetCount; iTarget++)
            {
                nn::tm::Error error;
                printf("%02d:\n", iTarget);

                nn::tm::TargetName nameBuffer = { 0 };
                error = nn::tm::GetDiscoveryTargetName(&nameBuffer, pDiscoveryHandles[iTarget]);
                PRINT_PARAMETER(error, "\tName: %s\n", nameBuffer.name);

                nn::tm::TargetIpAddress ipAddress;
                error = nn::tm::GetDiscoveryTargetIpAddress(&ipAddress, pDiscoveryHandles[iTarget]);
                PRINT_PARAMETER(error, "\tIPAddress: %15s\n", ipAddress.ipAddress);

                nn::tm::TargetSerialNumber serialNumber;
                error = nn::tm::GetDiscoveryTargetSerialNumber(&serialNumber, pDiscoveryHandles[iTarget]);
                PRINT_PARAMETER(error, "\tSerialNumber: %s\n", serialNumber.serialNumber);

                nn::tm::TargetMacAddress macAddress;
                error = nn::tm::GetDiscoveryTargetMacAddress(&macAddress, pDiscoveryHandles[iTarget]);
                PRINT_PARAMETER(error, "\tMAC Address: %s\n", macAddress.macAddress);

                nn::tm::TargetConnectionType connectionType;
                error = GetDiscoveryTargetConnectionType(&connectionType, pDiscoveryHandles[iTarget]);
                PRINT_PARAMETER(error, "\tConnectionType: %s\n", ConnectionTypeName[connectionType]);

                nn::tm::TargetHardwareType hardwareType;
                error = nn::tm::GetDiscoveryTargetHardwareType(&hardwareType, pDiscoveryHandles[iTarget]);
                PRINT_PARAMETER(error, "\tHardwareType: %s\n", HardwareTypeName[hardwareType]);
            }
        }
    }
}

void ShowRegisteredTargets()
{
    int32_t targetCount;
    RETURN_IF_FAILURE(nn::tm::GetTargetCount(&targetCount));

    std::unique_ptr<nn::tm::TargetHandle[]> pTargetHandles(new nn::tm::TargetHandle[targetCount]);
    RETURN_IF_FAILURE(nn::tm::GetTargets(pTargetHandles.get(), &targetCount, targetCount));

    nn::tm::TargetHandle defaultTargetHandle;
    RETURN_IF_FAILURE(nn::tm::GetDefaultTarget(&defaultTargetHandle));

    for (int32_t iTarget = 0; iTarget < targetCount; iTarget++)
    {
        nn::tm::Error error;
        printf("%02d:%s\n", iTarget, (defaultTargetHandle.handle == pTargetHandles[iTarget].handle) ? "(Default Target)" : "");

        nn::tm::TargetName nameBuffer = { 0 };
        error = nn::tm::GetTargetName(&nameBuffer, pTargetHandles[iTarget]);
        PRINT_PARAMETER(error, "\tName: %s\n", nameBuffer.name);

        nn::tm::TargetIpAddress ipAddress;
        error = nn::tm::GetTargetIpAddress(&ipAddress, pTargetHandles[iTarget]);
        PRINT_PARAMETER(error, "\tIPAddress: %15s\n", ipAddress.ipAddress);

        nn::tm::TargetSerialNumber serialNumber;
        error = nn::tm::GetTargetSerialNumber(&serialNumber, pTargetHandles[iTarget]);
        PRINT_PARAMETER(error, "\tSerialNumber: %s\n", serialNumber.serialNumber);

        nn::tm::TargetMacAddress macAddress;
        error = nn::tm::GetTargetMacAddress(&macAddress, pTargetHandles[iTarget]);
        PRINT_PARAMETER(error, "\tMAC Address: %15s\n", macAddress.macAddress);

        nn::tm::TargetStatus status;
        error = nn::tm::GetTargetStatus(&status, pTargetHandles[iTarget]);
        PRINT_PARAMETER(error, "\tStatus: %s\n", TargetStatusName[status]);

        nn::tm::TargetConnectionType connectionType;
        error = nn::tm::GetTargetConnectionType(&connectionType, pTargetHandles[iTarget]);
        PRINT_PARAMETER(error, "\tConnectionType: %s\n", ConnectionTypeName[connectionType]);

        nn::tm::TargetHardwareType hardwareType;
        error = nn::tm::GetTargetHardwareType(&hardwareType, pTargetHandles[iTarget]);
        PRINT_PARAMETER(error, "\tHardwareType: %s\n", HardwareTypeName[hardwareType]);

        bool isSleeping;
        error = nn::tm::GetTargetIsSleeping(&isSleeping, pTargetHandles[iTarget]);
        PRINT_PARAMETER(error, "\tIsSleeping: %s\n", (isSleeping) ? "Yes" : "No");

        bool isConnected;
        error = nn::tm::GetTargetIsConnected(&isConnected, pTargetHandles[iTarget]);
        PRINT_PARAMETER(error, "\tIsConnected: %s\n", (isConnected) ? "Yes" : "No");
    }
}

void AddTarget()
{
    // StartDiscovery should be called in ShowDiscovoeredTargets already so just call GetDiscoveryTargetCount and so on
    int32_t targetCount;
    RETURN_IF_FAILURE(nn::tm::GetDiscoveryTargetCount(&targetCount));
    if (targetCount == 0)
    {
        printf("No target to add is found. Skip AddTarget\n");
        return;
    }
    std::unique_ptr<nn::tm::DiscoveryHandle[]> pDiscoveryHandles(new nn::tm::DiscoveryHandle[targetCount]);
    RETURN_IF_FAILURE(nn::tm::GetDiscoveryTargets(pDiscoveryHandles.get(), &targetCount, targetCount));

    printf("Please input the option number to add:\n");
    for (int32_t iTarget = 0; iTarget < targetCount; iTarget++)
    {
        nn::tm::Error error;
        printf("%d):", iTarget);

        nn::tm::TargetName nameBuffer = { 0 };
        error = nn::tm::GetDiscoveryTargetName(&nameBuffer, pDiscoveryHandles[iTarget]);
        PRINT_PARAMETER(error, "\tName: %s", nameBuffer.name);

        nn::tm::TargetIpAddress ipAddress;
        error = nn::tm::GetDiscoveryTargetIpAddress(&ipAddress, pDiscoveryHandles[iTarget]);
        PRINT_PARAMETER(error, "\tIPAddress: %s", ipAddress.ipAddress);

        nn::tm::TargetSerialNumber serialNumber;
        error = nn::tm::GetDiscoveryTargetSerialNumber(&serialNumber, pDiscoveryHandles[iTarget]);
        PRINT_PARAMETER(error, "\tSerialNumber: %s\n", serialNumber.serialNumber);
    }
    int selectedTarget = GetInputNumber();
    if (0 <= selectedTarget && selectedTarget < targetCount)
    {
        nn::tm::TargetHandle targetHandle;
        RETURN_IF_FAILURE(nn::tm::AddTarget(&targetHandle, pDiscoveryHandles[selectedTarget]));
        printf("Added.\n");
    }
}

const std::string GetSampleNspPath()
{
    char buffer[512];
    DWORD ret = GetEnvironmentVariableA("NINTENDO_SDK_ROOT", buffer, sizeof(buffer));
    _ASSERT(ret < sizeof(buffer));
    (void)ret;
    return std::string(buffer) + "\\Resources\\Firmwares\\NX\\Sample\\Binaries\\GfxPrimitiveRenderer.nsp";
}

void RunProgram()
{
    int32_t targetCount;
    RETURN_IF_FAILURE(nn::tm::GetTargetCount(&targetCount));
    if (targetCount == 0)
    {
        printf("No target is found. Skip RunProgram\n");
        return;
    }

    std::unique_ptr<nn::tm::TargetHandle[]> pTargetHandles(new nn::tm::TargetHandle[targetCount]);
    RETURN_IF_FAILURE(nn::tm::GetTargets(pTargetHandles.get(), &targetCount, targetCount));

    printf("Please input the option number of the target to run:\n");
    ShowRegisteredTargetsWithoutDetails(pTargetHandles.get(), targetCount);
    int selectedTarget = GetInputNumber();
    if (0 <= selectedTarget && selectedTarget < targetCount)
    {
        nn::tm::TargetHandle target = pTargetHandles[selectedTarget];
        // Guarantee that the target is connected
        RETURN_IF_FAILURE(nn::tm::ConnectTarget(target));
        printf("Loading %s...\n", GetSampleNspPath().c_str());
        RETURN_IF_FAILURE(nn::tm::LaunchTargetProgram(target, GetSampleNspPath().c_str(), "", "", ""));

        nn::tm::TargetProgramName programName;
        nn::tm::Error getNameError = nn::tm::GetTargetProgramName(&programName, target);
        if (getNameError == nn::tm::Error_Ok)
        {
            printf("Running program is %s\n", programName.name);
        }
        else
        {
            nn::tm::ErrorText errorText;
            nn::tm::GetErrorTextEn(&errorText, getNameError);
            printf("GetTargetProgramName failed because %s(%d).\n", errorText.text, getNameError);
        }

        nn::tm::TargetProgramState programState;
        nn::tm::Error getStateError = nn::tm::GetTargetProgramState(&programState, target);
        if (getStateError == nn::tm::Error_Ok)
        {
            printf("Program state is %s\n", ProgramStateName[programState + 1]);
        }
        else
        {
            nn::tm::ErrorText errorText;
            nn::tm::GetErrorTextEn(&errorText, getStateError);
            printf("GetTargetProgramName failed because %s(%d).\n", errorText.text, getStateError);
        }

        printf("Terminating the application... ");
        RETURN_IF_FAILURE(nn::tm::KillTargetProgram(target));
        printf("Success\n");
    }
}

int main()
{
    ShowDiscovoeredTargets();
    ShowRegisteredTargets();
    AddTarget();
    RunProgram();
    getchar();
    return 0;
}

