﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{SwkbdSimple.cpp,PageSampleSwkbdSimple}
 *
 * @brief
 *  nn::swkbd ライブラリの使い方を示すサンプルプログラム
 */

/**
 * @page PageSampleSwkbdSimple ソフトウェアキーボードアプレットの起動
 * @tableofcontents
 *
 * @brief
 * nn::swkbd ライブラリを用いてソフトウェアキーボードアプレットを起動するサンプルプログラムの解説です。
 *
 * @section PageSampleSwkbdSimple_SectionBrief 概要
 *  ソフトウェアキーボードアプレットを起動します。
 *
 * @section PageSampleSwkbdSimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/SwkbdSimple
 *  Samples/Sources/Applications/SwkbdSimple @endlink 以下にあります。
 *
 * @section PageSampleSwkbdSimple_SectionNecessaryEnvironment 必要な環境
 *  本サンプルプログラムは NX プラットフォームでのみビルドと実行が可能です。
 *
 * @section PageSampleSwkbdSimple_SectionHowToOperate 操作方法
 *  サンプルプログラムを実行した後、Aボタンを押すと、ソフトウェアキーボードアプレットが立ち上がります。
 *
 * @section PageSampleSwkbdSimple_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleSwkbdSimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleSwkbdSimple_SectionDetail 解説
 *  処理の流れは nn::swkbd ライブラリのプログラミングマニュアルに示されています。
 */

#include <cstdlib>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_BitTypes.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/init/init_Malloc.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/os.h>
#include <nn/oe.h>
#include <nn/util/util_BitUtil.h>
#include <nn/swkbd/swkbd_Api.h>
#include <nn/swkbd/swkbd_Result.h>
#include <nns/hid.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#include "ApplicationHeap.h"
#include "GraphicsSystem.h"
#include "FontSystem.h"
#include "Color.h"

namespace {

const size_t MemoryHeapSize = 512 * 1024 * 1024;

const size_t MallocHeapSize = 256 * 1024 * 1024;

const size_t ApplicationHeapSize = 128 * 1024 * 1024;

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
const size_t GraphicsMemorySize = 8 * 1024 * 1024;
#endif

const int FrameBufferWidth = 1280;

const int FrameBufferHeight = 720;

const int FrameRate = 60;


#if defined(NN_BUILD_TARGET_PLATFORM_NX)
void* NvAllocate( size_t size, size_t alignment, void* userPtr ) NN_NOEXCEPT
{
    NN_UNUSED( userPtr );

    return aligned_alloc( alignment, nn::util::align_up(size , alignment));
}

void NvDeallocate( void* addr, void* userPtr ) NN_NOEXCEPT
{
    NN_UNUSED( userPtr );

    free( addr );
}

void* NvReallocate( void* addr, size_t newSize, void* userPtr ) NN_NOEXCEPT
{
    NN_UNUSED( userPtr );
    return realloc( addr, newSize );
}
#endif

} // namespace


extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::SetMemoryHeapSize(MemoryHeapSize));

    uintptr_t address = uintptr_t();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::AllocateMemoryBlock(&address, MallocHeapSize));

    nn::init::InitializeAllocator(
        reinterpret_cast<void*>(address), MallocHeapSize);
}

// アプリケーションのエントリポイント
extern "C" void nnMain()
{
    nn::oe::Initialize();

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    nv::SetGraphicsAllocator( NvAllocate, NvDeallocate, NvReallocate, nullptr );
    nv::SetGraphicsDevtoolsAllocator( NvAllocate, NvDeallocate, NvReallocate, NULL );
    nv::InitializeGraphics( std::malloc( GraphicsMemorySize ), GraphicsMemorySize );
#endif

    nns::hid::ControllerManager controllerManager;
    nns::hid::util::SetControllerManagerWithDefault( &controllerManager );

    ApplicationHeap applicationHeap;
    applicationHeap.Initialize( ApplicationHeapSize );

    GraphicsSystem* pGraphicsSystem = new GraphicsSystem();
    pGraphicsSystem->Initialize(
        &applicationHeap, FrameBufferWidth, FrameBufferHeight);

    FontSystem* pFontSystem = new FontSystem();
    pFontSystem->Initialize(&applicationHeap, pGraphicsSystem);

    while( NN_STATIC_CONDITION( true ) )
    {
        controllerManager.Update();

        nns::hid::Controller* pController = controllerManager.GetController( nns::hid::ControllerId_DebugPad, 0 );

        // A ボタンを押したら、ソフトウェアキーボードアプレットを起動します。
        if( pController->HasAnyButtonsDown( nns::hid::Button::A::Mask ) )
        {
            NN_LOG( "-----------\n" );
            nn::swkbd::ShowKeyboardArg showKeyboardArg;
            nn::swkbd::MakePreset( &( showKeyboardArg.keyboardConfig ), nn::swkbd::Preset_Default );

            // ガイド文字列の設定
            const char* guide_string = u8"please input word.";
            nn::swkbd::SetGuideTextUtf8( &showKeyboardArg.keyboardConfig,
                guide_string );

            // 共有メモリ用バッファの割り当て
            size_t in_heap_size = nn::swkbd::GetRequiredWorkBufferSize( false );
            void* swkbd_work_buffer = applicationHeap.Allocate( in_heap_size, nn::os::MemoryPageSize );

            showKeyboardArg.workBuf = swkbd_work_buffer;
            showKeyboardArg.workBufSize = in_heap_size;

            // 終了パラメータの設定
            size_t out_heap_size = nn::swkbd::GetRequiredStringBufferSize();
            nn::swkbd::String output_string;
            output_string.ptr = reinterpret_cast<char16_t*>(
                applicationHeap.Allocate( out_heap_size, nn::os::MemoryPageSize ) );
            output_string.bufSize = out_heap_size;

            nn::Result result = nn::swkbd::ShowKeyboard( &output_string, showKeyboardArg );
            if( nn::swkbd::ResultCanceled::Includes( result ) )
            {
                NN_LOG( " -- cancel --\n" );
            }
            else if( result.IsSuccess() )
            {
                NN_LOG( " -- ok --\n" );
            }

            // 結果文字列を受け取る
            NN_LOG( "SwkbdSimple: return :\n" );
            char16_t* str_ptr = reinterpret_cast< char16_t* >( output_string.ptr );
            int str_index = 0;
            while( str_ptr[ str_index ] != 0 )
            {
                NN_LOG( "0x%04x,", str_ptr[ str_index ] );
                str_index++;
            }
            NN_LOG( "\n" );

            // 共有メモリ用バッファの解放
            applicationHeap.Deallocate( output_string.ptr );
            applicationHeap.Deallocate( swkbd_work_buffer );
        }

        nn::gfx::util::DebugFontTextWriter&
            textWriter = pFontSystem->GetDebugFontTextWriter();

        const nn::util::Unorm8x4& textColor = Color::White;

        textWriter.SetTextColor( textColor );
        textWriter.SetScale( 1, 1 );
        textWriter.SetCursor( 25, 15 );
        textWriter.Print( "Software Keyboard" );

        pGraphicsSystem->BeginDraw();
        pFontSystem->Draw();
        pGraphicsSystem->EndDraw();

        pGraphicsSystem->Synchronize(
            nn::TimeSpan::FromNanoSeconds( 1000 * 1000 * 1000 / FrameRate ) );
    }

    pFontSystem->Finalize();
    delete pFontSystem;

    pGraphicsSystem->Finalize();
    delete pGraphicsSystem;

    applicationHeap.Finalize();
}
