﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/init/init_Malloc.h>
#include <nn/pl.h>

#include <nns/hid.h>

#include <nn/swkbd/swkbd_InlineKeyboardApi.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_GraphicsFramework.h>

#include "ApplicationHeap.h"
#include "Color.h"
#include "Capture.h"

namespace {

    const size_t MemoryHeapSize = 512 * 1024 * 1024;

    const size_t MallocHeapSize = 512 * 1024 * 1024;

    const size_t ApplicationHeapSize = 256 * 1024 * 1024;

    const int FrameBufferWidth = 1280;

    const int FrameBufferHeight = 720;

    // 入力管理 ( libnns_hid )
    nns::hid::ControllerManager g_ControllerManager;
    ApplicationHeap g_ApplicationHeap;
    // グラフィックスフレームワーク ( libnns_gfx )
    nns::gfx::GraphicsFramework g_GfxFramework;
    nns::gfx::PrimitiveRenderer::Renderer* g_pPrimitiveRenderer = nullptr;

    // インラインモードでソフトウェアキーボードを立ち上げるためのインスタンス
    nn::swkbd::InlineKeyboard g_InlineKeyboard;
    // インラインモードのソフトウェアキーボードが起動時に要求するバッファ格納用
    void* g_pSwkbdInlineWorkBuffer = nullptr;
    // インラインモードのソフトウェアキーボードの１つ前の状態保持用（状態変化を監視する際に使用）
    nn::swkbd::State g_PrevState;
    // インラインモードのソフトウェアキーボードの初期化が完了したかどうかを管理するフラグ
    bool g_IsInitializedSoftwareKeyboard = false;
    // インラインモードのソフトウェアキーボードが終了したかどうかを管理するフラグ
    bool g_IsFinalizedSoftwareKeyboard = false;
    // インラインモードのソフトウェアキーボードの表示/非表示を管理するフラグ
    bool g_IsShowSoftwareKeyboard = false;

    // 描画結果の画像格納用バッファ（インラインモードのソフトウェアキーボードは、描画結果をアプリ側で描画する必要があるため）
    void* g_pImageBuf;
    // 描画結果の画像格納用バッファサイズ
    size_t g_ImageBufSize;
    Capture* g_pCapture = nullptr;

    //
    char g_SoftwareKeyboardText[ ( nn::swkbd::TextMaxLength * 4 ) + 1 ];

    void onFinishedInitialize_()
    {
        NN_LOG( "onFinishedInitialize_\n" );

        g_IsInitializedSoftwareKeyboard = true;
    }

    void onFinishedKeyboard_()
    {
        NN_LOG( "onFinishedKeyboard_\n" );
        g_IsFinalizedSoftwareKeyboard = true;
        g_IsInitializedSoftwareKeyboard = false;
    }

    void onChangedString_( const nn::swkbd::ChangedStringArgUtf8* pChangedStringArg )
    {
        NN_UNUSED( pChangedStringArg );
        NN_LOG( "onChangedString_\n" );

        std::memcpy( g_SoftwareKeyboardText, pChangedStringArg->text, sizeof( char ) * ( ( nn::swkbd::TextMaxLength * 4 ) + 1 ) );
    }

    void onMovedCursor_( const nn::swkbd::MovedCursorArgUtf8* pMovedCursorArg )
    {
        NN_UNUSED( pMovedCursorArg );
        NN_LOG( "onMovedCursor_\n" );

        std::memcpy( g_SoftwareKeyboardText, pMovedCursorArg->text, sizeof( char ) * ( ( nn::swkbd::TextMaxLength * 4 ) + 1 ) );
    }

    void onDecidedEnter_( const nn::swkbd::DecidedEnterArgUtf8* pDecidedEnterArg )
    {
        NN_UNUSED( pDecidedEnterArg );
        NN_LOG( "onDecidedEnter_\n" );

        std::memcpy( g_SoftwareKeyboardText, pDecidedEnterArg->text, sizeof( char ) * ( ( nn::swkbd::TextMaxLength * 4 ) + 1 ) );

        g_IsShowSoftwareKeyboard = false;
    }

    void onDecidedCancel_()
    {
        NN_LOG( "onDecidedCancel_\n" );

        g_IsShowSoftwareKeyboard = false;
    }



} // namespace


void InitializeSoftwareKeyboardInline( ApplicationHeap& applicationHeap )
{
    //
    std::memset( g_SoftwareKeyboardText, 0, sizeof( char ) * ( ( nn::swkbd::TextMaxLength * 4 ) + 1 ) );

    // インラインモードのソフトウェアキーボードを起動するために必要な事前バッファの割り当てをします。
    size_t inline_heap_size = nn::swkbd::InlineKeyboard::GetRequiredWorkBufferSize();
    g_pSwkbdInlineWorkBuffer = applicationHeap.Allocate( inline_heap_size );

    // インラインモードのソフトウェアキーボードを初期化します。
    g_InlineKeyboard.Initialize( g_pSwkbdInlineWorkBuffer );

    // 起動時や文字入力時など、ソフトウェアキーボードで設定可能なコールバック関数を設定します。
    g_InlineKeyboard.SetFinishedInitializeCallback( onFinishedInitialize_ );
    g_InlineKeyboard.SetFinishedKeyboardCallback( onFinishedKeyboard_ );
    g_InlineKeyboard.SetChangedStringCallbackUtf8( onChangedString_ );
    g_InlineKeyboard.SetMovedCursorCallbackUtf8( onMovedCursor_ );
    g_InlineKeyboard.SetDecidedEnterCallbackUtf8( onDecidedEnter_ );
    g_InlineKeyboard.SetDecidedCancelCallback( onDecidedCancel_ );

    g_IsInitializedSoftwareKeyboard = false;
    g_IsFinalizedSoftwareKeyboard = false;
}


void UpdateSoftwareKeyboardInline()
{
    // インラインモードのソフトウェアキーボードのメイン処理を実行します。
    // nn::swkbd::InlineKeyboard::Calc 関数はゲームフレーム毎に呼んでください。
    // この関数内で nn::swkbd::InlineKeyboard::SetFinishedInitializeCallback などで設定したコールバックが呼ばれ、
    // nn::swkbd::InlineKeyboard::SetCursorPos などで設定したパラメータがソフトウェアキーボードに送信されます。
    // 詳細は nn::swkbd::InlineKeyboard の API を参照してください。
    nn::swkbd::State state = g_InlineKeyboard.Calc();
    if( state != g_PrevState )
    {
        switch( state )
        {
        case nn::swkbd::State_None:
            NN_LOG( "state : None\n" );
            break;
        case nn::swkbd::State_Disappear:
            NN_LOG( "state : Disappear\n" );
            break;
        case nn::swkbd::State_InAppear:
            NN_LOG( "state : InAppear\n" );
            break;
        case nn::swkbd::State_Appear:
            NN_LOG( "state : Appear\n" );
            break;
        case nn::swkbd::State_InDisappear:
            NN_LOG( "state : InDisappear\n" );
            break;
        default:
            break;
        }
        g_PrevState = state;
    }
}


void ShowSoftwareKeyboardInline( ApplicationHeap& applicationHeap )
{
    nn::swkbd::AppearArg appear_arg;

    g_InlineKeyboard.SetUtf8Mode( true );

    // 表示要求を送信します。
    g_InlineKeyboard.Appear( appear_arg );
}


void HideSoftwareKeyboardInline( ApplicationHeap& applicationHeap )
{
    // 非表示要求を送信します。
    g_InlineKeyboard.Disappear();
}


void FinalizeSoftwareKeyboardInline( ApplicationHeap& applicationHeap )
{
    // インラインモードのソフトウェアキーボードを終了します。
    // nn::swkbd::InlineKeyboard::Finalize は実際にソフトウェアキーボードが終了するまで処理をブロックします。
    g_InlineKeyboard.Finalize();

    // 共有メモリ用バッファの解放
    if( g_pSwkbdInlineWorkBuffer )
    {
        applicationHeap.Deallocate( g_pSwkbdInlineWorkBuffer );
        g_pSwkbdInlineWorkBuffer = nullptr;
    }

    g_IsFinalizedSoftwareKeyboard = true;
    g_IsInitializedSoftwareKeyboard = false;
}


void UpdateController()
{
    g_ControllerManager.Update();

    nns::hid::Controller* pController = g_ControllerManager.GetController( nns::hid::ControllerId_DebugPad, 0 );

    // ZL ボタンを押したら、ソフトウェアキーボードアプレットを起動します。
    if( pController->HasAnyButtonsDown( nns::hid::Button::ZL::Mask ) )
    {
        if( g_IsInitializedSoftwareKeyboard )
        {
            if( g_IsShowSoftwareKeyboard )
            {
                HideSoftwareKeyboardInline( g_ApplicationHeap );
            }
            else
            {
                ShowSoftwareKeyboardInline( g_ApplicationHeap );
            }

            g_IsShowSoftwareKeyboard = !g_IsShowSoftwareKeyboard;
        }
    }
    else if( pController->HasAnyButtonsDown( nns::hid::Button::ZR::Mask ) )
    {
        if( g_IsInitializedSoftwareKeyboard )
        {
            FinalizeSoftwareKeyboardInline( g_ApplicationHeap );
        }
        else if( g_IsFinalizedSoftwareKeyboard )
        {
            InitializeSoftwareKeyboardInline( g_ApplicationHeap );
        }
    }
}


//------------------------------------------------------------------------------
//  計算処理コールバック
//------------------------------------------------------------------------------
void CalculateCallback( nns::gfx::GraphicsFramework* pGraphicsFramework, void* pUserData )
{
    NN_UNUSED( pGraphicsFramework );
    NN_UNUSED( pUserData );

    UpdateSoftwareKeyboardInline();

    // コントローラの入力を更新します。
    UpdateController();

    if( g_IsInitializedSoftwareKeyboard && !g_IsFinalizedSoftwareKeyboard )
    {
        // ソフトウェアキーボードが表示中であれば、レンダリング結果の画像を取得します。
        // ソフトウェアキーボードの絵が正しく書き込まれると true が返るので、
        // そのタイミングで描画を行ってください。
        bool ret = g_InlineKeyboard.GetImage(
            reinterpret_cast<char*>( g_pImageBuf ),
            g_ImageBufSize );
        if( ret )
        {
            g_pCapture->Set( g_pImageBuf );
        }
    }
}

//------------------------------------------------------------------------------
//  コマンド生成時に渡すユーザデータ
//------------------------------------------------------------------------------
struct MakeCommandUserData
{
    nn::gfx::util::DebugFontTextWriter* pWriter;
};


//------------------------------------------------------------------------------
//  コマンド生成コールバック
//------------------------------------------------------------------------------
void MakeCommandCallback( nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData )
{
    MakeCommandUserData* pData = reinterpret_cast<MakeCommandUserData*>( pUserData );

    // プリミティブレンダラを更新します。
    g_pPrimitiveRenderer->Update( 0 );

    pGraphicsFramework->BeginFrame( bufferIndex );
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = pGraphicsFramework->GetRootCommandBuffer( bufferIndex );

        nn::gfx::ColorTargetView* target = pGraphicsFramework->GetColorTargetView();
        nn::gfx::DepthStencilView* depthStencil = pGraphicsFramework->GetDepthStencilView();

        rootCommandBuffer->ClearColor( target, 0.1f, 0.1f, 0.1f, 1.0f, NULL );
        rootCommandBuffer->SetRenderTargets( 1, &target, depthStencil );
        rootCommandBuffer->SetViewportScissorState( pGraphicsFramework->GetViewportScissorState() );

        // model, view, projection行列をデフォルトにします。
        g_pPrimitiveRenderer->SetDefaultParameters();

        // インラインモードのソフトウェアキーボードが起動しているときも
        // アプリ側は毎フレーム描画を行っているのを確認するため、
        // アプリ側で線が移動するアニメーションを描画します。
        {
            static float g_StartInterval = -1.f;
            g_StartInterval += 0.001f;
            if( g_StartInterval > -0.9f )
            {
                g_StartInterval -= 0.1f;
            }
            float interval = g_StartInterval;
            nn::util::Vector3fType begin;
            nn::util::Vector3fType end;
            g_pPrimitiveRenderer->SetLineWidth( 1.f );
            for( int i = 0; i < 21; i++ )
            {
                nn::util::VectorSet( &begin, -1.f, interval, 0.f );
                nn::util::VectorSet( &end, 1.f, interval, 0.f );

                nn::util::Unorm8x4 color = { { 255, 255, 255, 255 } };
                g_pPrimitiveRenderer->SetColor( color );

                g_pPrimitiveRenderer->DrawLine( rootCommandBuffer, begin, end );
                nn::util::VectorSet( &begin, interval, -1.f, 0.f );
                nn::util::VectorSet( &end, interval, 1.f, 0.f );
                g_pPrimitiveRenderer->DrawLine( rootCommandBuffer, begin, end );
                interval += 0.1f;
            }
        }

        // キャプチャした絵を描画します。
        if( g_IsInitializedSoftwareKeyboard )
        {
            g_pCapture->Draw( g_pPrimitiveRenderer, bufferIndex );
        }

        // デバッグ用のメッセージを描画します。
        const nn::util::Unorm8x4& textColor = Color::White;
        pData->pWriter->SetTextColor( textColor );
        pData->pWriter->SetScale( 1, 1 );
        pData->pWriter->SetCursor( 25, 15 );
        pData->pWriter->Print( "Software Keyboard ( ZR : Initialize/Finalize   ZL : Appear/Disappear )" );
        pData->pWriter->SetCursor( 25, 45 );
        pData->pWriter->Print( "Initialize : %d", g_IsInitializedSoftwareKeyboard );
        pData->pWriter->SetCursor( 25, 75 );
        pData->pWriter->Print( "Finalize   : %d", g_IsFinalizedSoftwareKeyboard );
        pData->pWriter->SetCursor( 25, 105 );
        pData->pWriter->Print( "Show       : %d", g_IsShowSoftwareKeyboard );
        pData->pWriter->SetCursor( 25, 135 );
        pData->pWriter->Print( "Text       : %s", g_SoftwareKeyboardText );

        // デバッグフォント用のコマンドを生成します。
        pData->pWriter->Draw( rootCommandBuffer );

    }
    pGraphicsFramework->EndFrame( bufferIndex );
}


extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::SetMemoryHeapSize( MemoryHeapSize ) );

    uintptr_t address = uintptr_t();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::AllocateMemoryBlock( &address, MallocHeapSize ) );

    nn::init::InitializeAllocator(
        reinterpret_cast<void*>( address ), MallocHeapSize );
}

extern "C" void nnMain()
{
    const int BufferCount = 2;

    nns::hid::util::SetControllerManagerWithDefault( &g_ControllerManager );

    g_ApplicationHeap.Initialize( ApplicationHeapSize );

    // フレームワークを初期化します。
    {
        const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
        nns::gfx::GraphicsFramework::InitializeGraphicsSystem( GraphicsSystemMemorySize );

        nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
        fwInfo.SetDefault();
        fwInfo.SetDisplayWidth( FrameBufferWidth );
        fwInfo.SetDisplayHeight( FrameBufferHeight );
        fwInfo.SetBufferCount( BufferCount );
        fwInfo.SetSwapChainBufferCount( BufferCount );
        fwInfo.SetMemoryPoolSize( nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, ( BufferCount * 8 + 8 ) * 1024 * 1024 );
        fwInfo.SetMemoryPoolSize( nns::gfx::GraphicsFramework::MemoryPoolType_CommandBuffer, ( BufferCount * 8 + 8 ) * 1024 * 1024 );
        fwInfo.SetRootCommandBufferCommandMemorySize( 8 * 1024 * 1024 );

        g_GfxFramework.Initialize( fwInfo );
    }

    // デバッグ用フォントを初期化します。
    nns::gfx::GraphicsFramework::DebugFontTextWriter writer;
    g_GfxFramework.InitializeDebugFontTextWriter( &writer, 1024 * 16, BufferCount );

    // プリミティブレンダラを初期化します。
    {
        nns::gfx::PrimitiveRenderer::RendererInfo rendererInfo;
        rendererInfo.SetDefault();
        rendererInfo.SetAllocator( nns::gfx::GraphicsFramework::DefaultAllocateFunction, nullptr );
        rendererInfo.SetAdditionalBufferSize( 1024 * 4 );
        rendererInfo.SetDrawCallCountMax( 1024 * 4 );
        rendererInfo.SetViewFunctionCallCountMax( 1024 * 4 );

        // ダブルバッファで運用する場合は、SetMultiBufferQuantity で BufferCount を指定し、
        // プリミティブレンダラ内部のバッファをダブルにしたうえで、
        // g_pPrimitiveRenderer->Update(); で利用するバッファを選択( 0 -> 1 -> 0 -> 1 )します。
        rendererInfo.SetMultiBufferQuantity( BufferCount );

        // PrimitiveRendererのインスタンス
        g_pPrimitiveRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer( g_GfxFramework.GetDevice(), rendererInfo );
        g_pPrimitiveRenderer->SetScreenWidth( g_GfxFramework.GetDisplayWidth() );
        g_pPrimitiveRenderer->SetScreenHeight( g_GfxFramework.GetDisplayHeight() );
    }

    MakeCommandUserData userData;

    // コールバックの設定をします。
    {
        g_GfxFramework.SetCalculateCallback( CalculateCallback, nullptr );

        userData.pWriter = &writer.object;
        g_GfxFramework.SetMakeCommandCallback( MakeCommandCallback, &userData );
    }

    g_pCapture = new Capture();
    g_pCapture->Initialize( &g_ApplicationHeap, &g_GfxFramework );

    // ソフトウェアキーボードを初期化・起動します。
    InitializeSoftwareKeyboardInline( g_ApplicationHeap );

    // インラインモードのソフトウェアキーボードは、描画結果をアプリ側で描画する必要があるため、
    // 描画結果を受け取るために要求されるバッファを事前に準備します。
    size_t alignment = 0;
    g_InlineKeyboard.GetImageMemoryRequirement( &g_ImageBufSize, &alignment );
    g_pImageBuf = g_ApplicationHeap.Allocate( g_ImageBufSize, alignment );

    int frame = 0;
    while( NN_STATIC_CONDITION( true ) )
    {
        // フレーム更新を行います。
        {
            int bufferCount = g_GfxFramework.GetBufferCount();
            int currentBufferIndex = frame % bufferCount;
            int previousBufferIndex = ( currentBufferIndex + bufferCount - 1 ) % bufferCount;

            if( frame > 0 )
            {
                // QueueAppCommand
                g_GfxFramework.ExecuteCommand( previousBufferIndex, g_GfxFramework.GetGpuFence( previousBufferIndex ) );
                // QueuePresentTexture
                g_GfxFramework.QueuePresentTexture( g_GfxFramework.GetPresentInterval() );
            }
            // Calculate ( SetCalculateCallback で設定した処理が呼ばれる )
            g_GfxFramework.Calculate();
            // AcquireTexture
            g_GfxFramework.AcquireTexture( currentBufferIndex );
            // MakeCommand
            g_GfxFramework.MakeCommand( currentBufferIndex );
            // WaitDisplaySync
            g_GfxFramework.WaitDisplaySync( currentBufferIndex, g_GfxFramework.GetWaitSyncTimeout() );

            if( frame > 0 )
            {
                // WaitGpuSync
                g_GfxFramework.WaitGpuSync( previousBufferIndex, g_GfxFramework.GetWaitSyncTimeout() );
            }
        }
        g_GfxFramework.ProcessFrame();
    } // while( true )

    g_GfxFramework.QueueFinish();

    g_ApplicationHeap.Deallocate( g_pImageBuf );

    if( g_IsInitializedSoftwareKeyboard )
    {
        FinalizeSoftwareKeyboardInline( g_ApplicationHeap );
    }

    if( g_pCapture )
    {
        g_pCapture->Finalize();
        delete g_pCapture;
        g_pCapture = nullptr;
    }

    // プリミティブレンダラの破棄
    {
        // プリミティブレンダラー終了
        nns::gfx::PrimitiveRenderer::DestroyRenderer( g_pPrimitiveRenderer, g_GfxFramework.GetDevice(), nns::gfx::GraphicsFramework::DefaultFreeFunction, nullptr );
    }

    // デバッグフォント終了
    g_GfxFramework.FinalizeDebugFontTextWriter( &writer );

    // グラフィックフレームワーク終了
    g_GfxFramework.Finalize();

    g_ApplicationHeap.Finalize();
}
