﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
 * Spy の Atk 追加サウンドアーカイブ対応のテスト用サンプル
 * (パッケージには含まれません)
 */

#include "Base.fsid"
#include "Addon.fsid"

#include <nns/nns_Log.h>
#include <nn/atk.h>
#include <nn/spy.h>
#include <nn/spy/atk/spy_AtkSpyModule.h>

#include <nns/atk/atk_SampleCommon.h>

namespace
{
    const char* const Title = "SpyAtkAddonSoundArchive";
    const char* const ArchiveRelativePath = "Base.bfsar";
    const char* const AddonArchiveRelativePath = "Addon.bfsar";

    const int SoundHeapSize = 4 * 1024 * 1024;

    nn::atk::SoundHeap          g_SoundHeap;
    nn::atk::FsSoundArchive     g_SoundArchive;
    nn::atk::SoundArchivePlayer g_SoundArchivePlayer;
    nn::atk::SoundDataManager   g_SoundDataManager;

    nn::audio::MemoryPoolType   g_MemoryPool;

    void* g_pMemoryForSoundSystem;
    void* g_pMemoryForSoundHeap;
    void* g_pMemoryForInfoBlock;
    void* g_pMemoryForLabelData;
    void* g_pMemoryForSoundDataManager;
    void* g_pMemoryForSoundArchivePlayer;
    void* g_pMemoryForStreamBuffer;

    nn::atk::SoundHandle        g_SoundHandle;
    nn::atk::SoundHandle        g_SoundHandleHold;

    const char* const AddonSoundArchiveName1 = "AddonArchive1";
    const char* const AddonSoundArchiveName2 = "AddonArchive2";
    const int AddonSoundArchiveCount = 2;

    nn::atk::AddonSoundArchive g_AddonSoundArchive1;
    nn::atk::AddonSoundArchive g_AddonSoundArchive2;
    nn::atk::SoundDataManager g_AddonSoundDataManager;
    nn::atk::SoundDataManager g_AddonSoundDataManager2;
    void* g_pMemoryForAddonSoundArchive;
    void* g_pMemoryForAddonSoundDataManager;

    nn::spy::SpyController g_SpyController;
    nn::spy::atk::AtkSpyModule g_AtkSpyModule;
}

void Initialize()
{
    nns::atk::InitializeHeap();
    nns::atk::InitializeFileSystem();
    nns::atk::InitializeHidDevices();

    {
#if defined(NN_BUILD_CONFIG_SPY_ENABLED)
        nn::htcs::Initialize(nns::atk::Allocate, nns::atk::Free);
#endif

        nn::spy::SpyController::InitializeArg initializeArg;
        initializeArg.dataBufferLength = 1024 * 1024;
        size_t bufferSize = nn::spy::SpyController::GetRequiredMemorySize(initializeArg);
        void* buffer = nns::atk::Allocate(bufferSize);
        g_SpyController.Initialize(initializeArg, buffer, bufferSize);
    }
}

void Finalize()
{
    g_SpyController.Finalize();
#if defined(NN_BUILD_CONFIG_SPY_ENABLED)
    nn::htcs::Finalize();
#endif

    nns::atk::FinalizeHidDevices();
    nns::atk::FinalizeFileSystem();
    nns::atk::FinalizeHeap();
}

void InitializeAtk()
{
    bool isSuccess = true;

    // SoundSystem の初期化
    nn::atk::SoundSystem::SoundSystemParam param;
    std::size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize(param);
    g_pMemoryForSoundSystem = nns::atk::Allocate(memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize);
    isSuccess = nn::atk::SoundSystem::Initialize(
        param,
        reinterpret_cast<uintptr_t>(g_pMemoryForSoundSystem),
        memSizeForSoundSystem);
    NN_ABORT_UNLESS(isSuccess, "cannot initialize SoundSystem");

    // SoundHeap の初期化
    g_pMemoryForSoundHeap = nns::atk::Allocate(SoundHeapSize);
    isSuccess = g_SoundHeap.Create(g_pMemoryForSoundHeap, SoundHeapSize);
    NN_ABORT_UNLESS(isSuccess, "cannot create SoundHeap");

    // SoundArchive の初期化
    const char* archiveAbsolutePath = nns::atk::GetAbsolutePath(ArchiveRelativePath);
    isSuccess = g_SoundArchive.Open(archiveAbsolutePath);
    NN_ABORT_UNLESS(isSuccess, "cannot open SoundArchive(%s)\n", archiveAbsolutePath);

    // SoundArchive のパラメータ情報をメモリにロード
    std::size_t infoBlockSize = g_SoundArchive.GetHeaderSize();
    g_pMemoryForInfoBlock = nns::atk::Allocate(infoBlockSize, nn::atk::FsSoundArchive::BufferAlignSize);
    isSuccess = g_SoundArchive.LoadHeader(g_pMemoryForInfoBlock, infoBlockSize);
    NN_ABORT_UNLESS(isSuccess, "cannot load InfoBlock");

    // SoundArchive のラベル文字列情報をメモリにロード
    std::size_t memSizeForLabelString = g_SoundArchive.GetLabelStringDataSize();
    g_pMemoryForLabelData = nns::atk::Allocate(memSizeForLabelString, nn::atk::FsSoundArchive::BufferAlignSize);
    g_SoundArchive.LoadLabelStringData(g_pMemoryForLabelData, memSizeForLabelString);

    // SoundDataManager の初期化
    std::size_t memSizeForSoundDataManager = g_SoundDataManager.GetRequiredMemSize(&g_SoundArchive);
    g_pMemoryForSoundDataManager = nns::atk::Allocate(memSizeForSoundDataManager, nn::atk::SoundDataManager::BufferAlignSize);
    isSuccess = g_SoundDataManager.Initialize(
        &g_SoundArchive,
        g_pMemoryForSoundDataManager,
        memSizeForSoundDataManager);
    NN_ABORT_UNLESS(isSuccess, "cannot initialize SoundDataManager");

    // SoundArchivePlayer で用いるストリームバッファの初期化
    // ストリームバッファはメモリプール管理されているヒープから確保する必要があります。
    std::size_t memSizeForStreamBuffer = g_SoundArchivePlayer.GetRequiredStreamBufferSize(&g_SoundArchive);
    g_pMemoryForStreamBuffer = nns::atk::AllocateForMemoryPool(memSizeForStreamBuffer);

    // SoundArchivePlayer の初期化 （AddonSoundArchive の数を指定して初期化）
    std::size_t memSizeForSoundArchivePlayer = g_SoundArchivePlayer.GetRequiredMemSize(&g_SoundArchive, 0, AddonSoundArchiveCount);
    g_pMemoryForSoundArchivePlayer = nns::atk::Allocate(memSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize);

    nn::atk::SoundArchivePlayer::InitializeParam initializeParam;
    initializeParam.pSoundArchive = &g_SoundArchive;
    initializeParam.pSoundDataManager = &g_SoundDataManager;
    initializeParam.pSetupBuffer = g_pMemoryForSoundArchivePlayer;
    initializeParam.setupBufferSize = memSizeForSoundArchivePlayer;
    initializeParam.pStreamBuffer = g_pMemoryForStreamBuffer;
    initializeParam.streamBufferSize = memSizeForStreamBuffer;
    initializeParam.addonSoundArchiveCount = AddonSoundArchiveCount;
    isSuccess = g_SoundArchivePlayer.Initialize(initializeParam);
    NN_ABORT_UNLESS(isSuccess, "cannot initialize SoundArchivePlayer");

    {
        nn::spy::atk::AtkSpyModule::InitializeArg initializeArg;
        initializeArg.addonSoundArchiveCountMax = AddonSoundArchiveCount;
        size_t bufferSize = nn::spy::atk::AtkSpyModule::GetRequiredMemorySize(initializeArg);
        void* buffer = (bufferSize == 0) ? nullptr : nns::atk::Allocate(bufferSize);
        g_AtkSpyModule.Initialize(initializeArg, buffer, bufferSize);

        g_SpyController.RegisterModule(g_AtkSpyModule);
    }
}

void FinalizeAtk()
{
    g_SpyController.UnregisterModule(g_AtkSpyModule);

    g_SoundArchivePlayer.Finalize();

    // 専用のヒープをメモリプールからデタッチ
    nn::atk::SoundSystem::DetachMemoryPool(&g_MemoryPool);

    g_SoundDataManager.Finalize();
    g_SoundArchive.Close();
    g_SoundHeap.Destroy();
    nn::atk::SoundSystem::Finalize();

    nns::atk::FreeForMemoryPool(g_pMemoryForStreamBuffer);
    nns::atk::Free(g_pMemoryForSoundArchivePlayer);
    nns::atk::Free(g_pMemoryForSoundDataManager);
    nns::atk::Free(g_pMemoryForLabelData);
    nns::atk::Free(g_pMemoryForInfoBlock);
    nns::atk::Free(g_pMemoryForSoundHeap);
    nns::atk::Free(g_pMemoryForSoundSystem);
}

void SetupAddonSoundArchive()
{
    const char* addonArchiveAbsolutePath = nns::atk::GetAbsolutePath(AddonArchiveRelativePath);

    // メモリ上にアーカイブをロード
    nn::fs::FileHandle fileHandle;
    nn::fs::OpenFile(&fileHandle, addonArchiveAbsolutePath, nn::fs::OpenMode_Read);
    int64_t fileSize = 0;
    nn::Result result = nn::fs::GetFileSize(&fileSize, fileHandle);
    NN_ABORT_UNLESS(result.IsSuccess());
    // メモリ上にロードしたアーカイブには波形が含まれているので、メモリプール管理されているヒープから確保する必要があります。
    g_pMemoryForAddonSoundArchive = nns::atk::AllocateForMemoryPool(static_cast<size_t>(fileSize));
    size_t readSize = 0;
    result = nn::fs::ReadFile(&readSize, fileHandle, 0, g_pMemoryForAddonSoundArchive, static_cast<size_t>(fileSize));
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(fileHandle);

    // 追加サウンドアーカイブの初期化
    g_AddonSoundArchive1.Initialize(g_pMemoryForAddonSoundArchive);
    g_AddonSoundArchive1.SetExternalFileRoot("content:/");

    g_AddonSoundArchive2.Initialize(g_pMemoryForAddonSoundArchive);
    g_AddonSoundArchive2.SetExternalFileRoot("content:/");

    // 追加サウンドアーカイブ用の SoundDataManager の初期化
    size_t memSizeForAddonSoundDataManager = g_AddonSoundDataManager.GetRequiredMemSize(&g_AddonSoundArchive1);
    g_pMemoryForAddonSoundDataManager = nns::atk::Allocate(memSizeForAddonSoundDataManager, nn::atk::SoundDataManager::BufferAlignSize);
    bool isSuccess = g_AddonSoundDataManager.Initialize(
        &g_AddonSoundArchive1,
        g_pMemoryForAddonSoundDataManager,
        memSizeForAddonSoundDataManager);
    NN_ABORT_UNLESS(isSuccess, "cannot initialize SoundDataManager");

    {
        size_t bufferSize = g_AddonSoundDataManager2.GetRequiredMemSize(&g_AddonSoundArchive2);
        void* buffer = nns::atk::Allocate(bufferSize, nn::atk::SoundDataManager::BufferAlignSize);
        bool isSuccess2 = g_AddonSoundDataManager2.Initialize(
            &g_AddonSoundArchive2,
            buffer,
            bufferSize);
        NN_ABORT_UNLESS(isSuccess2, "cannot initialize SoundDataManager2");
    }

    // SoundArchivePlayer に登録
    g_SoundArchivePlayer.AddAddonSoundArchive(AddonSoundArchiveName1, &g_AddonSoundArchive1, &g_AddonSoundDataManager);
    g_SoundArchivePlayer.AddAddonSoundArchive(AddonSoundArchiveName2, &g_AddonSoundArchive2, &g_AddonSoundDataManager2);

    // 専用のヒープをメモリプールにアタッチ
    nn::atk::SoundSystem::AttachMemoryPool(&g_MemoryPool, nns::atk::GetPoolHeapAddress(), nns::atk::GetPoolHeapSize());
}

void UnsetupAddonSoundArchive()
{
    g_SoundArchivePlayer.RemoveAddonSoundArchive(&g_AddonSoundArchive1);
    g_SoundArchivePlayer.RemoveAddonSoundArchive(&g_AddonSoundArchive2);

    g_AddonSoundDataManager.Finalize();
    g_AddonSoundArchive1.Finalize();

    g_AddonSoundDataManager2.Finalize();
    g_AddonSoundArchive2.Finalize();

    nns::atk::Free(g_pMemoryForAddonSoundDataManager);
    nns::atk::FreeForMemoryPool(g_pMemoryForAddonSoundArchive);
}

void LoadData()
{
    bool isSuccess = true;

    isSuccess = g_SoundDataManager.LoadData(SEQ_BASE, &g_SoundHeap);
    NN_ABORT_UNLESS(isSuccess, "LoadData(SEQ_BASE) failed.");

    isSuccess = g_SoundDataManager.LoadData(SE_YOSHI, &g_SoundHeap);
    NN_ABORT_UNLESS(isSuccess, "LoadData(SE_YOSHI) failed.");

    isSuccess = g_SoundDataManager.LoadData(BANK_BASE2, &g_SoundHeap);
    NN_ABORT_UNLESS(isSuccess, "LoadData(BANK_BASE2) failed.");
}

void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* archiveName, const char* debugLabelName)
{
    g_SoundHandle.Stop(0);

    bool result = g_SoundArchivePlayer.StartSound(&g_SoundHandle, soundId, archiveName).IsSuccess();

    NNS_LOG("StartSound(%s - %s) ... (%d)\n", debugLabelName, archiveName == nullptr ? "BaseArchive" : archiveName, result);
}

void PlayWithStartSound(const char* soundName, const char* archiveName)
{
    g_SoundHandle.Stop(0);

    bool result = g_SoundArchivePlayer.StartSound(&g_SoundHandle, soundName, archiveName).IsSuccess();

    NNS_LOG("StartSound(%s - %s) ... (%d)\n", soundName, archiveName == nullptr ? "BaseArchive" : archiveName, result);
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("[A]              StartSound(BaseArchive) SEQ  (SEQ_BASE)\n");
    NNS_LOG("[X]              StartSound(BaseArchive) WSD  (SE_YOSHI)\n");
    NNS_LOG("[Y]              StartSound(BaseArchive) STRM (STRM_MARIOKART)\n");
    NNS_LOG("[R + A]          StartSound(AddonArchive1) SEQ  (SEQ_ADDON)\n");
    NNS_LOG("[R + X]          StartSound(AddonArchive1) WSD  (SE_WIHAHO)\n");
    NNS_LOG("[R + Y]          StartSound(AddonArchive1) STRM (STRM_PIANO16)\n");
    NNS_LOG("[RIGHT]          StartSound(AddonArchive2) SEQ  (SEQ_ADDON)\n");
    NNS_LOG("[UP]             StartSound(AddonArchive2) WSD  (SE_WIHAHO)\n");
    NNS_LOG("[LEFT]           StartSound(AddonArchive2) STRM (STRM_PIANO16)\n");
    NNS_LOG("[DOWN]           StartSound(AddonArchive2) SEQ  (SEQ_ADDON_MAINBANK)\n");
    NNS_LOG("[B]              Stop Sound\n");
    NNS_LOG("[L]              Print Usage\n");
    NNS_LOG("[Start][Space]   Exit Application\n");
    NNS_LOG("--------------------------------------------------\n");
}

bool UpdateAtk()
{
    // StartSound / Stop
    if (nns::atk::IsTrigger<::nn::hid::DebugPadButton::A>())
    {
        if (nns::atk::IsHold<::nn::hid::DebugPadButton::R>())
        {
            PlayWithStartSound(SEQ_BASE, AddonSoundArchiveName1, "SEQ_ADDON");
        }
        else
        {
            PlayWithStartSound(SEQ_BASE, nullptr, "SEQ_BASE");
        }
    }
    else if (nns::atk::IsTrigger<::nn::hid::DebugPadButton::X>())
    {
        if (nns::atk::IsHold<::nn::hid::DebugPadButton::R>())
        {
            PlayWithStartSound(SE_WIHAHO, AddonSoundArchiveName1, "SE_WIHAHO");
        }
        else
        {
            PlayWithStartSound(SE_YOSHI, nullptr, "SE_YOSHI");
        }
    }
    else if (nns::atk::IsTrigger<::nn::hid::DebugPadButton::Y>())
    {
        if (nns::atk::IsHold<::nn::hid::DebugPadButton::R>())
        {
            PlayWithStartSound(STRM_PIANO16, AddonSoundArchiveName1, "STRM_PIANO16");
        }
        else
        {
            PlayWithStartSound(STRM_MARIOKART, nullptr, "STRM_MARIOKART");
        }
    }
    else if (nns::atk::IsTrigger<::nn::hid::DebugPadButton::Right>())
    {
        PlayWithStartSound("SEQ_ADDON", AddonSoundArchiveName2);
    }
    else if (nns::atk::IsTrigger<::nn::hid::DebugPadButton::Up>())
    {
        PlayWithStartSound("SE_WIHAHO", AddonSoundArchiveName2);
    }
    else if (nns::atk::IsTrigger<::nn::hid::DebugPadButton::Left>())
    {
        PlayWithStartSound("STRM_PIANO16", AddonSoundArchiveName2);
    }
    else if (nns::atk::IsTrigger<::nn::hid::DebugPadButton::Down>())
    {
        PlayWithStartSound("SEQ_ADDON_MAINBANK", AddonSoundArchiveName2);
    }
    else if (nns::atk::IsTrigger<::nn::hid::DebugPadButton::B>())
    {
        g_SoundHandle.Stop(0);
    }

    // Print Usage
    if (nns::atk::IsTrigger<::nn::hid::DebugPadButton::L>())
    {
        PrintUsage();
    }

    // Exit
    if (nns::atk::IsTrigger<::nn::hid::DebugPadButton::Start>())
    {
        return false;
    }

    g_SoundArchivePlayer.Update();

    return true;
}

extern "C" void nnMain()
{
    Initialize();
    InitializeAtk();

    g_SpyController.Open(nn::spy::SpyController::OpenArg());

    SetupAddonSoundArchive();

    LoadData();

    PrintUsage();

    for (;;)
    {
        g_SpyController.SetCurrentApplicationFrame(g_SpyController.GetCurrentApplicationFrame() + 1);

        nns::atk::UpdateHidDevices();

        if (!UpdateAtk())
        {
            break;
        }

        g_AtkSpyModule.PushSoundState(g_SoundArchivePlayer);
        g_SpyController.GetDebugModule().PushDataBufferUsage();

        // Vsync の代わり
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    UnsetupAddonSoundArchive();

    g_SpyController.Close();

    FinalizeAtk();
    Finalize();
}

