﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiIpAddress.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/nifm/nifm_ApiForTest.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <nn/socket/socket_ConstantsPrivate.h>

#include <algorithm>
#include <string>

/**
 * @examplesource{SocketSendMMsg.cpp,PageSampleSocketSendMMsg}
 *
 * @brief
 * Socket API によるソケット通信
 */

/**
 * @page PageSampleSocketSendMMsg Socket API によるソケット通信
 * @tableofcontents
 *
 * @brief
 * Socket API によるソケット通信のサンプルプログラムの解説です。
 *
 * @section PageSampleSocketSendMMsg_SectionBrief 概要
 * まず、 NIFM ライブラリを使用してネットワーク接続の利用をシステムに要求します。
 * その結果、ネットワーク接続が利用可能であれば、
 * ローカル・ループバック・アドレスを使用してサーバ・スレッドとクライアント・スレッドの間でソケット通信を行います。
 *
 * @section PageSampleSocketSendMMsg_SectionFileStoructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/SocketSendMMsg
 * Samples/Sources/Applications/SocketSendMMsg @endlink 以下にあります。
 *
 * @section PageSampleSocketSendMMsg_SectionNecessaryEnvironment 必要な環境
 * 事前に SettingsManager を使用してネットワーク接続設定をインポートする必要があります。
 * インポート方法については @confluencelink{104465190,NIFM ライブラリ} を参考にしてください。
 *
 * @section PageSampleSocketSendMMsg_SectionHowToOperate 操作方法
 * 特にありません。
 *
 * @section PageSampleSocketSendMMsg_SectionPrecaution 注意事項
 * ネットワーク接続に失敗した場合のハンドリング方法は未定です
 *
 * @section PageSampleSocketSendMMsg_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleSocketSendMMsg_SectionDetail 解説
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - NIFM ライブラリを使用してネットワーク接続の利用をシステムに要求します。
 * - socket ライブラリを初期化します。
 * - サーバ・スレッドを立ち上げます。
 * - 約 5 秒後に 2 つのクライアント・スレッドを立ち上げます。
 * - クライアントはサーバに接続し、サーバに対してメッセージを送信します。
 * - サーバはクライアントからメッセージを受信すると、そのクライアントに対して別のメッセージを送信します。
 * - クライアントはサーバからメッセージを受信すると、サーバから切断して処理を終了します。
 * - サーバは約 10 秒後にタイムアウトして処理を終了します。
 *
 * このサンプルプログラムの実行結果を以下に示します。
 * ただし、 Release ビルドではログメッセージが出力されません。
 *
 * @verbinclude SocketSendMMsg_ServerOutput.txt
 *
 * @verbinclude SocketSendMMsg_ClientOutput.txt
 */

namespace
{
    // Socket configuration and memory pool
    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

    const int IpAddressBufferSize   = 16;   /* includes null terminator ("xxx.xxx.xxx.xxx") */
    const int PortBufferSize        = 6;    /* includes null terminator */
    const int MaxMessages           = nn::socket::MaxNumMessages;

    // There are two ways to send the messages to a specific peer
    // 1) "Connect" to the peer. By calling Connect() on a nn::socket::Type::Sock_Dgram socket, the socket
    //    is associated with that peer, meaning that it is the only address which datagrams
    //    are sent to and received from on that socket.
    // 2) Set the peer's address in the msg_name field.
    const bool Connect              = true;
    const bool SetName              = false;

    const bool NoNetwork            = false;

    enum Mode
    {
        MODE_SENDMMSG = 0,
        MODE_RECVMMSG,
        MODE_SENDTO,
        MODE_RECVFROM,
        MODE_HELP,

        MAX_MODES
    };

    void RunSendMMsgTest(char *destIp, char *destPort)
    {
        int sockfd = -1;
        nn::socket::SockAddrIn saPeer = nn::socket::SockAddrIn();
        nn::socket::MMsgHdr msg[MaxMessages];
        nn::socket::Iovec iov1[2];
        ssize_t retval = -1;

        NN_LOG("-- Run SendMMsg test --\n");

        if( (sockfd = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp)) < 0 )
        {
            NN_LOG("socket() error\n");
            return;
        }

        nn::socket::InAddr peerAddr = nn::socket::InAddr();
        nn::socket::InetAton(destIp, &peerAddr);

        saPeer.sin_addr     = peerAddr;
        saPeer.sin_port     = nn::socket::InetHtons(static_cast<uint16_t>(std::atoi(destPort)));
        saPeer.sin_family   = nn::socket::Family::Af_Inet;

        if( Connect )
        {
            if( nn::socket::Connect(sockfd, reinterpret_cast<nn::socket::SockAddr *>(&saPeer), sizeof(saPeer)) == -1 ) {
                NN_LOG("connect() error\n");
                return;
            }
        }

        int packetNum = 0;
        while( 1 )
        {
            memset(iov1, 0, sizeof(iov1));
            iov1[0].iov_base = (void *)"sendmmsg";
            iov1[0].iov_len = strlen((const char *)iov1[0].iov_base);
            iov1[1].iov_base = (void *)"test";
            iov1[1].iov_len = strlen((const char *)iov1[1].iov_base);

            const int msgidx = packetNum % MaxMessages;

            memset(&msg[msgidx], 0, sizeof(msg[msgidx]));
            msg[msgidx].msg_hdr.msg_iov = iov1;
            msg[msgidx].msg_hdr.msg_iovlen = 2;
            if( SetName )
            {
                msg[msgidx].msg_hdr.msg_name = &peerAddr;
                msg[msgidx].msg_hdr.msg_namelen = sizeof(peerAddr);
            }

            packetNum++;  // not actually sent yet but

            if (packetNum % MaxMessages == 0)
            {
                retval = nn::socket::SendMMsg(sockfd, msg, MaxMessages, static_cast<nn::socket::MsgFlag>(0));
                if( retval == -1 )
                {
                    NN_LOG("SendMMsg() returned %d (errno %d)\n", retval, nn::socket::GetLastError());
                }
            }

            if (packetNum % (MaxMessages * 100) == 0)
            {
                NN_LOG("Packets sent: %d\n", packetNum);
            }


            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }
    }

    bool PrepareForReceive(int *psockfd)
    {
        nn::socket::SockAddrIn saPeer = nn::socket::SockAddrIn();
        nn::socket::SockAddrIn sa = nn::socket::SockAddrIn();
        nn::socket::SockLenT   saLen = sizeof(sa);
        nn::socket::InAddr outIpAddress = nn::socket::InAddr();
        nn::Result result = nn::Result();
        bool bRet = true;

        if( (*psockfd = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp)) < 0 )
        {
            NN_LOG("server:  Socket failed (error %d)\n", nn::socket::GetLastError());
            bRet = false;
            goto exit;
        }

        saPeer.sin_addr.S_addr = nn::socket::InetHtonl(nn::socket::InAddr_Any);
        saPeer.sin_port        = nn::socket::InetHtons(0);
        saPeer.sin_family      = nn::socket::Family::Af_Inet;

        if (nn::socket::Bind(*psockfd, reinterpret_cast<nn::socket::SockAddr *>(&saPeer), sizeof(saPeer)) < 0)
        {
            NN_LOG("server:  Bind failed (error %d)\n", nn::socket::GetLastError());
            bRet = false;
            goto exit;
        }

        if( nn::socket::GetSockName(*psockfd, reinterpret_cast<nn::socket::SockAddr *>(&sa), &saLen) < 0 )
        {
            NN_LOG("server:  GetSockName failed (error %d)\n", nn::socket::GetLastError());
            bRet = false;
            goto exit;
        }

        result = nn::nifm::GetCurrentPrimaryIpAddress(&outIpAddress);
        if( result.IsSuccess() )
        {
            NN_LOG("server:  listening for incoming messages at: %s:%d\n",
                nn::socket::InetNtoa(outIpAddress),
                static_cast<int>(nn::socket::InetNtohs(sa.sin_port)));
            bRet = true;
        }
        else
        {
            NN_LOG("server:  GetCurrentIpAddress failed (error %d)\n", result.GetDescription());
            bRet = false;
            goto exit;
        }

    exit:
        return bRet;
    }

    void RunRecvMMsgTest()
    {
        int sockfd = -1;
        nn::socket::MMsgHdr msgs[MaxMessages];
        nn::socket::Iovec iovecs[MaxMessages];
        char bufs[MaxMessages][1400];
        nn::socket::SockAddrIn names[MaxMessages];
        nn::TimeSpan timeout = nn::TimeSpan::FromNanoSeconds(30000000000);
        static size_t numReceivedMessages = 0;
        size_t numLoops = 0;
        nn::socket::TimeVal tv = { 20, 0 };

        NN_LOG("-- Run RecvMMsg test --\n");

        if( PrepareForReceive(&sockfd) == false )
        {
            goto exit;
        }

        memset(msgs, 0, sizeof(msgs));
        for (int i = 0; i < MaxMessages; i++)
        {
            iovecs[i].iov_base          = bufs[i];
            iovecs[i].iov_len           = 1400;
            msgs[i].msg_hdr.msg_iov     = &iovecs[i];
            msgs[i].msg_hdr.msg_iovlen  = 1;
            msgs[i].msg_hdr.msg_name    = &names[i];
            msgs[i].msg_hdr.msg_namelen = sizeof(names[i]);
        }

        nn::socket::SetSockOpt(sockfd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, reinterpret_cast<nn::socket::TimeVal *>(&tv), sizeof(tv));

        while( 1 )
        {
            ssize_t retval = nn::socket::RecvMMsg(sockfd, msgs, MaxMessages, static_cast<nn::socket::MsgFlag>(0), &timeout);
            if( retval == -1 )
            {
                NN_LOG("RecvMMsg() returned %d (errno %d)\n", retval, nn::socket::GetLastError());
                goto exit;
            }
            else
            {
                numReceivedMessages += retval;
                if (++numLoops % 100 == 0)
                {
                    NN_LOG("Packets received: %d\n", numReceivedMessages);
                    bufs[retval - 1][msgs[retval - 1].msg_len] = '\0';
                    char addrbuf[16] = { '\0' };
                    nn::socket::InetNtop(nn::socket::Family::Af_Inet, &(static_cast<nn::socket::SockAddrIn *>(msgs[retval - 1].msg_hdr.msg_name)->sin_addr), addrbuf, sizeof(addrbuf) / sizeof(addrbuf[0]));
                    NN_LOG("\tLast message from %s: %s\n", addrbuf, bufs[retval - 1]); // periodic sanity check
                }
            }
        }

    exit:
        return;
    }

    void RunSendToTest(char *destIp, char *destPort)
    {
        int sockfd = -1;
        nn::socket::SockAddrIn saPeer = nn::socket::SockAddrIn();
        ssize_t retval = -1;
        // for testing RecvMMsg() timeouts
        const int numDelays = 16;
        const int delays[numDelays] = {0, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000};

        NN_LOG("-- Run SendTo test --\n");

        sockfd = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
        if (sockfd == -1) {
            NN_LOG("socket() error\n");
            return;
        }

        nn::socket::InAddr peerAddr = nn::socket::InAddr();
        nn::socket::InetAton(destIp, &peerAddr);

        saPeer.sin_addr     = peerAddr;
        saPeer.sin_port     = nn::socket::InetHtons(static_cast<uint16_t>(std::atoi(destPort)));
        saPeer.sin_family   = nn::socket::Family::Af_Inet;

        int packetNum = 0;
        while( 1 )
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(delays[packetNum % numDelays]));

            const char *msg = "sendtotest";
            retval = nn::socket::SendTo(sockfd, msg, strlen(msg), static_cast<nn::socket::MsgFlag>(0), reinterpret_cast<nn::socket::SockAddr *>(&saPeer), sizeof(saPeer));
            if( retval == -1 )
            {
                NN_LOG("SendTo() returned %d (errno %d)\n", retval, nn::socket::GetLastError());
            }

            if (++packetNum % (MaxMessages * 100) == 0)
            {
                NN_LOG("Packets sent: %d\n", packetNum);
            }
        }
    }

    void RunRecvFromTest()
    {
        int sockfd = -1;
        const size_t outBufferLen = 1400;
        char outBuffer[outBufferLen + 1] = { 0 };
        nn::socket::SockAddr outAddress;
        nn::socket::SockLenT outAddressLen = sizeof(outAddress);
        size_t packetNum = 0;
        //nn::socket::TimeVal tv = { 20, 0 };

        NN_LOG("-- Run RecvFrom test --\n");

        if( PrepareForReceive(&sockfd) == false )
        {
            goto exit;
        }

        //nn::socket::SetSockOpt(sockfd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, reinterpret_cast<nn::socket::TimeVal>(&tv), sizeof(tv));

        while( 1 )
        {
            ssize_t retval = nn::socket::RecvFrom(sockfd, outBuffer, outBufferLen, static_cast<nn::socket::MsgFlag>(0), &outAddress, &outAddressLen);
            if( retval == -1 )
            {
                NN_LOG("RecvFrom() returned %d (errno %d)\n", retval, nn::socket::GetLastError());
                goto exit;
            }
            else
            {
                if (++packetNum % (MaxMessages * 100) == 0)
                {
                    NN_LOG("Packets received: %d\n", packetNum);
                    outBuffer[retval] = '\0';
                    NN_LOG("\tLast message: %s\n", outBuffer);  // sanity check
                }
            }
        }

    exit:
        return;
    }

    bool ParseArgs(Mode *pMode, char *destIp, char *destPort)
    {
        bool bRet = false;

        if( nn::os::GetHostArgc() > 1 )
        {
            const char *ModeSendMMsg = "-sendmmsg";
            const char *ModeRecvMMsg = "-recvmmsg";
            const char *ModeSendTo = "-sendto";
            const char *ModeRecvFrom = "-recvfrom";
            const char *ModeHelp = "-help";
            char *mode = nn::os::GetHostArgv()[1];

            if( strncmp(mode, ModeSendMMsg, strlen(ModeSendMMsg)) == 0 )
            {
                *pMode = MODE_SENDMMSG;
            }
            else if( strncmp(mode, ModeRecvMMsg, strlen(ModeRecvMMsg)) == 0 )
            {
                *pMode = MODE_RECVMMSG;
                bRet = true;
            }
            else if( strncmp(mode, ModeSendTo, strlen(ModeSendTo)) == 0 )
            {
                *pMode = MODE_SENDTO;
            }
            else if( strncmp(mode, ModeRecvFrom, strlen(ModeRecvFrom)) == 0 )
            {
                *pMode = MODE_RECVFROM;
                bRet = true;
            }
            else if( strncmp(mode, ModeHelp, strlen(ModeHelp)) == 0 )
            {
                *pMode = MODE_HELP;
                NN_LOG("\nUsage:\n");
                NN_LOG("  -sendmmsg <ip_addr>:<port> Send 16 messages at a time using SendMMsg().\n");
                NN_LOG("  -sendto <ip_addr>:<port>   Send 1 message at a time using SendTo()\n");
                NN_LOG("  -recvmmsg                  Call RecvMMsg() and wait for 16 messages at a time.\n");
                NN_LOG("  -recvfrom                  Call RecvFrom() and get one message at a time.\n");
            }

            if( (*pMode == MODE_SENDMMSG || *pMode == MODE_SENDTO))
            {
                if( nn::os::GetHostArgc() > 2 )
                {
                    std::string strIpArg = nn::os::GetHostArgv()[2];
                    size_t ipLength = strIpArg.find(':', 0);
                    if( ipLength != std::string::npos )
                    {
                        strncpy(destIp, strIpArg.substr(0, ipLength).c_str(), IpAddressBufferSize);
                        strncpy(destPort, strIpArg.substr(ipLength + 1).c_str(), PortBufferSize);

                        bRet = true;
                    }
                }
            }
        }

        if( bRet == false && *pMode != MODE_HELP )
        {
            NN_LOG("\nError: Invalid arguments. Use '-help' for usage.\n");
        }

        return bRet;
    }

} // namespace

extern "C" void nnMain()
{
    nn::Result result = nn::Result();
    Mode mode = MODE_SENDMMSG;
    char destPort[PortBufferSize] = { 0 };
    char destIp[IpAddressBufferSize] = { 0 };

    NN_LOG("-- SocketSendMMsg.nca --\n");

    if( !ParseArgs(&mode, destIp, destPort) )
    {
        return;
    }

    /* if socket library is used to communicate with external nodes, we must initialize
    network interface manager (NIFM) first */
    result = nn::nifm::Initialize();
    if( result.IsFailure() )
    {
        NN_LOG("\nError: nn::nifm::Initialize() failed. (error %d)\n", result.GetDescription());
        return;
    }

    // test code for working at NCL where NIFM fails because we fail internet connection test
    if( NoNetwork )
    {
        result = nn::nifm::SetExclusiveClient(nn::nifm::GetClientId());
        if( result.IsFailure() )
        {
            NN_LOG("\nError: nn::nifm::SetExclusiveClient() failed. (error %d)\n", result.GetDescription());
            return;
        }

        nn::nifm::NetworkConnection *pNifmConnection = new nn::nifm::NetworkConnection;
        if( nullptr == pNifmConnection )
        {
            NN_LOG("\nError: Failed to allocate nn::nifm::NetworkConnection.\n");
            return;
        }

        nn::util::Uuid netProfile = nn::util::InvalidUuid;
        netProfile.FromString("93188440-b6d5-14b7-056b-08cd57910964");
        nn::nifm::RequestHandle requestHandle = pNifmConnection->GetRequestHandle();
        nn::nifm::SetRequestConnectionConfirmationOption(requestHandle, nn::nifm::ConnectionConfirmationOption_Prohibited);
        nn::nifm::SetRequestPersistent(requestHandle, true);

        if( nn::util::InvalidUuid != netProfile )
        {
            NN_LOG("Setting Uuid.\n");
            result = nn::nifm::SetRequestNetworkProfileId(requestHandle, netProfile);
            if( result.IsFailure() )
            {
                NN_LOG("\nError: nn::nifm::SetRequestNetworkProfileId() failed. (error %d)\n", result.GetDescription());
                return;
            }
        }

        // Submit asynchronous request to NIFM
        pNifmConnection->SubmitRequest();

        // Wait while NIFM brings up the interface
        while( !pNifmConnection->IsAvailable() )
        {
            NN_LOG("Waiting for network interface...\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
    }
    else
    {
        /* this application is now requesting to use network interface */
        nn::nifm::SubmitNetworkRequest();

        /* wait for network interface availability, while providing status */
        while( nn::nifm::IsNetworkRequestOnHold() )
        {
            NN_LOG("Waiting for network interface availability...\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        if( !nn::nifm::IsNetworkAvailable() )
        {
            NN_LOG("\nError: network is not available.\n");
            goto CLEANUP_NIFM;
        }
    }

    /* Initialize socket library, while supplying configuration and memory pool */
    result = nn::socket::Initialize(g_SocketConfigWithMemory);

    if( result.IsFailure() )
    {
        NN_LOG("\nError: nn::socket::Initialize() failed. (error %d)\n\n", result.GetDescription());
        goto CLEANUP_NIFM;
    }

    switch( mode )
    {
    case MODE_SENDMMSG:
            RunSendMMsgTest(destIp, destPort);
            break;
    case MODE_RECVMMSG:
            RunRecvMMsgTest();
            break;
    case MODE_SENDTO:
            RunSendToTest(destIp, destPort);
            break;
    case MODE_RECVFROM:
    default:
            RunRecvFromTest();
            break;
    }

    nn::socket::Finalize();

CLEANUP_NIFM:
    /* this application no longer requires use of network interface */
    nn::nifm::CancelNetworkRequest();
}
