﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 *  @examplesource{ProfilerSimple.cpp,PageSampleProfilerSimple}
 *
 *  @brief
 *  Sample application showing how to use functions in the nn::profiler namespace.
 */

/**
 *  @page PageSampleProfilerSimple Nintendo NX CPU Profiler Sample
 *  @tableofcontents
 *
 *  @brief
 *  This sample shows how to use the various functions in the nn::profiler namespace.
 *
 *  @section PageSampleProfilerSimple_SectionBrief Overview
 *
 *  @section PageSampleProfilerSimple_SectionFileStructure File Structure
 *  This sample program is located in
 *  @link ../../../Samples/Sources/Applications/ProfilerSimple Samples/Sources/Applications/ProfilerSimple @endlink.
 *
 *  @section PageSampleProfilerSimple_SectionNecessaryEnvironment System Requirements
 *  The Nintendo NX CPU Profiler PC executable, found in Tools/NintendoNxCpuProfiler, should be running.
 *
 *  @section PageSampleProfilerSimple_SectionHowToOperate Operation Procedures
 *  It is not possible to interact directly with the sample.
 *
 *  @section PageSampleProfilerSimple_SectionPrecaution Cautions
 *  This demo does not display anything on the screen.
 *
 *  @section PageSampleProfilerSimple_SectionHowToExecute Execution Procedure
 *  Build the sample program and then run it.
 *  Once it is running, click Sync in the Nintendo NX CPU Profiler PC tool.
 *  On the Sampled Profile tab, click on Start to start taking a profile.
 *  Profiling can be stopped by either clicking on Stop or waiting for the internal buffers to fill.
 *
 *  @section PageSampleProfilerSimple_SectionDetail Description
 *
 *  @subsection PageSampleProfilerSimple_SectionSampleProgram Sample Program
 *  The following source code is used for this sample program.
 *
 *  ProfilerSimple.cpp
 *  @includelineno ProfilerSimple.cpp
 *
 *  @subsection PageSampleProfilerSimple_SectionSampleDetail Sample Program Description
 *  This sample makes use of the various function in the nn::profiler namespace.
 *  The intention is to provide an example on expected usage of these functions.
 *
 *  This sample is to be used in conjunction with the Nintendo NX CPU Profiler PC executable.
 *  The executable can be found in Tools/NintendoNxCpuProfiler.
 *
 *  The sample shows the basic method used to both Initialize and Finalize the profiler.
 *  It also shows how to record the framerate, record code blocks, and record data.
 */

#include <algorithm>
#include <cstdlib>

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/profiler.h>

namespace
{
    // Waits for the profiler to be in the ProfilerStatus_Active or ProfilerStatus_Offline state
    // before returning. This ensures that if profiling or transferring is in progress that the
    // data can be obtained in the PC before the application continues.
    void WaitForProfileTransferred()
    {
        nn::profiler::ProfilerStatus status = nn::profiler::GetProfilerStatus();
        while (status == nn::profiler::ProfilerStatus_Profiling ||
               status == nn::profiler::ProfilerStatus_Transferring)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
            status = nn::profiler::GetProfilerStatus();
        }
    }

    void RandomLoop()
    {
        int loopCount = (rand() & 0xFFFF);

        // Example of recording data with a string literal.
        nn::profiler::RecordData(u8"RandomLoop: Loop Count", static_cast<uint64_t>(loopCount));

        // Example of using a scope code block with a string literal.
        // The code block will automatically exit when the variable falls out of scope.
        // The string literal is built using the preprocessor to show __FILE__:__LINE__.
        nn::profiler::ScopedCodeBlock codeBlock(__FILE__ ":" NN_MACRO_STRINGIZE(__LINE__));
        while (loopCount > 0)
        {
            --loopCount;
        }
    }

    int64_t CalculateFibonacci(int n)
    {
        int64_t retVal = n;
        if (n > 1) { retVal = CalculateFibonacci(n - 1) + CalculateFibonacci(n - 2); }
        return retVal;
    }
}

extern "C" void nnMain()
{
    srand(0);

    // Initialize the profiler
    char* buffer = new char[nn::profiler::MinimumBufferSize];
    nn::Result result = nn::profiler::Initialize(buffer, nn::profiler::MinimumBufferSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    const int totalFrameCount = 120 * 60;
    for (int frame = 0; frame < totalFrameCount; ++frame)
    {
        auto startTime = nn::os::GetSystemTick();

        // Record the top of our main loop. This is used as the framerate in the GUI.
        nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_Main);

        RandomLoop();

        int fibonacciSeed = (rand() % 20) + 10;
        int64_t fibonacciResult = CalculateFibonacci(fibonacciSeed);

        // Example of recording data with a numerical ids.
        nn::profiler::RecordData(1, static_cast<uint64_t>(fibonacciSeed));
        nn::profiler::RecordData(2, static_cast<double>(fibonacciResult));

        // Determine how long we have spent on this frame so far.
        // Set the wait time so that we attempt to maintain a consistent framerate.
        auto stopTime = nn::os::GetSystemTick();
        int elapsedMicroSeconds = (stopTime - startTime).ToTimeSpan().GetMicroSeconds();
        int waitTime = std::max(0, 16667 - elapsedMicroSeconds);

        // Example of entering and exiting a code block using a string literal.
        nn::profiler::EnterCodeBlock("Main Thread Sleeping");
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(waitTime));
        nn::profiler::ExitCodeBlock("Main Thread Sleeping");
    }

    WaitForProfileTransferred();

    // Finalize the profiler
    result = nn::profiler::Finalize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    delete[] buffer;
}
