﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
* @examplesource{PerfSimple.cpp,PageSamplePerfSimple}
*
* @brief
*  nn::perf を使用した CPU 及び GPU パフォーマンス計測のサンプルプログラム
*/

/**
* @page PageSamplePerfSimple PerfSimple
* @tableofcontents
*
* @image html Applications\PerfSimple\perfsimpleimage.png
*
* @brief
*  CPU 及び GPU パフォーマンス計測サンプルの解説
*
* @section PageSamplePerfSimple_SectionBrief 概要
*  nn::perf::LoadMeterCenter を使用して CPU 及び GPU のパフォーマンス計測を行うサンプルです。@n
*  初期化や処理計測には nn::perf のマクロを利用しています。
*
* @section PageSamplePerfSimple_SectionFileStructure ファイル構成
*  本サンプルプログラムは @link ../../../Samples/Sources/Applications/PerfSimple
*  Samples/Sources/Applications/PerfSimple @endlink 以下にあります。
*
* @section PageSamplePerfSimple_SectionNecessaryEnvironment 必要な環境
*  画面表示が利用可能である必要があります。
*
* @section PageSamplePerfSimple_SectionHowToOperate 操作方法
*  特にありません。
*
* @section PageSamplePerfSimple_SectionPrecaution 注意事項
*  Debug ビルド版では 60fps に収まらずに処理落ちしてしまう場合があります。@n
*  これは処理落ちした場合の挙動を示すことを意図しているため不具合ではありません。
*
* @section PageSamplePerfSimple_SectionHowToExecute 実行手順
*  サンプルプログラムをビルドし、実行してください。
*
* @section PageSamplePerfSimple_SectionDetail 解説
*
* @subsection PageSamplePerfSimple_SectionSampleDetail サンプルプログラムの解説
* このサンプルではメインスレッドと 3つのサブスレッドを使用して、赤緑青黄の 4つのオブジェクト群を描画します。@n
* また、CPU 及び GPU のパフォーマンス計測を行い、その結果をコンソール及び画面に出力します。
*
* 3つのサブスレッドはそれぞれ赤緑青のオブジェクト描画コマンドを生成し、メインスレッドでは黄オブジェクトとその他の描画コマンドを生成します。@n
* 各スレッドは以下のように特定の CPU コア上で動作するように設定しています。
*
* - メインスレッド(黄・その他) → コア0
* - サブスレッド(赤) → コア0
* - サブスレッド(緑) → コア1
* - サブスレッド(青) → コア2
*
* サンプルプログラムの処理の流れは以下の通りです。
*
* - 以下を初期化
* -- グラフィックスフレームワーク
* -- 赤青緑オブジェクト用コマンドバッファ
* -- プリミティブレンダラ
* -- デバッグフォント
* -- スレッド
* -- イベント
* -- LoadMeterCenter
* -- 描画オブジェクトの位置
* - 各スレッドの開始
* - ループ開始
* -- プリミティブレンダラを更新
* -- 各スレッドで赤青緑のオブジェクト描画コマンド作成
* -- メインのコマンドを作成
* -- コマンドの実行
* - ループ開始に戻る
* - 各種オブジェクトを破棄
*
* このサンプルプログラムのコンソール出力結果を以下に示します。
*
*  @verbinclude  PerfSimple_OutputExample.txt
*/

#include <cstdlib>
#include <cstring>

#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Log.h>
#include <nn/gfx.h>
#include <nn/util/util_Color.h>

#define NN_PERF_PROFILE_ENABLED
#include <nn/perf.h>

#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeterDrawer.h>

namespace {
int g_Frame = 0;

nns::gfx::GraphicsFramework::DebugFontTextWriter g_Writer;

const int g_CoreCount = 3;

enum ObjectType
{
    ObjectType_Red,
    ObjectType_Green,
    ObjectType_Blue,

    ObjectType_End
};

// スレッドローカルストレージにセットする構造体
struct TlsSlotValue
{
    int                 coreNumber;
    const char*         name;
    ObjectType          objectType;
    nn::util::Color4u8  color;
};

void ThreadMain(void *tlsValue);

//---------------------------------------------------------------
// フレームワークを初期化
//---------------------------------------------------------------
nns::gfx::GraphicsFramework gfw;
void InitializeGraphicsFramework()
{
    const size_t graphicsSystemMemorySize = 8 * 1024 * 1024;
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(graphicsSystemMemorySize);
    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayHeight(720);
    fwInfo.SetDisplayWidth(1280);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_CommandBuffer, 64 * 1024 * 1024);
    fwInfo.SetBufferCount(2);
    fwInfo.SetSwapChainBufferCount(2);
    gfw.Initialize(fwInfo);
}

//---------------------------------------------------------------
// フレームワークを破棄
//---------------------------------------------------------------
void FinalizeGraphicsFramework()
{
    gfw.Finalize();
}

//---------------------------------------------------------------
// コマンドバッファを初期化
//---------------------------------------------------------------
int g_CurrentBufferIndex = 0;
nns::gfx::GraphicsFramework::CommandBuffer g_CommandBuffer[g_CoreCount][2];
void InitializeCommandBuffer()
{
    // 赤青緑オブジェクト描画コマンド用のコマンドバッファを初期化
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability(nn::gfx::QueueCapability_Graphics);
    info.SetCommandBufferType(nn::gfx::CommandBufferType_Nested);
    size_t commandMemorySize = 1024 * 1024 * 2;
    size_t controlMemorySize = 256;

    gfw.InitializeCommandBuffer(&g_CommandBuffer[ObjectType_Red][0], info, commandMemorySize, controlMemorySize);
    gfw.InitializeCommandBuffer(&g_CommandBuffer[ObjectType_Red][1], info, commandMemorySize, controlMemorySize);
    gfw.InitializeCommandBuffer(&g_CommandBuffer[ObjectType_Green][0], info, commandMemorySize, controlMemorySize);
    gfw.InitializeCommandBuffer(&g_CommandBuffer[ObjectType_Green][1], info, commandMemorySize, controlMemorySize);
    gfw.InitializeCommandBuffer(&g_CommandBuffer[ObjectType_Blue][0], info, commandMemorySize, controlMemorySize);
    gfw.InitializeCommandBuffer(&g_CommandBuffer[ObjectType_Blue][1], info, commandMemorySize, controlMemorySize);
}

//---------------------------------------------------------------
// コマンドバッファを破棄
//---------------------------------------------------------------
void FinalizeCommandBuffer()
{
    gfw.FinalizeCommandBuffer(&g_CommandBuffer[ObjectType_Red][0]);
    gfw.FinalizeCommandBuffer(&g_CommandBuffer[ObjectType_Red][1]);
    gfw.FinalizeCommandBuffer(&g_CommandBuffer[ObjectType_Green][0]);
    gfw.FinalizeCommandBuffer(&g_CommandBuffer[ObjectType_Green][1]);
    gfw.FinalizeCommandBuffer(&g_CommandBuffer[ObjectType_Blue][0]);
    gfw.FinalizeCommandBuffer(&g_CommandBuffer[ObjectType_Blue][1]);
}

//---------------------------------------------------------------
// プリミティブレンダラの初期化
//---------------------------------------------------------------
nns::gfx::PrimitiveRenderer::Renderer*  g_pRootPrimitiveRenderer;
nns::gfx::PrimitiveRenderer::Renderer*  g_pPrimitiveRenderer[g_CoreCount];
void InitializePrimitiveRenderer()
{
    nns::gfx::PrimitiveRenderer::RendererInfo info;
    info.SetDefault();
    info.SetAllocator(gfw.DefaultAllocateFunction, NULL);
    info.SetAdditionalBufferSize(1024 * 2);
    info.SetDrawCallCountMax(1024 * 2);
    info.SetViewFunctionCallCountMax(1024 * 2);
    info.SetMultiBufferQuantity(2);

    // メインで使用するプリミティブレンダラを初期化
    g_pRootPrimitiveRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer(gfw.GetDevice(), info);
    g_pRootPrimitiveRenderer->SetScreenWidth(gfw.GetDisplayWidth());
    g_pRootPrimitiveRenderer->SetScreenHeight(gfw.GetDisplayHeight());

    // 赤青緑オブジェクト描画コマンド用のプリミティブレンダラを初期化
    size_t rendererMemoryPoolSize = nns::gfx::PrimitiveRenderer::Renderer::GetRequiredMemoryPoolSize(gfw.GetDevice(), info);
    size_t rendererMemoryPoolAlignment = nns::gfx::PrimitiveRenderer::Renderer::GetMemoryPoolAlignment(gfw.GetDevice(), info);
    for (int i = 0; i < g_CoreCount; ++i)
    {
        ptrdiff_t rendererMemoryPoolOffset = gfw.AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_CommandBuffer, rendererMemoryPoolSize, rendererMemoryPoolAlignment);
        nn::gfx::MemoryPool* rendererMemoryPool = gfw.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_CommandBuffer);
        NN_ASSERT(rendererMemoryPoolOffset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset, "memoryPool shortage");
        void* rendererMemory = gfw.AllocateMemory(sizeof(nns::gfx::PrimitiveRenderer::Renderer), 16);
        g_pPrimitiveRenderer[i] = new (rendererMemory) nns::gfx::PrimitiveRenderer::Renderer();
        g_pPrimitiveRenderer[i]->Initialize(gfw.GetDevice(), info, nns::gfx::PrimitiveRenderer::GetGraphicsResource(), rendererMemoryPool, rendererMemoryPoolOffset, rendererMemoryPoolSize);
        g_pPrimitiveRenderer[i]->SetScreenWidth(gfw.GetDisplayWidth());
        g_pPrimitiveRenderer[i]->SetScreenHeight(gfw.GetDisplayHeight());
    }
}

//---------------------------------------------------------------
// プリミティブレンダラの更新
//---------------------------------------------------------------
void UpdatePrimitiveRenderer(int bufferIndex)
{
    g_pRootPrimitiveRenderer->Update(bufferIndex);
    g_pPrimitiveRenderer[ObjectType_Red]->Update(bufferIndex);
    g_pPrimitiveRenderer[ObjectType_Green]->Update(bufferIndex);
    g_pPrimitiveRenderer[ObjectType_Blue]->Update(bufferIndex);
}

//---------------------------------------------------------------
// プリミティブレンダラの破棄
//---------------------------------------------------------------
void FinalizePrimitiveRenderer()
{
    for (int i = 0; i < g_CoreCount; ++i)
    {
        g_pPrimitiveRenderer[i]->Finalize(gfw.GetDevice());
        void* pRenderer = g_pPrimitiveRenderer[i];
        gfw.FreeMemory(pRenderer);
        g_pPrimitiveRenderer[i] = NULL;
    }

    nns::gfx::PrimitiveRenderer::DestroyRenderer(g_pRootPrimitiveRenderer, gfw.GetDevice(), gfw.DefaultFreeFunction, NULL);
}

//---------------------------------------------------------------
// スレッド毎のコア番号を取得する関数
//---------------------------------------------------------------
nn::os::TlsSlot g_TlsSlot;
int GetCoreNumberFunction()
{
    // TlsSlotValue 構造体の coreNumber からコア番号を取得
    int coreNumber = reinterpret_cast<TlsSlotValue*>(nn::os::GetTlsValue(g_TlsSlot))->coreNumber;
    return *reinterpret_cast<int*>(&coreNumber);
}

//---------------------------------------------------------------
// スレッドの初期化
//---------------------------------------------------------------
nn::os::ThreadType              g_Thread[g_CoreCount];
const size_t                    g_ThreadStackSize = 0x4000;
NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[g_CoreCount][g_ThreadStackSize];
TlsSlotValue                    g_TlsSlotValueMain;
TlsSlotValue                    g_TlsSlotValue[g_CoreCount];
void InitializeThread()
{
    // アフィニティマスクの設定
    nn::Bit64 core0 = 1;
    nn::Bit64 core1 = 2;
    nn::Bit64 core2 = 4;

    // TLS スロットを 1 つ確保
    // 本サンプルではスレッド毎で動作するコア番号を perf で取得するために使用
    nn::Result result = nn::os::AllocateTlsSlot(&g_TlsSlot, NULL);
    NN_ASSERT(result.IsSuccess(), "Cannot allocate TLS slot.");

    // 赤オブジェクトのコマンド作成はコア0で行うように設定
    // スレッドローカルストレージに入れる値を設定
    g_TlsSlotValue[ObjectType_Red].coreNumber = 0;
    g_TlsSlotValue[ObjectType_Red].name = "Red";
    g_TlsSlotValue[ObjectType_Red].objectType = ObjectType_Red;
    g_TlsSlotValue[ObjectType_Red].color = nn::util::Color4u8::Red();
    result = nn::os::CreateThread(&g_Thread[ObjectType_Red], ThreadMain, &g_TlsSlotValue[ObjectType_Red], g_ThreadStack[ObjectType_Red], g_ThreadStackSize, nn::os::DefaultThreadPriority);
    NN_ASSERT(result.IsSuccess(), "Cannot create g_ThreadRed.");
    nn::os::SetThreadName(&g_Thread[ObjectType_Red], "ThreadRed");
    nn::os::SetThreadCoreMask(&g_Thread[ObjectType_Red], 0, core0);

    // 緑オブジェクトのコマンド作成はコア1で行うように設定
    g_TlsSlotValue[ObjectType_Green].coreNumber = 1;
    g_TlsSlotValue[ObjectType_Green].name = "Green";
    g_TlsSlotValue[ObjectType_Green].objectType = ObjectType_Green;
    g_TlsSlotValue[ObjectType_Green].color = nn::util::Color4u8::Green();
    result = nn::os::CreateThread(&g_Thread[ObjectType_Green], ThreadMain, &g_TlsSlotValue[ObjectType_Green], g_ThreadStack[ObjectType_Green], g_ThreadStackSize, nn::os::DefaultThreadPriority);
    NN_ASSERT(result.IsSuccess(), "Cannot create g_ThreadGreen.");
    nn::os::SetThreadName(&g_Thread[ObjectType_Green], "ThreadGreen");
    nn::os::SetThreadCoreMask(&g_Thread[ObjectType_Green], 1, core1);

    // 青オブジェクトのコマンド作成はコア2で行うように設定
    g_TlsSlotValue[ObjectType_Blue].coreNumber = 2;
    g_TlsSlotValue[ObjectType_Blue].name = "Blue";
    g_TlsSlotValue[ObjectType_Blue].objectType = ObjectType_Blue;
    g_TlsSlotValue[ObjectType_Blue].color = nn::util::Color4u8::Blue();
    result = nn::os::CreateThread(&g_Thread[ObjectType_Blue], ThreadMain, &g_TlsSlotValue[ObjectType_Blue], g_ThreadStack[ObjectType_Blue], g_ThreadStackSize, nn::os::DefaultThreadPriority);
    NN_ASSERT(result.IsSuccess(), "Cannot create g_ThreadBlue.");
    nn::os::SetThreadName(&g_Thread[ObjectType_Blue], "ThreadBlue");
    nn::os::SetThreadCoreMask(&g_Thread[ObjectType_Blue], 2, core2);

    // 現在のスレッド(メインスレッド)をコア0に割り当て
    g_TlsSlotValueMain.coreNumber = 0;
    g_TlsSlotValueMain.name = "Yellow";
    g_TlsSlotValueMain.objectType = ObjectType_End;
    g_TlsSlotValueMain.color = nn::util::Color4u8::Yellow();
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    nn::os::SetTlsValue(g_TlsSlot, reinterpret_cast<uintptr_t>(&g_TlsSlotValueMain));
}

//---------------------------------------------------------------
// スレッドの破棄
//---------------------------------------------------------------
void FinalizeThread()
{
    nn::os::DestroyThread(&g_Thread[ObjectType_Red]);
    nn::os::DestroyThread(&g_Thread[ObjectType_Green]);
    nn::os::DestroyThread(&g_Thread[ObjectType_Blue]);
    nn::os::FreeTlsSlot(g_TlsSlot);
}

//---------------------------------------------------------------
// イベントを初期化
//---------------------------------------------------------------
nn::os::EventType   g_EventStart[g_CoreCount];
nn::os::EventType   g_EventEnd[g_CoreCount];
bool                g_IsLoop = true;
void InitializeEvent()
{
    nn::os::InitializeEvent(&g_EventStart[ObjectType_Red], false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&g_EventStart[ObjectType_Green], false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&g_EventStart[ObjectType_Blue], false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&g_EventEnd[ObjectType_Red], false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&g_EventEnd[ObjectType_Green], false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&g_EventEnd[ObjectType_Blue], false, nn::os::EventClearMode_AutoClear);
}

//---------------------------------------------------------------
// イベントを破棄
//---------------------------------------------------------------
void FinalizeEvent()
{
    nn::os::FinalizeEvent(&g_EventStart[ObjectType_Red]);
    nn::os::FinalizeEvent(&g_EventStart[ObjectType_Green]);
    nn::os::FinalizeEvent(&g_EventStart[ObjectType_Blue]);
    nn::os::FinalizeEvent(&g_EventEnd[ObjectType_Red]);
    nn::os::FinalizeEvent(&g_EventEnd[ObjectType_Green]);
    nn::os::FinalizeEvent(&g_EventEnd[ObjectType_Blue]);
}

//---------------------------------------------------------------
// パフォーマンス計測クラスの初期化
//---------------------------------------------------------------
void*  g_MeterMemory;
nns::gfx::PrimitiveRenderer::MeterDrawer g_MeterDrawer;
ptrdiff_t g_MeterMemoryPoolOffset;
void InitializeLoadMeter()
{
    nn::perf::LoadMeterCenterInfo info;
    info.SetCoreCount(3);
    info.SetUserMeterCount(1);
    info.SetCpuBufferCount(2);
    info.SetGpuBufferCount(3);
    info.SetCpuSectionCountMax(64);
    info.SetGpuSectionCountMax(64);

    // 計測で使用するメモリの確保
    size_t memorySize = NN_PERF_GET_BUFFER_SIZE(info);
    size_t memoryAlignment = NN_PERF_GET_BUFFER_ALIGNMENT();
    g_MeterMemory = gfw.AllocateMemory(memorySize, memoryAlignment);

    // メモリプールの確保
    size_t memoryPoolSize = NN_PERF_GET_MEMORY_POOL_SIZE(gfw.GetDevice(), info);
    g_MeterMemoryPoolOffset = gfw.AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Data, memoryPoolSize, NN_PERF_GET_MEMORY_POOL_ALIGNMENT(gfw.GetDevice(), info));
    nn::gfx::MemoryPool* meterMemoryPool = gfw.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_Data);
    NN_PERF_INITIALIZE_METER(gfw.GetDevice(), info,
        g_MeterMemory, memorySize,
        meterMemoryPool, g_MeterMemoryPoolOffset, memoryPoolSize);
    NN_ASSERT(g_MeterMemoryPoolOffset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset, "memoryPool shortage");

    // スレッドが動作するコア番号を取得する関数を登録。デフォルトは nn::os::GetCurrentCoreNumber() が設定されている。
    NN_PERF_SET_GET_CORE_NUMBER_FUNCTION(GetCoreNumberFunction);

    // UserMeter に名前を付ける。デフォルト名は USER0, USER1...。
    NN_PERF_SET_METER_NAME(0, "SYNC");
}

//---------------------------------------------------------------
// 負荷メーターの破棄
//---------------------------------------------------------------
void FinalizeLoadMeter()
{
    NN_PERF_FINALIZE_METER(gfw.GetDevice());
    gfw.FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Data, g_MeterMemoryPoolOffset);
    gfw.FreeMemory(g_MeterMemory);
}

//---------------------------------------------------------------
// オブジェクトの位置を初期化
//---------------------------------------------------------------
static const int g_ObjectCountMax = 1200;
nn::util::Vector3f g_Position[g_ObjectCountMax];
void InitializeObjectPosition()
{
    for (int i = 0; i < g_ObjectCountMax; ++i)
    {
        g_Position[i].SetX(10 * static_cast< float >(rand()) / static_cast< float >(RAND_MAX));
        g_Position[i].SetY(2 * static_cast< float >(rand()) / static_cast< float >(RAND_MAX));
        g_Position[i].SetZ(10 * static_cast< float >(rand()) / static_cast< float >(RAND_MAX));
    }
}

//---------------------------------------------------------------
// ViewMatrix を計算
//---------------------------------------------------------------
void CalculateViewMatrix(nn::util::Matrix4x3fType* viewMatrix)
{
    float radius = 20.f;
    float x = radius * sin(g_Frame / 500.f);
    float z = radius * cos(g_Frame / 500.f);
    nn::util::Vector3fType camPos = { x, 10.f, z };
    nn::util::Vector3fType camTarget = { 0.f, 0.f, 0.f };
    nn::util::Vector3fType camUp = { 0.f, 1.f, 0.f };
    nn::util::MatrixLookAtRightHanded(viewMatrix, camPos, camTarget, camUp);
}

//---------------------------------------------------------------
// ProjectionMatrix を計算
//---------------------------------------------------------------
void CalculateProjectionMatrix(nn::util::Matrix4x4fType* projectionMatrix)
{
    const float fovy = nn::util::FloatPi / 3.0f;
    const float aspect = static_cast< float >(gfw.GetDisplayWidth()) / static_cast< float >(gfw.GetDisplayHeight());
    nn::util::MatrixPerspectiveFieldOfViewRightHanded(projectionMatrix, fovy, aspect, 0.1f, 1000.f);
}

//---------------------------------------------------------------
// ModelMatrix を計算
//---------------------------------------------------------------
void CalculateModelMatrix(nn::util::Matrix4x3f* modelMatrix)
{
    nn::util::Vector3f vecZero;
    nn::util::VectorZero(&vecZero);
    nn::util::Vector3f rotValue;
    nn::util::VectorSet(&rotValue, 0.4f, g_Frame * 0.05f, 0.4f);
    nn::util::MatrixIdentity(modelMatrix);
    nn::util::MatrixSetRotateXyz(modelMatrix, rotValue);
    nn::util::MatrixSetAxisW(modelMatrix, vecZero);
}

//---------------------------------------------------------------
// 赤青緑オブジェクトの描画コマンドを生成
//---------------------------------------------------------------
void MakeObjectCommand()
{
    // スレッドローカルストレージから値を取得
    TlsSlotValue* tlsSlotValue = reinterpret_cast<TlsSlotValue*>(nn::os::GetTlsValue(g_TlsSlot));
    NN_PERF_SET_COLOR_GPU(tlsSlotValue->color);
    NN_PERF_BEGIN_MEASURE_NAME_GPU(&g_CommandBuffer[tlsSlotValue->objectType][g_CurrentBufferIndex].object, tlsSlotValue->name);

    g_CommandBuffer[tlsSlotValue->objectType][g_CurrentBufferIndex].object.SetBlendState(g_pPrimitiveRenderer[tlsSlotValue->objectType]->GetBlendState(nns::gfx::PrimitiveRenderer::BlendType::BlendType_Normal));
    g_pPrimitiveRenderer[tlsSlotValue->objectType]->SetDepthStencilState(&g_CommandBuffer[tlsSlotValue->objectType][g_CurrentBufferIndex].object, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthWriteTest);
    g_pPrimitiveRenderer[tlsSlotValue->objectType]->SetDefaultParameters();

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;
    nn::util::Vector3f translate;
    nn::util::Vector3fType center = { 0.f, 0.f, 0.f };
    nn::util::Vector3fType size = { 0.5f, 3.f, 0.5f };

    CalculateViewMatrix(&viewMatrix);
    CalculateProjectionMatrix(&projectionMatrix);
    CalculateModelMatrix(&modelMatrix);

    g_pPrimitiveRenderer[tlsSlotValue->objectType]->SetViewMatrix(&viewMatrix);
    g_pPrimitiveRenderer[tlsSlotValue->objectType]->SetProjectionMatrix(&projectionMatrix);

    for (int i = 0; i < g_ObjectCountMax; ++i)
    {
        switch (tlsSlotValue->objectType)
        {
        case ObjectType_Red:
            nn::util::VectorSet(&translate, g_Position[i].GetX(), g_Position[i].GetY(), g_Position[i].GetZ());
            g_pPrimitiveRenderer[tlsSlotValue->objectType]->SetColor(nn::util::Color4u8(static_cast<uint8_t>(i), 0, 0, static_cast<uint8_t>(i)));
            break;
        case ObjectType_Green:
            nn::util::VectorSet(&translate, -g_Position[i].GetX(), g_Position[i].GetY(), g_Position[i].GetZ());
            g_pPrimitiveRenderer[tlsSlotValue->objectType]->SetColor(nn::util::Color4u8(0, static_cast<uint8_t>(i), 0, static_cast<uint8_t>(i)));
            break;
        case ObjectType_Blue:
            nn::util::VectorSet(&translate, g_Position[i].GetX(), g_Position[i].GetY(), -g_Position[i].GetZ());
            g_pPrimitiveRenderer[tlsSlotValue->objectType]->SetColor(nn::util::Color4u8(0, 0, static_cast<uint8_t>(i), static_cast<uint8_t>(i)));
            break;
        default:
            break;
        }
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        g_pPrimitiveRenderer[tlsSlotValue->objectType]->SetModelMatrix(&modelMatrix);
        g_pPrimitiveRenderer[tlsSlotValue->objectType]->SetLineWidth(1.f);
        g_pPrimitiveRenderer[tlsSlotValue->objectType]->DrawCone(&g_CommandBuffer[tlsSlotValue->objectType][g_CurrentBufferIndex].object, nns::gfx::PrimitiveRenderer::Surface::Surface_Wired, center, size);
    }

    NN_PERF_END_MEASURE_GPU(&g_CommandBuffer[tlsSlotValue->objectType][g_CurrentBufferIndex].object);
}

//---------------------------------------------------------------
// 赤青緑オブジェクトを描画するスレッドから呼ばれる関数
//---------------------------------------------------------------
void ThreadMain(void *tlsValue)
{
    // スレッドローカルストレージに値を設定
    nn::os::SetTlsValue(g_TlsSlot, reinterpret_cast<uintptr_t>(tlsValue));
    ObjectType objectType = reinterpret_cast<TlsSlotValue*>(tlsValue)->objectType;
    while (g_IsLoop)
    {
        // イベントを使用して同期を取りつつループを回す
        nn::os::WaitEvent(&g_EventStart[objectType]);
        if (g_IsLoop)
        {
            NN_PERF_SET_COLOR(reinterpret_cast<TlsSlotValue*>(tlsValue)->color);
            NN_PERF_AUTO_MEASURE_NAME(reinterpret_cast<TlsSlotValue*>(tlsValue)->name);
            g_CommandBuffer[objectType][g_CurrentBufferIndex].object.Begin();
            MakeObjectCommand();
            g_CommandBuffer[objectType][g_CurrentBufferIndex].object.End();
        }
        nn::os::SignalEvent(&g_EventEnd[objectType]);
    }
}

//---------------------------------------------------------------
// 黄オブジェクトの描画コマンドを生成
//---------------------------------------------------------------
void MakeYellowCommand()
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::Yellow());
    NN_PERF_AUTO_MEASURE_NAME("Yellow");
    {
        nn::gfx::CommandBuffer* pRootCommandBuffer = gfw.GetRootCommandBuffer(g_CurrentBufferIndex);
        NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Yellow());
        NN_PERF_BEGIN_MEASURE_NAME_GPU(pRootCommandBuffer, "Yellow");

        pRootCommandBuffer->SetBlendState(g_pRootPrimitiveRenderer->GetBlendState(nns::gfx::PrimitiveRenderer::BlendType::BlendType_Normal));
        g_pRootPrimitiveRenderer->SetDepthStencilState(pRootCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthWriteTest);
        g_pRootPrimitiveRenderer->SetDefaultParameters();

        nn::util::Matrix4x3fType viewMatrix;
        nn::util::Matrix4x4fType projectionMatrix;
        nn::util::Matrix4x3f modelMatrix;
        nn::util::Vector3f translate;
        nn::util::Vector3fType center = { 0.f, 0.f, 0.f };
        nn::util::Vector3fType size = { 0.5f, 3.f, 0.5f };

        CalculateViewMatrix(&viewMatrix);
        CalculateProjectionMatrix(&projectionMatrix);
        CalculateModelMatrix(&modelMatrix);

        g_pRootPrimitiveRenderer->SetViewMatrix(&viewMatrix);
        g_pRootPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);

        for (int i = 0; i < g_ObjectCountMax; ++i)
        {
            nn::util::VectorSet(&translate, -g_Position[i].GetX(), g_Position[i].GetY(), -g_Position[i].GetZ());
            nn::util::MatrixSetAxisW(&modelMatrix, translate);
            g_pRootPrimitiveRenderer->SetModelMatrix(&modelMatrix);
            g_pRootPrimitiveRenderer->SetLineWidth(1.f);
            g_pRootPrimitiveRenderer->SetColor(nn::util::Color4u8(static_cast<uint8_t>(i), static_cast<uint8_t>(i), 0, static_cast<uint8_t>(i)));
            g_pRootPrimitiveRenderer->DrawCone(pRootCommandBuffer, nns::gfx::PrimitiveRenderer::Surface::Surface_Wired, center, size);
        }
        NN_PERF_END_MEASURE_GPU(pRootCommandBuffer);
    }
}

//---------------------------------------------------------------
// メインコマンド生成
//---------------------------------------------------------------
void MakeCommand()
{
    gfw.BeginFrame(g_CurrentBufferIndex);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = gfw.GetRootCommandBuffer(g_CurrentBufferIndex);
        NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Gray());
        NN_PERF_BEGIN_MEASURE_NAME_GPU(rootCommandBuffer, "Others");

        nn::gfx::ColorTargetView* pTarget = gfw.GetColorTargetView();

        rootCommandBuffer->ClearColor(pTarget, 0.0f, 0.0f, 0.0f, 1.0f, NULL);
        rootCommandBuffer->ClearDepthStencil(gfw.GetDepthStencilView(), 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

        rootCommandBuffer->SetRenderTargets(1, &pTarget, gfw.GetDepthStencilView());
        rootCommandBuffer->SetViewportScissorState(gfw.GetViewportScissorState());
        rootCommandBuffer->SetRasterizerState(gfw.GetRasterizerState(nns::gfx::GraphicsFramework::RasterizerStateType::RasterizerStateType_FillSolid_CullNone));

        rootCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer
            | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer);

        g_pRootPrimitiveRenderer->SetDefaultParameters();
        nn::util::Matrix4x3fType viewMatrix;
        nn::util::Matrix4x4fType projectionMatrix;
        nn::util::Matrix4x3f modelMatrix;
        nn::util::MatrixIdentity(&modelMatrix);
        nn::util::Vector3f translate;
        nn::util::VectorSet(&translate, 0.f, 0.f, 1.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);

        // Blend
        rootCommandBuffer->SetBlendState(g_pRootPrimitiveRenderer->GetBlendState(nns::gfx::PrimitiveRenderer::BlendType::BlendType_Normal));

        CalculateViewMatrix(&viewMatrix);
        CalculateProjectionMatrix(&projectionMatrix);
        g_pRootPrimitiveRenderer->SetViewMatrix(&viewMatrix);
        g_pRootPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);

        // Depth Enable
        g_pRootPrimitiveRenderer->SetDepthStencilState(rootCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthWriteTest);

        // 軸を描画
        float interval = -10.f;
        nn::util::Vector3fType begin;
        nn::util::Vector3fType end;
        g_pRootPrimitiveRenderer->SetLineWidth(1.f);
        for (int i = 0; i < 21; i++)
        {
            nn::util::VectorSet(&begin, -10.f, 0.f, interval);
            nn::util::VectorSet(&end, 10.f, 0.f, interval);
            g_pRootPrimitiveRenderer->SetColor(nn::util::Color4u8::White());
            g_pRootPrimitiveRenderer->DrawLine(rootCommandBuffer, begin, end);
            nn::util::VectorSet(&begin, interval, 0.f, -10.f);
            nn::util::VectorSet(&end, interval, 0.f, 10.f);
            g_pRootPrimitiveRenderer->DrawLine(rootCommandBuffer, begin, end);
            interval += 1.0f;
        }

        NN_PERF_END_MEASURE_GPU(rootCommandBuffer);

        // 各スレッドで作成したコマンドをメインのコマンドバッファに追加
        rootCommandBuffer->CallCommandBuffer(&g_CommandBuffer[ObjectType_Red][g_CurrentBufferIndex].object);
        rootCommandBuffer->CallCommandBuffer(&g_CommandBuffer[ObjectType_Green][g_CurrentBufferIndex].object);
        rootCommandBuffer->CallCommandBuffer(&g_CommandBuffer[ObjectType_Blue][g_CurrentBufferIndex].object);

        // 黄オブジェクトの描画コマンド追加
        MakeYellowCommand();

        NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Gray());
        NN_PERF_BEGIN_MEASURE_NAME_GPU(rootCommandBuffer, "Others");

        rootCommandBuffer->SetBlendState(g_pRootPrimitiveRenderer->GetBlendState(nns::gfx::PrimitiveRenderer::BlendType::BlendType_Normal));
        g_pRootPrimitiveRenderer->SetDepthStencilState(rootCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);
        g_pRootPrimitiveRenderer->SetDefaultParameters();

        // 計測メーター描画
        // NN_PERF_IS_ENABLED() が false の場合は メーターと fps の描画をしないように分岐。
        // また、NN_PERF_PROFILE_ENABLED が未定義の場合は、NN_PERF_GET_FRAME_METER() 等の計測オブジェクトのポインタを取得するマクロが NULL を返すため
        // これらのマクロを使用する場合はハンドリングが必要。
        if (NN_STATIC_CONDITION(NN_PERF_IS_ENABLED()))
        {
            // 計測メーター描画
            nn::perf::CpuMeter* pFrameMeter = NN_PERF_GET_FRAME_METER();
            nn::util::Float2 pos = NN_UTIL_FLOAT_2_INITIALIZER(32.f, gfw.GetDisplayHeight() - g_MeterDrawer.GetHeight(pFrameMeter) - g_MeterDrawer.GetBarHeight());
            g_MeterDrawer.SetDebugFontTextWriter(&g_Writer.object);
            g_MeterDrawer.SetPosition(pos);
            g_MeterDrawer.SetWidth(gfw.GetDisplayWidth() - 64.f);
            g_MeterDrawer.Draw(rootCommandBuffer, g_pRootPrimitiveRenderer, pFrameMeter);

            // FPS 表示
            float fps = 1000.f / (static_cast<float>(pFrameMeter->GetLastTotalSpan().GetNanoSeconds()) / 1000000.0f);
            g_Writer.object.SetTextColor(nn::util::Color4u8::White());
            g_Writer.object.SetCursor(10, 10);
            g_Writer.object.Print("%4.1f FPS", fps);

            // テキスト描画
            g_Writer.object.Draw(rootCommandBuffer);
        }

        NN_PERF_END_MEASURE_GPU(rootCommandBuffer);

        // QueryBuffer に対してキャッシュのフラッシュを行う
        rootCommandBuffer->FlushMemory(nn::gfx::GpuAccess_QueryBuffer);
    }
    gfw.EndFrame(g_CurrentBufferIndex);
} //NOLINT(impl/function_size)
}

//---------------------------------------------------------------
//  Main Function
//  メイン関数です。
//---------------------------------------------------------------
extern "C" void nnMain()
{
    // フレームワークの初期化
    InitializeGraphicsFramework();

    // コマンドバッファの初期化
    InitializeCommandBuffer();

    // プリミティブレンダラの初期化
    InitializePrimitiveRenderer();

    // デバッグフォントの初期化
    gfw.InitializeDebugFontTextWriter(&g_Writer, 256, 2);

    // スレッドの初期化
    InitializeThread();

    // イベントの初期化
    InitializeEvent();

    // 処理メータの初期化
    InitializeLoadMeter();

    // 描画オブジェクトの位置を初期化
    InitializeObjectPosition();

    // スレッドの開始
    nn::os::StartThread(&g_Thread[ObjectType_Red]);
    nn::os::StartThread(&g_Thread[ObjectType_Green]);
    nn::os::StartThread(&g_Thread[ObjectType_Blue]);

    // 毎フレームのレンダリング
    for (g_Frame = 0; g_Frame < 60000; ++g_Frame)
    {
        NN_PERF_BEGIN_FRAME();
        {
            int currentBufferIndex = g_CurrentBufferIndex;
            int previousBufferIndex = 1 - currentBufferIndex;

            if (g_Frame > 0)
            {
                // コマンドの実行
                {
                    NN_PERF_SET_COLOR(nn::util::Color4u8::White());
                    NN_PERF_AUTO_MEASURE_NAME("ExecuteCommand");
                    gfw.ExecuteCommand(previousBufferIndex);
                }

                {
                    NN_PERF_SET_COLOR(nn::util::Color4u8::Black());
                    NN_PERF_AUTO_MEASURE_NAME("PresentTexture");
                    gfw.QueuePresentTexture(gfw.GetPresentInterval());
                }
            }

            {
                NN_PERF_SET_COLOR(nn::util::Color4u8::Cyan());
                NN_PERF_AUTO_MEASURE_NAME("AcquireTexure");
                gfw.AcquireTexture(currentBufferIndex);
                //gfw.QueueWaitSync(currentBufferIndex); // 不要
            }

            // プリミティブレンダラ更新
            {
                NN_PERF_SET_COLOR(nn::util::Color4u8::White());
                NN_PERF_AUTO_MEASURE_NAME("Update");
                UpdatePrimitiveRenderer(currentBufferIndex);
            }

            // 赤緑青オブジェクト描画のコマンド作成
            {
                //NN_PERF_SET_COLOR(nn::util::Color4u8::White());
                //NN_PERF_AUTO_MEASURE_NAME("MakeCommandThread");

                // コマンドバッファのリセット
                gfw.ResetCommandBuffer(&g_CommandBuffer[ObjectType_Red][currentBufferIndex]);
                gfw.ResetCommandBuffer(&g_CommandBuffer[ObjectType_Green][currentBufferIndex]);
                gfw.ResetCommandBuffer(&g_CommandBuffer[ObjectType_Blue][currentBufferIndex]);

                // イベントを使用してスレッドの同期を取る
                nn::os::SignalEvent(&g_EventStart[ObjectType_Red]);
                nn::os::SignalEvent(&g_EventStart[ObjectType_Green]);
                nn::os::SignalEvent(&g_EventStart[ObjectType_Blue]);

                nn::os::WaitEvent(&g_EventEnd[ObjectType_Red]);
                nn::os::WaitEvent(&g_EventEnd[ObjectType_Green]);
                nn::os::WaitEvent(&g_EventEnd[ObjectType_Blue]);
            }

            // メインのコマンド作成
            {
                NN_PERF_SET_COLOR(nn::util::Color4u8::Gray());
                NN_PERF_AUTO_MEASURE_NAME("MakeCommand");
                MakeCommand();
            }

            // コマンドの同期
            {
                NN_PERF_SET_COLOR_INDEX(0, nn::util::Color4u8::Magenta());
                NN_PERF_AUTO_MEASURE_INDEX_NAME(0, "Sync");
                gfw.WaitDisplaySync(currentBufferIndex, gfw.GetWaitSyncTimeout());
                if (g_Frame > 0)
                {
                    gfw.WaitGpuSync(previousBufferIndex, gfw.GetWaitSyncTimeout());
                }
            }

            g_CurrentBufferIndex = 1 - g_CurrentBufferIndex;
        }
        NN_PERF_END_FRAME();

        // 計測結果の出力
        if (g_Frame % 60 == 0)
        {
            NN_PERF_DUMP();
        }
    }
    gfw.QueueFinish();

    // スレッドのループを終了させる
    g_IsLoop = false;
    nn::os::SignalEvent(&g_EventStart[ObjectType_Red]);
    nn::os::SignalEvent(&g_EventStart[ObjectType_Green]);
    nn::os::SignalEvent(&g_EventStart[ObjectType_Blue]);
    nn::os::WaitEvent(&g_EventEnd[ObjectType_Red]);
    nn::os::WaitEvent(&g_EventEnd[ObjectType_Green]);
    nn::os::WaitEvent(&g_EventEnd[ObjectType_Blue]);

    // 処理メータ破棄
    FinalizeLoadMeter();

    // イベントの破棄
    FinalizeEvent();

    // スレッドの破棄
    FinalizeThread();

    // デバッグフォントの破棄
    gfw.FinalizeDebugFontTextWriter(&g_Writer);

    // プリミティブレンダラの破棄
    FinalizePrimitiveRenderer();

    // コマンドバッファの破棄
    FinalizeCommandBuffer();

    // フレームワークの破棄
    FinalizeGraphicsFramework();
}
