﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OsThreadLocalStorage.cpp,PageSampleOsThreadLocalStorage}
 *
 * @brief
 *  スレッド固有領域（TLS）機能のサンプルプログラム
 */

/**
 * @page PageSampleOsThreadLocalStorage ThreadLocalStorage
 * @tableofcontents
 *
 * @brief
 *  スレッド固有領域（TLS）機能のサンプルプログラムの解説です。
 *
 * @section PageSampleOsThreadLocalStorage_SectionBrief 概要
 *  ここでは、スレッド固有領域を使ったサンプルプログラムの説明をします。
 *
 *  スレッド固有領域の使い方については、スレッド固有領域マニュアル および
 *  @ref nn::os "OS の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOsThreadLocalStorage_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OsThreadLocalStorage Samples/Sources/Applications/OsThreadLocalStorage @endlink 以下にあります。
 *
 * @section PageSampleOsThreadLocalStorage_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleOsThreadLocalStorage_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOsThreadLocalStorage_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOsThreadLocalStorage_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOsThreadLocalStorage_SectionDetail 解説
 *
 * @subsection PageSampleOsThreadLocalStorage_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OsThreadLocalStorage.cpp
 *  @includelineno OsThreadLocalStorage.cpp
 *
 * @subsection PageSampleOsThreadLocalStorage_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nnMain() にて、3 つのスレッドを生成、1 つの TLS を確保
 *  - 3 つのスレッドは、いずれも ThreadSample() を共用
 *  - ThreadSample() は引数として "Thread1" のような文字列を受取る（スレッド毎にスレッド番号が異なる）
 *  - ThreadSample() では受取った引数を TLS に格納して CommonFunction() を呼ぶ
 *  - CommonFunction() では TLS から値を取得してメッセージを表示
 *  - ThreadSample() に戻り、スレッド終了直前にメッセージを表示
 *  - スレッドが終了すると、TLS のデストラクタ関数 TlsDestructFunction() が呼ばれる
 *  - TlsDestructFunction() では引数として受取った TLS 値をメッセージとして表示
 *  - 3 つのスレッドがそれぞれ上記の処理を並行して実行
 *  - 3 つのスレッドが終了するまで nnMain() は待機
 *  - nnMain() にて 3 つのスレッドを破棄、TLS も解放
 *
 *  本サンプルプログラムでは基本的な TLS の使い方を示しています。
 *  共通関数 CommonFunction() は、3 つのスレッドから発行されますが、
 *  TLS にアクセスすることで異なるパラメータを扱えることを示しています。
 *
 *  本サンプルプログラムでは、TLS に各スレッド関数の引数をそのまま設定していますが、
 *  実使用上は各スレッドで動的確保したメモリのアドレス等を格納しておき、
 *  デストラクタ関数ではそのアドレスを引数で受取ることができるので、
 *  その値でメモリを返却するといった使い方が想定されます。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  OsThreadLocalStorage_OutputExample.txt
 *
 *  3 つのスレッドはそれぞれ独立して動作しているため、
 *  これら 3 スレッドが出力メッセージが上記の順番になるとは限りません。
 *  実際には実行環境に依存して、メッセージ出力が前後します。
 *
 */

#include <stdint.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>

//-----------------------------------------------------------------------------

namespace {

const size_t            ThreadStackSize = 8192;             // スレッド操作スレッドのスタックサイズ
NN_OS_ALIGNAS_THREAD_STACK char   g_ThreadStack1[ ThreadStackSize ];  // 1つ目のスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char   g_ThreadStack2[ ThreadStackSize ];  // 2つ目のスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char   g_ThreadStack3[ ThreadStackSize ];  // 3つ目のスレッドのスタック

nn::os::TlsSlot         g_tlsSlot;

}   // namespace

//-----------------------------------------------------------------------------

//
//  スレッド固有領域（TLS）へのアクセス
//
void    CommonFunction()
{
    // TLS のパラメータを取得
    auto name = reinterpret_cast<char*>( nn::os::GetTlsValue( g_tlsSlot ) );

    // 得られた TLS 値はスレッドごとに異なる
    NN_LOG("%s: Succeeded in getting a TLS value\n", name);
}


//
//  スレッド関数（共通実装）
//
void ThreadSample(void *arg)
{
    auto name = reinterpret_cast<char*>( arg );

    // TLS に arg パラメータを設定
    nn::os::SetTlsValue( g_tlsSlot, reinterpret_cast<uintptr_t>( arg ) );
    NN_LOG("%s: Succeeded in setting a TLS value\n", name );

    // 共通関数を呼出す（全てのスレッドで同じ処理を行なう）
    CommonFunction();

    // スレッド終了前にメッセージを表示
    NN_LOG("%s: End of thread\n", name );
}


//
//  TLS デストラクタ関数
//
void    TlsDestructFunction(uintptr_t value)
{
    char* name = reinterpret_cast<char*>( value );

    NN_LOG("%s: TLS destructor function is called back.\n", name);
}


//
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::Result      result;
    nn::os::ThreadType  thread1;
    nn::os::ThreadType  thread2;
    nn::os::ThreadType  thread3;

    // TLS スロットを 1 つ確保
    result = nn::os::AllocateTlsSlot( &g_tlsSlot, TlsDestructFunction );
    NN_ASSERT( result.IsSuccess(), "Cannot allocate TLS slot.");

    // スレッドを生成する
    result = nn::os::CreateThread( &thread1, ThreadSample, const_cast<char*>("Thread1"), g_ThreadStack1, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create thread1." );

    result = nn::os::CreateThread( &thread2, ThreadSample, const_cast<char*>("Thread2"), g_ThreadStack2, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create thread2." );

    result = nn::os::CreateThread( &thread3, ThreadSample, const_cast<char*>("Thread3"), g_ThreadStack3, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create thread3." );

    // スレッドの実行を開始する
    nn::os::StartThread( &thread1 );
    nn::os::StartThread( &thread2 );
    nn::os::StartThread( &thread3 );

    // スレッドが終了するのを待つ
    nn::os::WaitThread( &thread1 );
    nn::os::WaitThread( &thread2 );
    nn::os::WaitThread( &thread3 );

    // スレッドを破棄する
    nn::os::DestroyThread( &thread1 );
    nn::os::DestroyThread( &thread2 );
    nn::os::DestroyThread( &thread3 );

    // TLS スロットを開放
    nn::os::FreeTlsSlot( g_tlsSlot );
}

