﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OsSharedMemory.cpp,PageSampleOsSharedMemory}
 *
 * @brief
 *  SharedMemory サンプルプログラム
 */

/**
 * @page PageSampleOsSharedMemory SharedMemory
 * @tableofcontents
 *
 * @brief
 *  共有メモリ機能のサンプルプログラムの解説です。
 *
 * @section PageSampleOsSharedMemory_SectionBrief 概要
 *  ここでは、共有メモリ機能を使ったサンプルプログラムの説明をします。
 *
 *  共有メモリ機能の使い方については、
 *  @ref nn::os "OS の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOsSharedMemory_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OsSharedMemory Samples/Sources/Applications/OsSharedMemory @endlink 以下にあります。
 *
 * @section PageSampleOsSharedMemory_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleOsSharedMemory_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOsSharedMemory_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOsSharedMemory_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOsSharedMemory_SectionDetail 解説
 *
 * @subsection PageSampleOsSharedMemory_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OsSharedMemory.cpp
 *  @includelineno OsSharedMemory.cpp
 *
 * @subsection PageSampleOsSharedMemory_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nnMain() にて、2 つのスレッドを生成
 *  - 2 つのスレッドは、 ServerThreadFunc() と ClientThreadFunc()
 *  - ServerThreadFunc() で 共有メモリを作成し、自コンテキスト用にマップ
 *  - 上記で作成した 共有メモリのハンドルを GetSharedMemoryHandle() で取得し、
 *    メッセージキューを介して ClientThreadFunc() に渡す
 *  - ClientThreadFunc() ではメッセージキューを介して取得した共有メモリの
 *    ハンドルを AttachSharedMemory() で SharedMemoryType オブジェクトに関連付ける
 *  - ClientThreadFunc() でも自コンテキスト用に共有メモリをマップ
 *  - ServerThreadFunc() で共有メモリに書き込んだ内容を、
 *    ClientThreadFunc() から共有メモリから参照してログに出力
 *  - 2 つのスレッドが終了するまで nnMain() は待機
 *  - nnMain() にて 2 つのスレッドを破棄
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  OsSharedMemory_OutputExample.txt
 *
 *  本サンプルでは、MessageQueue を使って NativeHandle を ClientThread から
 *  ServerThread へ渡していますが、実際にはシステムで定められたプロセス間通信
 *  の仕組みを使ってクライアントプロセスからサーバープロセスへ
 *  NativeHandle を転送して利用することを想定しています。
 *
 *  また、本サンプルプログラムでは、共有メモリ領域内のデータを使って
 *  クライアントとサーバーが同期を行なっています。本サンプルプログラムでは、
 *  SharedInfo 構造体内の prepared 変数がこの同期目的で利用されていますが、
 *  本サンプルプログラムで示すように、必要に応じてメモリフェンス API を
 *  利用する必要があります。詳細は @ref PageNotificationOsMemoryFence を
 *  参照して下さい。
 *
 *  なお、共有メモリの API 体系は近い将来変更する予定ですので、ご了承下さい。
 *
 */

#include <cstring>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/os/os_SharedMemory.h>

//-----------------------------------------------------------------------------

namespace {

// 共有メモリ操作スレッドのスタックの定義
const size_t           ThreadStackSize = 8192;
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack1[ ThreadStackSize ];
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack2[ ThreadStackSize ];

// サーバー・クライアント通信用のメッセージキュー
nn::os::MessageQueueType g_MessageQueue;
const size_t MessageQueueBufferCount = 1;
uintptr_t    g_MessageQueueBuffer[ MessageQueueBufferCount ];

// 共有メモリのサイズ
const size_t SharedMemorySize   = 8192;

// 共有メモリを使って、サーバーとクライアントが共有する情報。
// 共有メモリ領域の先頭に配置される。
struct SharedInfo
{
    volatile bool   prepared;       // サーバーが用意した文字列が準備完了か
    char            data[ SharedMemorySize - 1 ];
};
static_assert(sizeof(SharedInfo) == SharedMemorySize, "");

}   // namespace

// サーバーからクライアントへ渡すメッセージ
#define NNT_SAMPLE_MESSAGE  "This is a shared memory sample program.\n" \
                            "This message is sent via shared memory.\n"

//-----------------------------------------------------------------------------
//  クライアントスレッド（サーバーから共有メモリを受け取り、利用する）
//
void ClientThreadFunc(void *arg)
{
    NN_UNUSED(arg);

    // クライアントが使用する SharedMemoryType オブジェクト
    nn::os::SharedMemoryType clientSharedMemory;

    // サーバーから共有メモリのハンドルが送られるくるのを待つ
    uintptr_t data;
    nn::os::ReceiveMessageQueue(&data, &g_MessageQueue);
    NN_LOG("Client: Receieved handle. handle=0x%p\n", data);
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    auto handle = reinterpret_cast<nn::os::NativeHandle>(data);
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    auto handle = static_cast<nn::os::NativeHandle>(data & 0xffffffff);
#endif

    // 受け取ったハンドルを SharedMemoryType オブジェクトにアタッチ。
    // このサンプルでは、同一プロセスで共有メモリを扱っているため、
    // クライアントサイドではハンドルの管理フラグを false にしておく。
    nn::os::AttachSharedMemory(&clientSharedMemory, SharedMemorySize, handle, false);
    NN_LOG("Client: Attached shared memory handle.\n");

    // 共有メモリを仮想アドレス空間にマップ
    void* address = nn::os::MapSharedMemory(&clientSharedMemory, nn::os::MemoryPermission_ReadWrite);
    NN_ABORT_UNLESS( address != NULL );
    NN_LOG("Client: Mapped shared memory. address=0x%p\n", address);

    SharedInfo* info = reinterpret_cast<SharedInfo*>(address);

    // サーバーから共有メモリを介して文字列を受信し、ログ出力
    NN_LOG("------------------------------------------\n");
    {
        while (!info->prepared)
        {
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
        }

        nn::os::FenceMemoryLoadLoad();

        char* src = info->data;
        char  c;
        while ( (c = *src) != '\0' )
        {
            NN_LOG("%c", c);
            ++src;
        }
    }
    NN_LOG("------------------------------------------\n");

    // 必要がなくなったら 共有メモリをアンマップし、破棄する
    nn::os::UnmapSharedMemory(&clientSharedMemory);
    NN_LOG("Client: Unmapped shared memory.\n");
    nn::os::DestroySharedMemory(&clientSharedMemory);
    NN_LOG("Client: Destroyed shared memory.\n");
}


//-----------------------------------------------------------------------------
//  サーバースレッド（共有メモリを作成し、クライアントに渡す）
//
void ServerThreadFunc(void *arg)
{
    auto* clientThread = reinterpret_cast<nn::os::ThreadType*>(arg);

    // サーバーが使用する SharedMemoryType オブジェクト
    nn::os::SharedMemoryType serverSharedMemory;

    // 共有メモリを作成する
    auto result = nn::os::CreateSharedMemory(&serverSharedMemory, SharedMemorySize, nn::os::MemoryPermission_ReadWrite, nn::os::MemoryPermission_ReadWrite);
    NN_ABORT_UNLESS( result.IsSuccess() );
    NN_LOG("Server: Created shared memory.\n");

    // 共有メモリをマップ
    void* address = nn::os::MapSharedMemory(&serverSharedMemory, nn::os::MemoryPermission_ReadWrite);
    NN_ABORT_UNLESS( address != NULL );
    NN_LOG("Server: Mapped shared memory. address=0x%p\n", address);

    SharedInfo* info = reinterpret_cast<SharedInfo*>(address);
    info->prepared  = false;
    nn::os::FenceMemoryStoreAny();

    // SharedMemoryType オブジェクトからハンドルを取得
    auto handle = nn::os::GetSharedMemoryHandle(&serverSharedMemory);
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    auto data   = reinterpret_cast<uintptr_t>(handle);
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    auto data   = static_cast<uintptr_t>(handle);
#endif

    // 共有メモリのハンドルをメッセージキュー経由でクライアントに送る
    NN_LOG("Server: Send shared memory handle. handle=0x%p\n", data);
    nn::os::SendMessageQueue(&g_MessageQueue, data);

    // クライアントへ共有メモリを介して文字列を送信
    {
        std::memcpy(info->data, NNT_SAMPLE_MESSAGE, sizeof(NNT_SAMPLE_MESSAGE));

        nn::os::FenceMemoryStoreStore();

        info->prepared  = true;
    }

    // クライアントスレッドの終了待ち
    nn::os::WaitThread( clientThread );

    // 共有メモリをアンマップして、オブジェクトを破棄
    nn::os::UnmapSharedMemory(&serverSharedMemory);
    NN_LOG("Server: Unmapped shared memory.\n");
    nn::os::DestroySharedMemory(&serverSharedMemory);
    NN_LOG("Server: Destroyed shared memory.\n");
}


//-----------------------------------------------------------------------------
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::os::ThreadType  thread1;
    nn::os::ThreadType  thread2;
    nn::Result          result;

    // メッセージキューを初期化する
    nn::os::InitializeMessageQueue( &g_MessageQueue, g_MessageQueueBuffer, MessageQueueBufferCount);

    // スレッドを生成する
    result = nn::os::CreateThread( &thread1, ClientThreadFunc, NULL, g_ThreadStack1, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ABORT_UNLESS( result.IsSuccess(), "Cannot create client thread." );

    result = nn::os::CreateThread( &thread2, ServerThreadFunc, &thread1, g_ThreadStack2, ThreadStackSize, nn::os::DefaultThreadPriority + 1 );
    NN_ABORT_UNLESS( result.IsSuccess(), "Cannot create server thread." );

    // スレッドの実行を開始する
    nn::os::StartThread( &thread1 );
    nn::os::StartThread( &thread2 );

    // スレッドが終了するのを待つ
    nn::os::WaitThread( &thread1 );
    nn::os::WaitThread( &thread2 );

    // スレッドを破棄する
    nn::os::DestroyThread( &thread1 );
    nn::os::DestroyThread( &thread2 );

    // メッセージキューをファイナライズする
    nn::os::FinalizeMessageQueue(&g_MessageQueue);
}

